# VBCorLib コードリーディングガイドライン

## はじめに

このガイドラインは、VBCorLibのコードベースを効率的に理解するための手引きです。
Visual Basic 6（VB6）に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- .NETからVB6への移植や比較を行う開発者
- コードレビューを行う担当者

**プロジェクト概要:**
VBCorLibは、.NET Frameworkの多くのクラスをVisual Basic 6で実装したライブラリです。.NETアプリケーションとVB6間でのデータ共有を容易にし、コレクション型、暗号化アルゴリズム、ハッシュアルゴリズム、エンコーディング、文字列/配列/日付操作などの機能を提供します。

---

## 1. 言語基礎

> このセクションでは、Visual Basic 6（VB6）の基本構文と概念を解説します。

### 1.1 プログラム構造

VB6のプログラムは主に以下の3種類のファイルで構成されます：

| 拡張子 | 種類 | 説明 |
|--------|------|------|
| `.cls` | クラスモジュール | オブジェクト指向のクラス定義 |
| `.bas` | 標準モジュール | グローバル関数・変数の定義 |
| `.frm` | フォーム | GUIの定義（本プロジェクトではテスト用のみ） |
| `.vbp` | プロジェクトファイル | プロジェクト設定とファイル一覧 |

**クラスモジュールの基本構造:**
```vb
' ファイル: Source/CorLib/System/Exception.cls:1-13
VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
  Persistable = 1  'Persistable
  DataBindingBehavior = 0  'vbNone
  DataSourceBehavior  = 0  'vbNone
  MTSTransactionMode  = 0  'NotAnMTSObject
END
Attribute VB_Name = "Exception"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
```

- `VERSION 1.0 CLASS`: クラスファイルのバージョン宣言
- `Attribute VB_Name`: クラス名の定義
- `Attribute VB_Exposed = True`: 外部からアクセス可能なパブリッククラス

### 1.2 データ型と変数

VB6の主要なデータ型：

| VB6型 | .NET相当 | サイズ | 範囲 |
|-------|----------|--------|------|
| `Byte` | byte | 1バイト | 0 - 255 |
| `Integer` | short | 2バイト | -32,768 - 32,767 |
| `Long` | int | 4バイト | -2,147,483,648 - 2,147,483,647 |
| `Single` | float | 4バイト | 浮動小数点 |
| `Double` | double | 8バイト | 浮動小数点 |
| `String` | string | 可変 | 文字列 |
| `Boolean` | bool | 2バイト | True/False |
| `Variant` | object | 16バイト | 任意の型 |
| `Currency` | decimal | 8バイト | 通貨型 |
| `Object` | Object | 4バイト | オブジェクト参照 |

**変数宣言の例:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:70-73
Private mItems()    As Variant
Private mCount      As Long
Private mCapacity   As Long
Private mVersion    As Long
```

- `Private`: クラス内部でのみアクセス可能
- `Public`: 外部からアクセス可能
- `Dim`: ローカル変数の宣言

### 1.3 制御構造

**条件分岐:**
```vb
' ファイル: Source/CorLib/System/CorArray.cls:574-581
Select Case VarType(Arr) And &HFF
    Case vbObject
        IIDOffset = vbSizeOfGuid
    Case vbString, vbUserDefinedType
        IIDOffset = 4
    Case Else
        IIDOffset = 0
End Select
```

**If文:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:175-180
If Index < 0 Then _
    Error.ArgumentOutOfRange "Index", ArgumentOutOfRange_NeedNonNegNum
If Count < 0 Then _
    Error.ArgumentOutOfRange "Count", ArgumentOutOfRange_NeedNonNegNum
If Index + Count > mCount Then _
    Error.Argument Argument_InvalidOffLen
```

**ループ構造:**
```vb
' For...Next ループ
' ファイル: Source/CorLib/System/CorArray.cls:241-252
For i = LBound(Arr) To UBound(Arr)
    If Predicate.Invoke(ByVal ElementPtr) Then
        ' 処理
    End If
    ElementPtr = ElementPtr + Info.ElementSize
Next i

' Do...Loop
' ファイル: Source/CorLib/System/CorArray.cls:1023-1031
Do While ElemCount > 0
    If Object.Equals(Arr(ElemIndex), Value) Then
        LastIndexOf = ElemIndex
        Exit Function
    End If
    ElemIndex = ElemIndex - 1
    ElemCount = ElemCount - 1
Loop
```

### 1.4 関数/メソッド定義

**パブリックメソッド（外部から呼び出し可能）:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:88-91
Public Function Add(ByRef Value As Variant) As Long
    Add = mCount
    Insert mCount, Value
End Function
```

**プライベートメソッド（クラス内部のみ）:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:734-748
Private Sub EnsureCapacity(ByVal RequiredCapacity As Long)
    If RequiredCapacity <= mCapacity Then
        Exit Sub
    End If

    Dim NewCapacity As Long
    NewCapacity = mCapacity * 2

    If RequiredCapacity > NewCapacity Then
        NewCapacity = RequiredCapacity
    End If

    ReDim Preserve mItems(0 To NewCapacity - 1)
    mCapacity = NewCapacity
End Sub
```

**パラメータの受け渡し:**
- `ByRef`: 参照渡し（デフォルト）
- `ByVal`: 値渡し

### 1.5 モジュール/インポート

VB6ではインポート文はなく、プロジェクトファイル（.vbp）で参照を管理します。

```
' ファイル: Source/CorLib/CorLib.vbp:2-4
Reference=*\G{00020430-0000-0000-C000-000000000046}#2.0#0#..\..\..\..\..\..\..\Windows\SysWOW64\stdole2.tlb#OLE Automation
Reference=*\G{CDFFA366-1D4F-43F9-B4BF-5438BC7C7D29}#1.1#0#..\..\TypeLibrary\vbvm6lib.tlb#Visual Basic 6 Virtual Machine Type Library
Reference=*\G{1F852AFD-5C0A-46D0-8AA1-4319E8EF72C0}#3.0#0#..\..\TypeLibrary\CorType.tlb#VB.EXT: VBCorLib 3.0 Type Library
```

---

## 2. プロジェクト固有の概念

> このセクションでは、VBCorLib特有の概念を解説します。

### 2.1 .NET互換API設計

VBCorLibは.NET Frameworkのクラスライブラリを模倣した設計となっています。

**.NET対応クラスの例:**

| VBCorLib | .NET Framework |
|----------|----------------|
| `ArrayList` | `System.Collections.ArrayList` |
| `Hashtable` | `System.Collections.Hashtable` |
| `StringBuilder` | `System.Text.StringBuilder` |
| `CorDateTime` | `System.DateTime` |
| `CorArray` | `System.Array` |
| `Exception` | `System.Exception` |

**ネーミング規則:**
- VB6の予約語と競合する型には `Cor` プレフィックスを付与
  - 例: `CorArray`, `CorString`, `CorDateTime`

### 2.2 インターフェース実装パターン

VB6でのインターフェース実装は `Implements` キーワードを使用します。

```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:52-57
Implements IObject
Implements ICollection
Implements IEnumerable
Implements ICloneable
Implements IList
Implements IVersionable
```

インターフェースメソッドの実装：
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:901-911
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'   IObject
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Private Function IObject_Equals(Value As Variant) As Boolean
    IObject_Equals = Equals(Value)
End Function

Private Function IObject_GetHashCode() As Long
    IObject_GetHashCode = GetHashCode
End Function

Private Function IObject_ToString() As String
    IObject_ToString = ToString
End Function
```

### 2.3 例外処理パターン

VBCorLibは独自の例外処理フレームワークを提供しています。

**例外クラス階層:**
```
Exception
├── SystemException
│   ├── ArgumentException
│   │   ├── ArgumentNullException
│   │   └── ArgumentOutOfRangeException
│   ├── InvalidOperationException
│   ├── NotSupportedException
│   └── IndexOutOfRangeException
└── ApplicationException
```

**例外のスロー:**
```vb
' ファイル: Source/CorLib/System/CorArray.cls:563-570
If pSafeArray = vbNullPtr Then _
    Error.ArgumentNull "Arr", ArgumentNull_Array
If Index < LBound(Arr) Then _
    Error.IndexOutOfRange IndexOutOfRange_ArrayBounds
If Length < 0 Then _
    Error.IndexOutOfRange IndexOutOfRange_ArrayBounds
```

### 2.4 Static/Singleton パターン

VBCorLibでは「Static」クラスパターンを採用しています。

```vb
' ファイル: Source/CorLib/System.Collections/ComparerStatic.cls
' Comparer.Default のようなスタティックプロパティを提供
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `{TypeName}.cls` | 主要クラス | `ArrayList.cls`, `Exception.cls` |
| `{TypeName}Static.cls` | スタティックメソッド用クラス | `ArrayListStatic.cls`, `ComparerStatic.cls` |
| `I{InterfaceName}.cls` | インターフェース | `IComparer.cls`, `ICollection.cls` |
| `{TypeName}Enumerator.cls` | 列挙子クラス | `ArrayListEnumerator.cls` |
| `mod{Name}.bas` | 標準モジュール（旧命名） | 現在は `{Name}.bas` |
| `{Name}Helper.bas` | ヘルパー関数モジュール | `HashtableHelper.bas` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `m` | メンバー変数（Private） | `mCount`, `mItems`, `mCapacity` |
| `p` | ポインタ変数 | `pSafeArray`, `pReturnSafeArray` |
| `Cor` | VB予約語との競合回避 | `CorArray`, `CorString`, `CorDateTime` |
| `Static` | スタティッククラス | `ArrayListStatic`, `ComparerStatic` |
| `I` | インターフェース | `IComparer`, `ICollection`, `IEnumerable` |
| `Ex` | 拡張メソッド | `BinarySearchEx`, `CopyEx` |
| `New` | コンストラクタヘルパー | `NewArrayList`, `NewException` |

### 3.3 プログラム分類一覧

| 分類 | ディレクトリ | 説明 |
|------|-------------|------|
| 基本システム | `System` | 基本的な例外、配列、文字列操作 |
| コレクション | `System.Collections` | ArrayList, Hashtable等のコレクション |
| グローバリゼーション | `System.Globalization` | カルチャ、カレンダー |
| 入出力 | `System.IO` | ファイル、ストリーム操作 |
| セキュリティ | `System.Security.Cryptography` | 暗号化、ハッシュ |
| テキスト | `System.Text` | エンコーディング、StringBuilder |
| スレッド | `System.Threading` | スレッド、タイマー |
| 数値 | `System.Numerics` | BigInteger |
| リソース | `System.Resources` | リソースの読み書き |
| レジストリ | `Microsoft.Win32` | レジストリ操作 |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
VBCorLib-master/
├── Source/
│   ├── CorLib/                    # メインライブラリ
│   │   ├── CorLib.vbp             # プロジェクトファイル
│   │   ├── CorLib.Build.vbp       # ビルド用プロジェクト
│   │   ├── Cor/                   # コア機能（変換、インタラクション）
│   │   ├── System/                # 基本クラス（95ファイル）
│   │   ├── System.Collections/    # コレクション（42ファイル）
│   │   ├── System.Globalization/  # グローバリゼーション（28ファイル）
│   │   ├── System.IO/             # 入出力（35ファイル）
│   │   ├── System.IO.MemoryMappedFiles/ # メモリマップドファイル
│   │   ├── System.Numerics/       # BigInteger
│   │   ├── System.Resources/      # リソース管理（32ファイル）
│   │   ├── System.Security/       # セキュリティ基本
│   │   ├── System.Security.Cryptography/ # 暗号化（76ファイル）
│   │   ├── System.Text/           # テキスト処理（44ファイル）
│   │   ├── System.Threading/      # スレッド
│   │   ├── Microsoft.Win32/       # Windows API
│   │   ├── Microsoft.Win32.SafeHandles/ # セーフハンドル
│   │   ├── DocIncludes/           # ドキュメントインクルード
│   │   └── Resources/             # 組み込みリソース
│   ├── Tests/                     # 単体テスト
│   │   ├── CorLib.Tests.vbp       # テストプロジェクト
│   │   └── [各System.*]/          # 対応するテストファイル
│   └── Fakes/                     # テスト用モック
├── Examples/                      # 使用例
├── Install/                       # インストール関連
├── TypeLibrary/                   # 型ライブラリ
└── Template/                      # テンプレート
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `Source/CorLib/System` | 基本型・例外・ユーティリティ | `CorArray.cls`, `Exception.cls`, `Environment.cls` |
| `Source/CorLib/System.Collections` | コレクション型 | `ArrayList.cls`, `Hashtable.cls`, `Queue.cls` |
| `Source/CorLib/System.IO` | ファイル・ストリーム | `FileStream.cls`, `StreamReader.cls`, `Path.cls` |
| `Source/CorLib/System.Security.Cryptography` | 暗号化 | `RSACryptoServiceProvider.cls`, `SHA256.cls` |
| `Source/CorLib/System.Text` | エンコーディング | `UTF8Encoding.cls`, `StringBuilder.cls` |
| `Source/Tests` | 単体テスト | 各機能のテストクラス |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

VBCorLibは **.NET Framework互換レイヤー** として設計されています。

```
┌─────────────────────────────────────────────────────────────┐
│                    VB6 アプリケーション                      │
├─────────────────────────────────────────────────────────────┤
│                      VBCorLib API                           │
│  ┌──────────┬──────────┬──────────┬──────────┬──────────┐  │
│  │Collections│   IO     │  Text    │ Security │Globalization│
│  └──────────┴──────────┴──────────┴──────────┴──────────┘  │
├─────────────────────────────────────────────────────────────┤
│                     System (Core)                           │
│  ┌──────────┬──────────┬──────────┬──────────┬──────────┐  │
│  │ CorArray │CorString │Exception │Environment│  Buffer  │  │
│  └──────────┴──────────┴──────────┴──────────┴──────────┘  │
├─────────────────────────────────────────────────────────────┤
│                   Windows API / COM                         │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| API層 | 公開クラス・インターフェース | `ArrayList.cls`, `FileStream.cls` |
| 実装層 | 内部実装・ヘルパー | `HashtableHelper.bas`, `EncodingHelper.bas` |
| インターフェース層 | 抽象化 | `IComparer.cls`, `ICollection.cls` |
| ネイティブ層 | Windows API呼び出し | `Win32Native.bas`, `ExternalHelper.bas` |

### 5.3 データフロー

**コレクション操作の流れ:**
```
[クライアントコード] → [ArrayList.Add] → [EnsureCapacity]
                                      → [Insert] → [VariantCopyInd]
                                      → [mVersion++]
```

**ファイル読み込みの流れ:**
```
[クライアントコード] → [StreamReader.ReadLine]
    → [FileStream.Read] → [Windows API: ReadFile]
    → [Encoding.GetChars] → [文字列返却]
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

ライブラリの初期化は `Startup.bas` で行われます。

```vb
' ファイル: Source/CorLib/System/Startup.bas:38-46
Private Sub Main()
    InitHelper
    SetInIDE
    SetInDebugger
    InitMissing
    InitMathematics
    InitGlobalization
    InitEncoding
End Sub
```

### 6.2 ビジネスロジック

**コレクション操作（ArrayList）:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:424-433
Public Sub Insert(ByVal Index As Long, ByRef Value As Variant)
    If Index < 0 Or Index > mCount Then _
        Error.ArgumentOutOfRange "Index", ArgumentOutOfRange_ArrayListInsert

    EnsureCapacity mCount + 1
    InsertSpace Index, 1
    VariantCopyInd mItems(Index), Value
    mCount = mCount + 1
    mVersion = mVersion + 1
End Sub
```

**配列操作（CorArray）:**
```vb
' ファイル: Source/CorLib/System/CorArray.cls:491-494
Public Function BinarySearch(ByRef Arr As Variant, ByRef Value As Variant, Optional ByVal Comparer As IComparer) As Long
    ValidateArray Arr
    BinarySearch = InternalBinarySearch(Arr, LBound(Arr), Len1D(Arr), Value, Comparer)
End Function
```

### 6.3 データアクセス

**ファイルストリーム:**
- `FileStream.cls`: ファイルの読み書きを提供
- Windows APIを直接呼び出してファイル操作を実行

**レジストリアクセス:**
- `RegistryKey.cls`: Windowsレジストリへのアクセス
- `Microsoft.Win32/Registry.cls`: レジストリのルートキー

### 6.4 ユーティリティ/共通機能

**文字列操作:**
```vb
' Source/CorLib/System/CorString.cls
' .NET の String クラスに相当
```

**配列操作:**
```vb
' Source/CorLib/System/CorArray.cls
' .NET の Array クラスに相当
```

**日付操作:**
```vb
' Source/CorLib/System/CorDateTime.cls
' .NET の DateTime クラスに相当
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| インターフェース実装 | `Implements` による多態性 | 高 | 全クラス |
| スタティッククラス | `*Static.cls` パターン | 高 | `ArrayListStatic.cls` |
| 例外スロー | `Error.{Type}` パターン | 高 | 全クラス |
| バージョン管理 | `mVersion` による変更検出 | 中 | コレクションクラス |
| コールバック | `AddressOf` による関数ポインタ | 中 | `CorArray.cls` |

### 各パターンの詳細

#### パターン1: インターフェース実装パターン

**目的:** 複数のインターフェースを実装して多態性を実現

**実装例:**
```vb
' ファイル: Source/CorLib/System.Collections/ArrayList.cls:52-57
Implements IObject
Implements ICollection
Implements IEnumerable
Implements ICloneable
Implements IList
Implements IVersionable
```

**解説:** VB6ではクラスの継承がサポートされていないため、インターフェースを使用して多態性を実現しています。各インターフェースのメソッドは `Private` で実装し、`IInterface_MethodName` という命名規則に従います。

#### パターン2: スタティッククラスパターン

**目的:** インスタンス化せずに使用できるクラスメソッドを提供

**実装例:**
```vb
' 使用側コード
Dim list As ArrayList
Set list = ArrayList.Adapter(existingIList)

' Adapterメソッドは ArrayListStatic.cls で定義
```

**解説:** VB6にはスタティックメソッドがないため、`{ClassName}Static.cls` という別クラスを作成してスタティックメソッドを実装しています。

#### パターン3: コールバックパターン

**目的:** カスタム比較やフィルタリングを可能にする

**実装例:**
```vb
' ファイル: Source/CorLib/System/CorArray.cls:151-184
Public Function FindIndex(ByRef Arr As Variant, ByVal AddressOfPredicate As Long, ...) As Long
    ' ...
    Set Predicate = NewDelegate(AddressOfPredicate)

    For i = 1 To ElementCount
        If Predicate.Invoke(ByVal ElementPtr) Then
            FindIndex = ElementIndex + i - 1
            Exit Function
        End If
        ElementPtr = ElementPtr + Info.ElementSize
    Next
    ' ...
End Function
```

**解説:** `AddressOf` 演算子で関数ポインタを取得し、デリゲートとして呼び出すことでカスタム処理を実現しています。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: ArrayList への要素追加

**概要:** ArrayListに新しい要素を追加する処理の流れ

**処理フロー:**
```
[クライアント: list.Add(item)] → [Add] → [Insert] → [EnsureCapacity]
                                                   → [InsertSpace]
                                                   → [VariantCopyInd]
```

**詳細な追跡:**

1. **Add メソッド呼び出し** (`Source/CorLib/System.Collections/ArrayList.cls:88-91`)
   ```vb
   Public Function Add(ByRef Value As Variant) As Long
       Add = mCount
       Insert mCount, Value
   End Function
   ```

2. **Insert メソッド** (`Source/CorLib/System.Collections/ArrayList.cls:424-433`)
   ```vb
   Public Sub Insert(ByVal Index As Long, ByRef Value As Variant)
       If Index < 0 Or Index > mCount Then _
           Error.ArgumentOutOfRange "Index", ArgumentOutOfRange_ArrayListInsert

       EnsureCapacity mCount + 1
       InsertSpace Index, 1
       VariantCopyInd mItems(Index), Value
       mCount = mCount + 1
       mVersion = mVersion + 1
   End Sub
   ```

3. **EnsureCapacity** (`Source/CorLib/System.Collections/ArrayList.cls:734-748`)
   ```vb
   Private Sub EnsureCapacity(ByVal RequiredCapacity As Long)
       If RequiredCapacity <= mCapacity Then
           Exit Sub
       End If

       Dim NewCapacity As Long
       NewCapacity = mCapacity * 2

       If RequiredCapacity > NewCapacity Then
           NewCapacity = RequiredCapacity
       End If

       ReDim Preserve mItems(0 To NewCapacity - 1)
       mCapacity = NewCapacity
   End Sub
   ```

#### 例2: 配列のバイナリサーチ

**概要:** ソート済み配列から値を検索する処理

**処理フロー:**
```
[CorArray.BinarySearch] → [InternalBinarySearch] → [IComparer.Compare]
                                                 → [結果返却]
```

**詳細な追跡:**

1. **BinarySearch 呼び出し** (`Source/CorLib/System/CorArray.cls:491-494`)
   ```vb
   Public Function BinarySearch(ByRef Arr As Variant, ByRef Value As Variant, ...) As Long
       ValidateArray Arr
       BinarySearch = InternalBinarySearch(Arr, LBound(Arr), Len1D(Arr), Value, Comparer)
   End Function
   ```

2. **InternalBinarySearch** (`Source/CorLib/System/CorArray.cls:1300-1338`)
   ```vb
   Friend Function InternalBinarySearch(...) As Long
       ' ...
       Do While LowIndex <= HighIndex
           MiddleIndex = (LowIndex + HighIndex) \ 2

           Select Case Comparer.Compare(Arr(MiddleIndex), Value)
               Case 0
                   InternalBinarySearch = MiddleIndex
                   Exit Function
               Case Is > 0
                   HighIndex = MiddleIndex - 1
               Case Else
                   LowIndex = MiddleIndex + 1
           End Select
       Loop
       ' ...
   End Function
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `Source/CorLib/README.TXT` - バージョン履歴・変更点
3. `Source/CorLib/CorLib.vbp` - プロジェクト構成

#### 特定機能を理解したい場合
1. 対応する `.cls` ファイルのドキュメントコメント
2. `DocIncludes/` 内のHTMLドキュメント
3. `Tests/` 内のテストコード

#### 改修作業を行う場合
1. 対象クラスの `.cls` ファイル
2. 関連するインターフェース定義
3. 対応するテストクラス
4. `Static` クラス（存在する場合）

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初に参照 |
| `Source/CorLib/README.TXT` | バージョン履歴 | 変更点の確認 |
| `DocIncludes/*.html` | API詳細ドキュメント | API仕様の確認 |
| `Source/CorLib/CorLib.chm` | コンパイル済みヘルプ | オフラインでの参照 |
| `docs/code-to-docs/` | 生成ドキュメント | 設計仕様の確認 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `Attribute VB_Exposed = True` と `False` の違いは？
A: `True` の場合、そのクラスは外部から利用可能（パブリック）。`False` の場合はプロジェクト内部でのみ使用可能（プライベート）。

#### Q: `ByRef` と `ByVal` はいつ使い分ける？
A: VBCorLibでは、パフォーマンス向上のため、文字列や大きな構造体は `ByRef` を使用することが多い。値の変更を防ぎたい場合は `ByVal` を使用。

#### Q: `mVersion` 変数の目的は？
A: コレクションが変更されたことを追跡するため。列挙中にコレクションが変更された場合にエラーを発生させる。

#### Q: `Option Explicit` は何を意味する？
A: 変数の暗黙的な宣言を禁止し、すべての変数を明示的に宣言することを強制する。コード品質向上のため全ファイルで使用。

### デバッグのヒント

1. **IDE検出コード**: `Startup.bas` の `SetInIDE` メソッドでIDEでの実行を検出
2. **バージョン追跡**: コレクションクラスの `mVersion` でイテレーション中の変更を検出
3. **エラーハンドリング**: `On Error GoTo` パターンで例外をキャッチ
4. **ドキュメントコメント**: `''` で始まるコメントはドキュメント生成用

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| VB6 | Visual Basic 6.0 |
| ActiveX DLL | VB6で作成されるDLLコンポーネント |
| クラスモジュール | `.cls` ファイル、オブジェクト指向のクラス定義 |
| 標準モジュール | `.bas` ファイル、グローバル関数・変数 |
| SafeArray | VB6の配列を表すCOM構造体 |
| Variant | 任意の型を格納できるVB6のデータ型 |
| PropertyBag | VB6のシリアライズ用オブジェクト |
| IUnknown | COMの基本インターフェース |

### B. ファイル一覧（主要）

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `CorLib.vbp` | プロジェクトファイル | 全ファイルの参照・設定 |
| `System/CorArray.cls` | 配列操作 | Sort, BinarySearch, Copy |
| `System/Exception.cls` | 例外基底 | 全例外の基底クラス |
| `System.Collections/ArrayList.cls` | 動的配列 | Add, Remove, Sort |
| `System.Collections/Hashtable.cls` | ハッシュテーブル | キーバリューストア |
| `System.IO/FileStream.cls` | ファイルストリーム | ファイル読み書き |
| `System.Text/StringBuilder.cls` | 文字列ビルダー | 効率的な文字列連結 |

### C. 参考資料

- [VBCorLib オンラインドキュメント](http://www.kellyethridge.com/vbcorlib/doc/CorLib.html)
- [VBCorLib ブログ](http://vbcorlib.blogspot.com/)
- [.NET Framework ドキュメント](https://docs.microsoft.com/en-us/dotnet/api/) - API設計の参考
- [Visual Basic 6.0 ドキュメント](https://docs.microsoft.com/en-us/previous-versions/visualstudio/visual-basic-6/)
