# 機能設計書 106-IOException

## 概要

本ドキュメントは、VBCorLibライブラリにおける`IOException`クラスの機能設計を定義する。このクラスは、I/O（入出力）エラーが発生した場合にスローされる例外を表現する。

### 本機能の処理概要

**業務上の目的・背景**：ファイル操作やネットワーク通信などのI/O処理は、様々な理由で失敗する可能性がある。IOExceptionは、ディスクの空き容量不足、ファイルのロック、ネットワーク障害など、I/O関連のエラーを包括的に表現し、適切なエラーハンドリングを可能にする。FileNotFoundExceptionやDirectoryNotFoundExceptionなど、より具体的なI/O例外の基底クラスとしても機能する。

**機能の利用シーン**：
- ファイルの読み書き時にI/Oエラーが発生した場合
- ディスクの空き容量が不足している場合
- ファイルが他のプロセスによってロックされている場合
- ストリーム操作中にエラーが発生した場合
- デバイスI/Oエラーが発生した場合

**主要な処理内容**：
1. 例外インスタンスの生成と初期化（メッセージ、HResult、エラーコード、内部例外を設定）
2. 例外情報のプロパティ提供（Message、HResult、ErrorNumber等）
3. 例外チェーンの追跡（GetBaseException）とシリアライズ対応
4. IObject、Exception、SystemExceptionインターフェースの実装

**関連システム・外部連携**：VBCorLibの例外階層において、SystemException、Exceptionインターフェースを実装し、.NET Frameworkの例外モデルとの互換性を提供する。FileNotFoundException、DirectoryNotFoundException、PathTooLongExceptionの基底として機能する。

**権限による制御**：本クラスは権限制御なく、すべてのコードから利用可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし |

## 機能種別

例外処理 / エラーハンドリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Message | String | Yes | エラーメッセージ | なし（空の場合はデフォルトメッセージ使用） |
| HResult | Long | No | HRESULT値（デフォルト: COR_E_IO） | なし |
| ErrorCode | Long | No | VBエラーコード（デフォルト: vbDeviceIOError） | なし |
| InnerException | Exception | No | 内部例外 | なし |

### 入力データソース

コンストラクタ`Cor.NewIOException`関数経由でのパラメータ設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Message | String | エラーメッセージ |
| HResult | Long | 設定されたHRESULT値（デフォルト: COR_E_IO） |
| ErrorNumber | Long | 設定されたVBエラー番号（デフォルト: vbDeviceIOError (57)） |
| Source | String | 例外発生元の情報 |
| Data | IDictionary | ユーザー定義の追加情報 |
| InnerException | Exception | 内部例外オブジェクト |

### 出力先

例外オブジェクトとしてスロー、またはCatchブロックで捕捉

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し（NewIOException）
   └─ 新しいIOExceptionインスタンスを生成
2. Initメソッド実行
   └─ Message、HResult、ErrorNumber、InnerExceptionを設定
3. 例外スロー（Throw）
   └─ 例外がエラーハンドラに伝播
4. 例外捕捉（Catch）
   └─ 例外タイプを判定し、例外オブジェクトを取得
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[NewIOException呼び出し]
    B --> C[新しいインスタンス生成]
    C --> D[Class_InitProperties実行]
    D --> E[デフォルトHResult/ErrorNumber設定]
    E --> F[Initメソッド実行]
    F --> G[パラメータ値で上書き]
    G --> H[例外オブジェクト完成]
    H --> I[Throw実行]
    I --> J[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-106-1 | デフォルトメッセージ | Messageが未指定の場合、Arg_IOExceptionリソースからデフォルトメッセージを取得 | Message引数がvbNullStringの場合 |
| BR-106-2 | デフォルトErrorNumber | ErrorCodeが未指定の場合、vbDeviceIOError (57)を使用 | 常時 |
| BR-106-3 | デフォルトHResult | HResultが未指定の場合、COR_E_IOを使用 | 常時 |
| BR-106-4 | カスタムHResult/ErrorNumber | HResultとErrorCodeを個別に設定可能 | コンストラクタでパラメータ指定時 |

### 計算ロジック

特になし

## データベース操作仕様

該当なし（本クラスはデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本クラス自体が例外クラスであり、内部でエラーは発生しない | - |

### リトライ仕様

リトライ不要（例外クラスであり、再試行の概念なし）

## トランザクション仕様

該当なし

## パフォーマンス要件

例外生成は即座に完了すること（ミリ秒単位）

## セキュリティ考慮事項

- Messageにファイルパスが含まれる場合、機密情報（ユーザー名を含むパス等）の漏洩に注意

## 備考

- .NET FrameworkのSystem.IO.IOExceptionクラスとの互換性を意図した設計
- Persistable属性によりシリアライズ可能
- HResultとErrorNumberを個別に設定できる柔軟な設計
- FileNotFoundException、DirectoryNotFoundException、PathTooLongExceptionの基底として使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、例外クラスの継承階層とインターフェース実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | 例外の共通機能を提供するヘルパークラス |

**読解のコツ**: IOExceptionは他の例外クラスと異なり、HResultとErrorNumberをコンストラクタで設定できる。Class_InitPropertiesでデフォルト値が設定され、Initで上書きされる。

#### Step 2: エントリーポイントを理解する

例外オブジェクトの生成起点となるコンストラクタ関数を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Constructors.cls | `Source/CorLib/System/Constructors.cls` | NewIOException関数（396-398行目）が例外生成の起点 |

**主要処理フロー**:
1. **396行目**: 関数定義。Message（必須）、HResult、ErrorCode、InnerException（すべて任意）を受け取る
2. **397行目**: New IOExceptionでインスタンス生成
3. **398行目**: Initメソッドを呼び出し、各プロパティを初期化

#### Step 3: 例外クラスの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IOException.cls | `Source/CorLib/System.IO/IOException.cls` | 主要な例外クラス実装 |

**主要処理フロー**:
- **45-48行目**: Implements宣言。IObject、Exception、SystemExceptionを実装
- **52行目**: mBase As ExceptionBase - 共通機能をExceptionBaseに委譲
- **96-98行目**: HResultプロパティ。mBase.HResultを返却（設定可能）
- **105-107行目**: ErrorNumberプロパティ。mBase.ErrorNumberを返却（設定可能）
- **143-145行目**: Messageプロパティ。GetExceptionMessage関数でメッセージを取得
- **214-218行目**: Class_InitProperties。デフォルト値（COR_E_IO、vbDeviceIOError）を設定
- **220-225行目**: Initメソッド。Message、HResult、ErrorNumber、InnerExceptionを設定

### プログラム呼び出し階層図

```
Cor.NewIOException (Constructors.cls)
    │
    ├─ New IOException
    │      └─ Class_InitProperties
    │             ├─ Cor.NewExceptionBase (ExceptionBase生成)
    │             ├─ mBase.HResult = COR_E_IO
    │             └─ mBase.ErrorNumber = vbDeviceIOError
    │
    └─ IOException.Init
           ├─ mBase.Message = Message
           ├─ mBase.HResult = HResult
           ├─ mBase.ErrorNumber = ErrorNumber
           └─ Set mBase.InnerException = InnerException
```

### データフロー図

```
[入力]                          [処理]                              [出力]

Message ───────────┐
                   │
HResult ───────────┼──▶ IOException.Init ──────────────▶ 例外オブジェクト
                   │           │                                │
ErrorCode ─────────┤           ├─ mBase設定                     ├─ Message
                   │           │   (Message)                    ├─ HResult
InnerException ────┘           │   (HResult)                    ├─ ErrorNumber
                               │   (ErrorNumber)                └─ InnerException
                               └─   (InnerException)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IOException.cls | `Source/CorLib/System.IO/IOException.cls` | ソース | 主要な例外クラス実装 |
| ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | ソース | 例外共通機能のヘルパークラス |
| Constructors.cls | `Source/CorLib/System/Constructors.cls` | ソース | コンストラクタ関数群 |
| FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | ソース | 派生例外クラス |
| DirectoryNotFoundException.cls | `Source/CorLib/System.IO/DirectoryNotFoundException.cls` | ソース | 派生例外クラス |
| PathTooLongException.cls | `Source/CorLib/System.IO/PathTooLongException.cls` | ソース | 派生例外クラス |
| IOError.cls | `Source/CorLib/System.IO/IOError.cls` | ソース | I/Oエラー生成ヘルパー |
