# 機能設計書 107-FileNotFoundException

## 概要

本ドキュメントは、VBCorLibライブラリにおける`FileNotFoundException`クラスの機能設計を定義する。このクラスは、指定されたファイルが見つからない場合にスローされる例外を表現する。

### 本機能の処理概要

**業務上の目的・背景**：ファイル操作を行うアプリケーションにおいて、指定されたファイルが存在しない場合は頻繁に発生するエラーである。FileNotFoundExceptionは、ファイルが見つからないという具体的な状況を明示し、ファイル名を含むエラーメッセージを提供することで、問題の特定と解決を容易にする。

**機能の利用シーン**：
- ファイルを開こうとしたが存在しない場合
- ファイル読み込み時にファイルが見つからない場合
- ファイルコピー/移動時にソースファイルが存在しない場合
- DLLやリソースファイルのロード時にファイルが見つからない場合

**主要な処理内容**：
1. 例外インスタンスの生成と初期化（ファイル名、メッセージ、内部例外を設定）
2. 例外情報のプロパティ提供（FileName、Message、HResult、ErrorNumber等）
3. ファイル名に基づく自動メッセージ生成
4. IObject、Exception、SystemException、IOExceptionインターフェースの実装

**関連システム・外部連携**：VBCorLibの例外階層において、IOException、SystemException、Exceptionインターフェースを実装し、.NET Frameworkの例外モデルとの互換性を提供する。

**権限による制御**：本クラスは権限制御なく、すべてのコードから利用可能である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面との直接的な関連なし |

## 機能種別

例外処理 / エラーハンドリング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| FileName | String | No | 見つからなかったファイルの名前 | なし |
| Message | String | No | エラーメッセージ（省略時はファイル名から自動生成） | なし |
| InnerException | Exception | No | 内部例外 | なし |

### 入力データソース

コンストラクタ`Cor.NewFileNotFoundException`関数経由でのパラメータ設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| FileName | String | 見つからなかったファイルの名前 |
| Message | String | エラーメッセージ |
| HResult | Long | COR_E_FILENOTFOUND定数値 |
| ErrorNumber | Long | vbFileNotFound (53) |
| Source | String | 例外発生元の情報 |
| Data | IDictionary | ユーザー定義の追加情報 |
| InnerException | Exception | 内部例外オブジェクト |

### 出力先

例外オブジェクトとしてスロー、またはCatchブロックで捕捉

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し（NewFileNotFoundException）
   └─ 新しいFileNotFoundExceptionインスタンスを生成
2. Initメソッド実行
   ├─ FileName、InnerExceptionを設定
   └─ Message生成（指定がなければファイル名から自動生成）
3. 例外スロー（Throw）
   └─ 例外がエラーハンドラに伝播
4. 例外捕捉（Catch）
   └─ 例外タイプを判定し、例外オブジェクトを取得
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[NewFileNotFoundException呼び出し]
    B --> C[新しいインスタンス生成]
    C --> D[Initメソッド実行]
    D --> E[FileName設定]
    E --> F{Messageが指定されている?}
    F -->|Yes| G[指定されたMessageを設定]
    F -->|No| H{FileNameが指定されている?}
    H -->|Yes| I[ファイル名を含むメッセージを自動生成]
    H -->|No| J[デフォルトメッセージを使用]
    G --> K[例外オブジェクト完成]
    I --> K
    J --> K
    K --> L[Throw実行]
    L --> M[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-107-1 | デフォルトメッセージ | MessageもFileNameも未指定の場合、IO_FileNotFoundリソースからデフォルトメッセージを取得 | Message、FileNameともにvbNullStringの場合 |
| BR-107-2 | ファイル名含むメッセージ | FileNameが指定され、Messageが未指定の場合、"Could not find file '{FileName}'."形式のメッセージを生成 | FileNameが指定、Messageが未指定の場合 |
| BR-107-3 | 固定ErrorNumber | ErrorNumberは常にvbFileNotFound (53)を返却 | 常時 |
| BR-107-4 | 固定HResult | HResultは常にCOR_E_FILENOTFOUNDを返却 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

該当なし（本クラスはデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 本クラス自体が例外クラスであり、内部でエラーは発生しない | - |

### リトライ仕様

リトライ不要（例外クラスであり、再試行の概念なし）

## トランザクション仕様

該当なし

## パフォーマンス要件

例外生成は即座に完了すること（ミリ秒単位）

## セキュリティ考慮事項

- FileNameにフルパスが含まれる場合、機密情報（ユーザー名を含むパス等）の漏洩に注意
- ログ出力時やエラー表示時にファイルパスの扱いに注意

## 備考

- .NET FrameworkのSystem.IO.FileNotFoundExceptionクラスとの互換性を意図した設計
- Persistable属性によりシリアライズ可能
- IOExceptionを継承（Implements）
- ファイル名を含む自動メッセージ生成機能が特徴

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、例外クラスの継承階層とインターフェース実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | 例外の共通機能を提供するヘルパークラス |
| 1-2 | IOException.cls | `Source/CorLib/System.IO/IOException.cls` | 基底となるI/O例外クラス |

**読解のコツ**: FileNotFoundExceptionはIOExceptionインターフェースを実装するが、HResultとErrorNumberは固定値を使用する（IOExceptionの可変値とは異なる）。

#### Step 2: エントリーポイントを理解する

例外オブジェクトの生成起点となるコンストラクタ関数を確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Constructors.cls | `Source/CorLib/System/Constructors.cls` | NewFileNotFoundException関数（408-410行目）が例外生成の起点 |

**主要処理フロー**:
1. **408行目**: 関数定義。FileName、Message、InnerException（すべて任意）を受け取る
2. **409行目**: New FileNotFoundExceptionでインスタンス生成
3. **410行目**: Initメソッドを呼び出し、各プロパティを初期化

#### Step 3: 例外クラスの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | 主要な例外クラス実装 |

**主要処理フロー**:
- **45-49行目**: Implements宣言。IObject、Exception、SystemException、IOExceptionを実装
- **54行目**: mBase As ExceptionBase - 共通機能をExceptionBaseに委譲
- **74-76行目**: FileNameプロパティ。mBase.GetValueで取得
- **108-110行目**: HResultプロパティ。COR_E_FILENOTFOUND定数を返却
- **117-119行目**: ErrorNumberプロパティ。vbFileNotFound (53)を返却
- **155-157行目**: Messageプロパティ。GetExceptionMessage関数でメッセージを取得
- **230-240行目**: Initメソッド。FileName設定、Messageの条件付き生成

#### Step 4: テストコードで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | FileNotFoundExceptionTests.cls | `Source/Tests/System.IO/FileNotFoundExceptionTests.cls` | 各機能の期待動作を確認 |

**主要処理フロー**:
- **24-31行目**: デフォルトコンストラクタでのMessageテスト（"Unable to find the specified file."）
- **42-49行目**: ファイル名コンストラクタでのFileNameテスト
- **52-59行目**: ファイル名からの自動メッセージ生成テスト（"Could not find file 'abc.txt'."）

### プログラム呼び出し階層図

```
Cor.NewFileNotFoundException (Constructors.cls)
    │
    ├─ New FileNotFoundException
    │      └─ Class_InitProperties
    │             └─ Cor.NewExceptionBase (ExceptionBase生成)
    │
    └─ FileNotFoundException.Init
           ├─ mBase.SetValue(PropFileName, FileName)
           ├─ [FileNameあり && Messageなし]
           │      └─ mBase.Message = GetResourceString(IO_FileNotFound_Name, FileName)
           ├─ [それ以外]
           │      └─ mBase.Message = Message
           └─ Set mBase.InnerException = InnerException
```

### データフロー図

```
[入力]                          [処理]                              [出力]

FileName ──────────┐
                   │
                   ├──▶ FileNotFoundException.Init ────▶ 例外オブジェクト
                   │           │                                │
Message ───────────┤           ├─ mBase.SetValue(FileName)      ├─ FileName
                   │           ├─ 条件付きMessage生成            ├─ Message
InnerException ────┘           └─ mBase設定                      ├─ HResult (固定)
                                                                ├─ ErrorNumber (53)
                                                                └─ InnerException
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | ソース | 主要な例外クラス実装 |
| ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | ソース | 例外共通機能のヘルパークラス |
| Constructors.cls | `Source/CorLib/System/Constructors.cls` | ソース | コンストラクタ関数群 |
| IOException.cls | `Source/CorLib/System.IO/IOException.cls` | ソース | 基底I/O例外クラス |
| FileNotFoundExceptionTests.cls | `Source/Tests/System.IO/FileNotFoundExceptionTests.cls` | テスト | ユニットテスト |
