# 機能設計書 11-Random

## 概要

本ドキュメントは、VBCorLibライブラリにおける擬似乱数生成機能「Random」クラスの機能設計について記述する。

### 本機能の処理概要

Randomクラスは、擬似乱数生成器として、整数、浮動小数点数、バイト配列に対する乱数生成機能を提供する。.NET FrameworkのSystem.Randomクラスと互換性のあるAPIを持ち、VBA/VB6環境で高品質な乱数生成を実現する。

**業務上の目的・背景**：アプリケーション開発において、テストデータの生成、シミュレーション、ゲームロジック、暗号化前の初期化ベクトル生成など、様々な場面で乱数が必要となる。VBAの標準Rnd関数をラップし、より使いやすいオブジェクト指向インターフェースを提供することで、開発者の生産性を向上させる。

**機能の利用シーン**：
- テストデータの自動生成
- ゲームやシミュレーションにおけるランダム要素の実装
- 統計処理におけるサンプリング
- 一意識別子の生成補助

**主要な処理内容**：
1. 指定範囲内の整数乱数生成（NextLong、NextRange）
2. 0.0以上1.0未満の浮動小数点乱数生成（NextDouble）
3. バイト配列への乱数充填（NextBytes）
4. シード値による乱数系列の初期化

**関連システム・外部連携**：VBAのRnd関数およびRandomize文を内部で使用している。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内の乱数生成テスト（RandomTests）の実行 |

## 機能種別

計算処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| MaxValue | Long | No | NextLongメソッドの上限値（デフォルト: &H7FFFFFFF） | 0以上であること |
| MinValue | Long | Yes | NextRangeメソッドの下限値 | MaxValue以下であること |
| MaxValue | Long | Yes | NextRangeメソッドの上限値 | MinValue以上であること |
| Bytes | Byte() | Yes | NextBytesメソッドの出力先配列 | 初期化済み配列であること |
| Seed | Long | No | コンストラクタのシード値 | 任意の整数値 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| NextLong戻り値 | Long | 0以上MaxValue未満の整数乱数 |
| NextRange戻り値 | Long | MinValue以上MaxValue未満の整数乱数 |
| NextDouble戻り値 | Double | 0.0以上1.0未満の浮動小数点乱数 |
| Bytes配列 | Byte() | 乱数で充填されたバイト配列 |

### 出力先

呼び出し元への戻り値として返却、またはByRef引数への直接書き込み

## 処理フロー

### 処理シーケンス

```
1. インスタンス生成
   └─ シード値の初期化（Randomize文またはユーザー指定シード）
2. NextLong/NextRange呼び出し
   └─ パラメータバリデーション
   └─ VBA.Rnd関数による乱数生成
   └─ 範囲調整して戻り値として返却
3. NextBytes呼び出し
   └─ 配列の初期化チェック
   └─ 配列の各要素に0-255の乱数を設定
4. NextDouble呼び出し
   └─ VBA.Rnd関数の戻り値をそのまま返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|NextLong| C{MaxValue < 0?}
    C -->|Yes| D[ArgumentOutOfRangeException]
    C -->|No| E[NextNumber呼び出し]
    E --> F[結果返却]
    B -->|NextRange| G{MaxValue < MinValue?}
    G -->|Yes| H[ArgumentOutOfRangeException]
    G -->|No| I[NextNumber呼び出し]
    I --> F
    B -->|NextBytes| J{配列初期化済み?}
    J -->|No| K[ArgumentNullException]
    J -->|Yes| L[ループで配列充填]
    L --> F
    B -->|NextDouble| M[VBA.Rnd返却]
    M --> F
    F --> N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 上限値制約 | NextLongのMaxValueは0以上でなければならない | NextLong呼び出し時 |
| BR-002 | 範囲制約 | NextRangeのMaxValueはMinValue以上でなければならない | NextRange呼び出し時 |
| BR-003 | 配列初期化必須 | NextBytesの引数配列は初期化されていなければならない | NextBytes呼び出し時 |
| BR-004 | シード再現性 | 同一シード値からは同一の乱数系列が生成される | 新規インスタンス生成時 |

### 計算ロジック

```
NextNumber(MinValue, MaxValue) = Int(VBA.Rnd * (MaxValue - MinValue)) + MinValue
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentOutOfRangeException | NextLongのMaxValueが負の値 | MaxValueを0以上に修正 |
| - | ArgumentOutOfRangeException | NextRangeのMinValue > MaxValue | 範囲指定を修正 |
| - | ArgumentNullException | NextBytesの配列が未初期化 | 配列を初期化してから呼び出し |

### リトライ仕様

乱数生成処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 単一乱数生成: 数マイクロ秒以内
- 大規模配列充填（10,000要素）: 数ミリ秒以内

## セキュリティ考慮事項

- 本クラスは暗号学的に安全な乱数生成器ではない
- 暗号用途にはRNGCryptoServiceProviderを使用すること
- シード値が既知の場合、乱数系列を予測可能

## 備考

- VBA標準のRnd関数をラップしているため、乱数品質はVBAの実装に依存する
- .NET FrameworkのSystem.Randomとの完全な互換性は保証されない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Randomクラスは内部状態を持たないシンプルな設計となっている。VBAのRnd関数がグローバルな乱数状態を管理するため、クラス独自の状態管理は最小限となっている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Random.cls | `Source/CorLib/System/Random.cls` | クラスの全体構造とIObjectインターフェースの実装 |

**読解のコツ**: VB6/VBAのクラスモジュールでは、`Attribute`行でクラスの特性（Persistable等）が定義されている。`Implements IObject`によりIObjectインターフェースを実装している。

#### Step 2: エントリーポイントを理解する

Randomクラスのエントリーポイントは各乱数生成メソッドである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Random.cls | `Source/CorLib/System/Random.cls` | 公開メソッド（NextLong、NextRange、NextBytes、NextDouble） |

**主要処理フロー**:
1. **54-59行目**: NextLongメソッド - MaxValue未満の乱数を返す
2. **69-74行目**: NextRangeメソッド - 指定範囲内の乱数を返す
3. **80-88行目**: NextBytesメソッド - 配列を乱数で充填
4. **96-98行目**: NextDoubleメソッド - 0.0-1.0の乱数を返す

#### Step 3: コア処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Random.cls | `Source/CorLib/System/Random.cls` | NextNumber関数とInit処理 |

**主要処理フロー**:
- **130-132行目**: Class_Initialize - InitRandom呼び出しで乱数エンジン初期化
- **134-136行目**: Init - シード値を使ってVBA.Rndを初期化
- **142-144行目**: NextNumber - 実際の乱数計算ロジック

### プログラム呼び出し階層図

```
Random (インスタンス)
    │
    ├─ NextLong(MaxValue)
    │      └─ NextNumber(0, MaxValue)
    │              └─ VBA.Rnd
    │
    ├─ NextRange(MinValue, MaxValue)
    │      └─ NextNumber(MinValue, MaxValue)
    │              └─ VBA.Rnd
    │
    ├─ NextBytes(Bytes())
    │      └─ NextNumber(0, 256) [ループ]
    │              └─ VBA.Rnd
    │
    └─ NextDouble()
           └─ VBA.Rnd
```

### データフロー図

```
[入力]               [処理]                    [出力]

MaxValue ─────────▶ NextLong ────────────────▶ Long型乱数
                         │
                         ▼
                    NextNumber
                         │
                         ▼
                      VBA.Rnd

MinValue,MaxValue ──▶ NextRange ──────────────▶ Long型乱数
                         │
                         ▼
                    NextNumber

Byte配列 ──────────▶ NextBytes ───────────────▶ 充填済み配列
                    (ByRef)

（なし） ──────────▶ NextDouble ──────────────▶ Double型乱数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Random.cls | `Source/CorLib/System/Random.cls` | ソース | メインクラス定義 |
| RandomTests.cls | `Source/Tests/System/RandomTests.cls` | テスト | 単体テストケース |
| Constructors.cls | `Source/CorLib/Constructors.cls` | ソース | NewRandomコンストラクタ提供 |
| Error.cls | `Source/CorLib/System/Error.cls` | ソース | 例外スロー処理 |
