# 機能設計書 12-BitConverter

## 概要

本ドキュメントは、VBCorLibライブラリにおけるデータ型変換機能「BitConverter」クラスの機能設計について記述する。

### 本機能の処理概要

BitConverterクラスは、基本データ型（Boolean、Integer、Long、Double、Single、Currency、Date、Decimal、Int64）とバイト配列間の相互変換機能を提供する静的クラスである。.NET FrameworkのSystem.BitConverterクラスと互換性のあるAPIを持ち、バイナリデータ処理、ファイルI/O、ネットワーク通信などで必要となる低レベルデータ変換を実現する。

**業務上の目的・背景**：ファイルフォーマットの解析、バイナリ通信プロトコルの実装、データのシリアライズ/デシリアライズなど、バイト列と数値型の相互変換が必要な場面は多い。VBAには標準でこのような変換機能がないため、BitConverterクラスがこのギャップを埋める役割を果たす。

**機能の利用シーン**：
- バイナリファイルの読み書き処理
- ネットワークプロトコルのパケット構築・解析
- データのシリアライズ/デシリアライズ
- ハッシュ値やチェックサムの計算補助
- メモリ上のバイト表現の検査（デバッグ用途）

**主要な処理内容**：
1. 各種データ型からバイト配列への変換（GetBytes）
2. バイト配列から各種データ型への変換（ToBoolean、ToInt16、ToInt32、ToInt64、ToSingle、ToDouble、ToCurrency、ToDate、ToDecimal）
3. バイト配列の16進数文字列表現への変換（ToString）
4. 浮動小数点数と整数間のビットレベル変換（DoubleToInt64Bits、Int64BitsToDouble等）

**関連システム・外部連携**：特定の外部システムとの連携はないが、暗号化クラスやストリームクラスと組み合わせて使用されることが多い。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内のビット変換テスト（BitConverterTests）の実行 |
| 13 | File Hex Viewer | 補助機能 | バイト配列から整数への変換（コメントアウト状態） |

## 機能種別

データ変換処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Value | Variant | Yes | GetBytesの変換対象値 | サポートされるデータ型であること |
| Value | Byte() | Yes | To*メソッドの変換元バイト配列 | 初期化済みであること |
| StartIndex | Long | Yes | To*メソッドの開始位置 | 配列範囲内であること |
| Count | Long | No | ToStringメソッドの変換バイト数 | 正の整数 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| GetBytes戻り値 | Byte() | データ型のバイト表現 |
| ToBoolean戻り値 | Boolean | バイトからの変換結果 |
| ToInt16戻り値 | Integer | 2バイトからの変換結果 |
| ToInt32戻り値 | Long | 4バイトからの変換結果 |
| ToInt64戻り値 | Int64 | 8バイトからの変換結果 |
| ToSingle戻り値 | Single | 4バイトからの変換結果 |
| ToDouble戻り値 | Double | 8バイトからの変換結果 |
| ToCurrency戻り値 | Currency | 8バイトからの変換結果 |
| ToDate戻り値 | Date | 8バイトからの変換結果 |
| ToDecimal戻り値 | Variant(Decimal) | 16バイトからの変換結果 |
| ToString戻り値 | String | 16進数文字列表現 |

### 出力先

呼び出し元への戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. GetBytes呼び出し
   └─ データ型判定（VarType）
   └─ SafeArrayCreateVectorExで適切なサイズのバイト配列作成
   └─ メモリコピーでバイト列に変換
   └─ バイト配列を返却

2. To*メソッド呼び出し
   └─ 配列とインデックスのバリデーション
   └─ 開始位置から必要バイト数を読み取り
   └─ As*関数で型変換して返却

3. ToString呼び出し
   └─ 配列バリデーション
   └─ 範囲パラメータの解決
   └─ 各バイトを2桁16進数に変換
   └─ ハイフン区切りで連結して返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|GetBytes| C[VarType判定]
    C --> D{サポート型?}
    D -->|No| E[NotSupportedException]
    D -->|Yes| F[バイト配列作成]
    F --> G[メモリコピー]
    G --> H[結果返却]
    B -->|To*| I[配列バリデーション]
    I --> J{範囲内?}
    J -->|No| K[ArgumentOutOfRangeException]
    J -->|Yes| L[バイト読み取り]
    L --> M[型変換]
    M --> H
    B -->|ToString| N[配列バリデーション]
    N --> O[16進数変換ループ]
    O --> P[文字列連結]
    P --> H
    H --> Q[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | リトルエンディアン | すべてのバイト配列はリトルエンディアン形式 | 全変換処理 |
| BR-002 | 型サポート制約 | GetBytesはBoolean、Byte、Integer、Long、Int64、Single、Double、Currency、Date、Decimalのみサポート | GetBytes呼び出し時 |
| BR-003 | Decimal互換性 | DecimalのバイトレイアウトはVBとは異なり、.NET互換形式 | Decimal変換時 |
| BR-004 | 配列初期化必須 | To*メソッドの配列引数は初期化されていなければならない | To*メソッド呼び出し時 |
| BR-005 | Boolean変換 | 0以外のバイト値はTrueに変換される | ToBoolean呼び出し時 |

### 計算ロジック

**Decimalレイアウト変換**:
```
VB Decimal:
  bytes 0-1: Variant type (14)
  bytes 2-3: precision and sign
  bytes 4-7: highest 32 bits
  bytes 8-11: lowest 32 bits
  bytes 12-15: middle 32 bits

.NET Decimal (出力形式):
  bytes 0-3: lowest 32 bits
  bytes 4-7: middle 32 bits
  bytes 8-11: highest 32 bits
  bytes 12-13: unused (zero)
  bytes 14-15: precision and sign
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | NotSupportedException | サポート外のデータ型をGetBytesに渡した | サポートされるデータ型を使用 |
| - | ArgumentNullException | 未初期化配列をTo*メソッドに渡した | 配列を初期化してから呼び出し |
| - | ArgumentOutOfRangeException | StartIndexが配列範囲外 | 有効なインデックスを指定 |
| - | ArgumentOutOfRangeException | Decimalの精度が28を超過 | 有効なDecimal値を使用 |
| - | OverflowException | ToDateでDate範囲外の値 | 有効な日付範囲のミリ秒値を使用 |

### リトライ仕様

データ変換処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 単一変換: 数マイクロ秒以内
- SafeArrayCreateVectorExを使用し、配列記述子とデータを一括確保して効率化

## セキュリティ考慮事項

- メモリ操作を伴うため、不正なインデックス指定によるバッファオーバーフローを防止するバリデーション実装済み
- センシティブなデータのバイト表現には注意が必要

## 備考

- このクラスは静的クラスとして設計されており、インスタンス化不要
- x86アーキテクチャのリトルエンディアンを前提としている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

BitConverterはステートレスな静的クラスであり、内部状態を持たない。重要なのは各データ型のバイトサイズとメモリレイアウトである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BitConverter.cls | `Source/CorLib/System/BitConverter.cls` | クラス構造、VB_Creatable = False（静的クラス） |

**読解のコツ**: VB6/VBAでは静的クラスをVB_Creatable = Falseで表現し、モジュールのように使用する。

#### Step 2: GetBytesメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BitConverter.cls | `Source/CorLib/System/BitConverter.cls` | GetBytesメソッドの型別処理 |

**主要処理フロー**:
1. **160-242行目**: GetBytesメソッド - VarTypeによる型判定と変換処理
2. **166-169行目**: vbLong型の処理 - 4バイト配列作成とAsLongによるコピー
3. **170-176行目**: vbBoolean型の処理 - 1バイト配列作成、Trueなら1をセット
4. **194-202行目**: vbUserDefinedType(Int64)型の処理 - 8バイト配列作成
5. **203-237行目**: vbDecimal型の処理 - VBから.NETレイアウトへの変換

#### Step 3: To*メソッド群を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | BitConverter.cls | `Source/CorLib/System/BitConverter.cls` | 各To*メソッドの実装 |

**主要処理フロー**:
- **264-267行目**: ToBoolean - バイト値が0より大きければTrue
- **291-294行目**: ToDouble - 8バイトをAsDoubleで変換
- **351-354行目**: ToInt16 - 2バイトをAsWordで変換
- **378-381行目**: ToInt32 - 4バイトをAsLongで変換
- **406-409行目**: ToInt64 - 8バイトをAssignQWordで変換
- **521-537行目**: ToDecimal - .NETレイアウトからVBレイアウトへの変換

#### Step 4: ヘルパーメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | BitConverter.cls | `Source/CorLib/System/BitConverter.cls` | ValidateArrayArguments関数 |

**主要処理フロー**:
- **605-615行目**: ValidateArrayArguments - 配列、インデックス、必要バイト数のバリデーション

### プログラム呼び出し階層図

```
BitConverter (静的クラス)
    │
    ├─ GetBytes(Value)
    │      ├─ SafeArrayCreateVectorEx
    │      ├─ AsLong/AsWord/AsDouble/AsCurr/AssignQWord
    │      └─ CopyMemory (Decimal用)
    │
    ├─ ToBoolean/ToInt16/ToInt32/ToInt64/ToSingle/ToDouble/ToCurrency/ToDate/ToDecimal
    │      ├─ ValidateArrayArguments
    │      └─ AsLong/AsWord/AsDouble/AsCurr/AssignQWord
    │
    ├─ ToString(Value, StartIndex, Count)
    │      ├─ MakeDefaultStepRange
    │      ├─ ValidateArrayArguments
    │      ├─ SysAllocStringLen
    │      └─ AllocChars/FreeChars
    │
    └─ DoubleToInt64Bits/Int64BitsToDouble/SingleToInt32Bits/Int32BitsToSingle/CurrencyToInt64Bits/Int64BitsToCurrency
           └─ AsDouble/AsSingle/AsCurr (メモリ再解釈)
```

### データフロー図

```
[入力]               [処理]                    [出力]

Long値 ───────────▶ GetBytes ────────────────▶ Byte(0 To 3)
                         │
                         ▼
                 SafeArrayCreateVectorEx
                         │
                         ▼
                     AsLong

Byte(0 To 7) ────────▶ ToDouble ─────────────▶ Double値
    │                    │
    │                    ▼
    │            ValidateArrayArguments
    │                    │
    │                    ▼
    └──────────────▶ AsDouble

Byte配列 ────────────▶ ToString ─────────────▶ "01-EF-CD-AB"
                         │
                         ▼
                   16進数変換ループ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| BitConverter.cls | `Source/CorLib/System/BitConverter.cls` | ソース | メインクラス定義 |
| BitConverterTests.cls | `Source/Tests/System/BitConverterTests.cls` | テスト | 単体テストケース |
| Statics.bas | `Source/CorLib/Statics.bas` | ソース | 静的クラスのグローバル変数定義 |
| Helper.bas | `Source/CorLib/Helper.bas` | ソース | AsLong/AsDouble等の低レベル関数 |
| MemoryMethods.bas | `Source/CorLib/MemoryMethods.bas` | ソース | CopyMemory等のメモリ操作 |
