# 機能設計書 13-Convert

## 概要

本ドキュメントは、VBCorLibライブラリにおけるデータ型変換機能「Convert」クラスの機能設計について記述する。

### 本機能の処理概要

Convertクラスは、データ型間の変換機能を提供する静的クラスである。特にBase64エンコーディング/デコーディング、数値の基数変換（2進数、8進数、16進数）、型変換メソッドなど、.NET FrameworkのSystem.Convertクラスと互換性のあるAPIを提供する。

**業務上の目的・背景**：データのシリアライズ、暗号化データの文字列表現、設定ファイルへの数値保存、デバッグ出力など、様々な場面でデータ型変換が必要となる。VBAの標準変換関数を拡張し、Base64や任意基数変換など、より高度な変換機能を提供する。

**機能の利用シーン**：
- バイナリデータのBase64文字列への変換（メール添付、JSONペイロード等）
- Base64文字列からのバイナリデータ復元
- デバッグ用の数値表示（16進数、2進数）
- 設定ファイルへの数値保存（文字列変換）
- 型安全な変換処理

**主要な処理内容**：
1. Base64エンコーディング（ToBase64String、ToBase64CharArray）
2. Base64デコーディング（FromBase64String、FromBase64CharArray）
3. 基数変換（ToString - 2/8/10/16進数）
4. 型変換メソッド（ToBoolean、ToByte、ToInt16、ToInt32、ToInt64、ToSingle、ToDouble、ToDateTime等）

**関連システム・外部連携**：暗号化クラスと組み合わせてBase64エンコーディングを使用するケースが多い。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内のConvertテスト（ConvertTests）の実行 |

## 機能種別

データ変換処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| InArray | Byte() | Yes | ToBase64Stringの入力バイト配列 | 初期化済みであること |
| Offset | Long | No | ToBase64Stringの開始オフセット | 配列範囲内 |
| Length | Long | No | ToBase64Stringの変換長 | 0以上、配列範囲内 |
| Options | Base64FormattingOptions | No | 改行挿入オプション | 有効な列挙値 |
| s | String | Yes | FromBase64Stringの入力文字列 | 有効なBase64文字列 |
| InArray | Integer() | Yes | FromBase64CharArrayの入力文字配列 | 有効なBase64文字 |
| Value | Variant | Yes | ToStringの変換対象値 | 数値型であること |
| ToBase | Long | No | ToStringの基数（2/8/10/16） | 2、8、10、16のいずれか |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ToBase64String戻り値 | String | Base64エンコードされた文字列 |
| ToBase64CharArray戻り値 | Long | 書き込まれた文字数 |
| FromBase64String戻り値 | Byte() | デコードされたバイト配列 |
| FromBase64CharArray戻り値 | Byte() | デコードされたバイト配列 |
| ToString戻り値 | String | 指定基数での文字列表現 |
| To*型戻り値 | 各型 | 変換された値 |

### 出力先

呼び出し元への戻り値として返却、またはByRef引数への直接書き込み

## 処理フロー

### 処理シーケンス

```
1. ToBase64String呼び出し
   └─ パラメータバリデーション
   └─ 3バイト → 4文字の変換ループ
   └─ パディング処理（'='）
   └─ オプションに応じて76文字毎に改行挿入
   └─ Base64文字列を返却

2. FromBase64String呼び出し
   └─ 空白文字除去
   └─ 長さバリデーション（4の倍数チェック）
   └─ 4文字 → 3バイトの変換ループ
   └─ パディング処理
   └─ バイト配列を返却

3. ToString(Value, ToBase)呼び出し
   └─ 型バリデーション
   └─ 基数バリデーション（2/8/10/16のみ）
   └─ 数値を文字列に変換（負数の2の補数表現対応）
   └─ 文字列を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|ToBase64String| C[配列バリデーション]
    C --> D[3バイトずつ変換]
    D --> E[パディング追加]
    E --> F{改行オプション?}
    F -->|Yes| G[76文字毎に改行挿入]
    F -->|No| H[結果返却]
    G --> H
    B -->|FromBase64String| I[空白除去]
    I --> J{長さが4の倍数?}
    J -->|No| K[FormatException]
    J -->|Yes| L[4文字ずつ変換]
    L --> M[パディング処理]
    M --> H
    B -->|ToString| N[型・基数バリデーション]
    N --> O{基数は2/8/10/16?}
    O -->|No| P[ArgumentException]
    O -->|Yes| Q[基数変換]
    Q --> H
    H --> R[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Base64文字セット | A-Z、a-z、0-9、+、/、=（パディング）を使用 | Base64変換時 |
| BR-002 | Base64改行挿入 | InsertLineBreaksオプション時、76文字毎にCRLF挿入 | ToBase64String、InsertLineBreaks指定時 |
| BR-003 | 基数制限 | ToStringの基数は2、8、10、16のみサポート | ToString呼び出し時 |
| BR-004 | 負数の表現 | 2進数、8進数、16進数での負数は2の補数表現 | ToString(負数, 非10進数)呼び出し時 |
| BR-005 | 空白許容 | FromBase64Stringは入力文字列内の空白を無視 | FromBase64String呼び出し時 |

### 計算ロジック

**Base64エンコーディング**:
```
3バイト (24ビット) → 4文字 (6ビット x 4)
入力バイト: byte1, byte2, byte3
出力文字:
  char1 = Base64Table[byte1 >> 2]
  char2 = Base64Table[((byte1 & 3) << 4) | (byte2 >> 4)]
  char3 = Base64Table[((byte2 & 15) << 2) | (byte3 >> 6)]
  char4 = Base64Table[byte3 & 63]
```

**基数変換（16進数の例）**:
```
Value = 255
Result = "FF"
処理: 255 mod 16 = 15 → 'F'
      255 / 16 = 15 mod 16 = 15 → 'F'
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | 入力配列/文字列がNull | 有効な値を指定 |
| - | ArgumentOutOfRangeException | オフセット/長さが範囲外 | 有効な範囲を指定 |
| - | FormatException | Base64文字列の長さが4の倍数でない | 有効なBase64文字列を指定 |
| - | FormatException | 無効なBase64文字を含む | 有効なBase64文字のみを使用 |
| - | ArgumentException | サポート外の基数 | 2、8、10、16のいずれかを指定 |
| - | InvalidCastException | 型変換不可能な値 | 変換可能な値を指定 |

### リトライ仕様

データ変換処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- Base64エンコード: 1MB程度のデータを数十ミリ秒以内
- 基数変換: 数マイクロ秒以内

## セキュリティ考慮事項

- Base64はエンコーディングであり暗号化ではない
- センシティブなデータのBase64表現をログ出力しないよう注意

## 備考

- Base64FormattingOptions.InsertLineBreaksは76文字毎に改行を挿入（RFC 2045準拠）
- 空のバイト配列のBase64エンコード結果は空文字列

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Convertクラスはステートレスな静的クラスである。Base64変換用のルックアップテーブルが重要なデータ構造となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Convert.cls | `Source/CorLib/System/Convert.cls` | クラス構造、定数定義、Base64テーブル |

**主要な定数**:
- **58行目**: `Base64Chars` - エンコード用の64文字テーブル
- **59行目**: `Base64CharToBits` - デコード用の逆引きテーブル

#### Step 2: Base64変換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Convert.cls | `Source/CorLib/System/Convert.cls` | ToBase64String、FromBase64String |

**主要処理フロー**:
1. **200-230行目**: ToBase64String - バイト配列からBase64文字列への変換
2. **240-280行目**: ToBase64StringWithOptions - 改行オプション付き変換
3. **140-180行目**: FromBase64String - Base64文字列からバイト配列への変換

#### Step 3: 基数変換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Convert.cls | `Source/CorLib/System/Convert.cls` | ToString(Value, ToBase)メソッド |

**主要処理フロー**:
- **350-420行目**: ToString - 数値から指定基数の文字列への変換
- 負数の2の補数表現処理が含まれる

#### Step 4: 型変換メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Convert.cls | `Source/CorLib/System/Convert.cls` | ToBoolean、ToByte、ToInt*、ToSingle、ToDouble等 |

### プログラム呼び出し階層図

```
Convert (静的クラス)
    │
    ├─ ToBase64String(InArray, Offset, Length, Options)
    │      ├─ ValidateArrayRange
    │      ├─ Base64Encode (内部ループ)
    │      └─ InsertLineBreaks (オプション)
    │
    ├─ ToBase64CharArray(InArray, OffsetIn, Length, OutArray, OffsetOut)
    │      └─ Base64Encode (内部ループ)
    │
    ├─ FromBase64String(s)
    │      ├─ RemoveWhitespace (内部)
    │      ├─ ValidateBase64Length
    │      └─ Base64Decode (内部ループ)
    │
    ├─ FromBase64CharArray(InArray, Offset, Length)
    │      ├─ ValidateBase64Length
    │      └─ Base64Decode (内部ループ)
    │
    ├─ ToString(Value, ToBase)
    │      ├─ ValidateToBase
    │      └─ ConvertToBase (再帰的変換)
    │
    └─ To*(Value, Provider)  [ToBoolean, ToByte, ToInt16, ToInt32, ToInt64, ToSingle, ToDouble, ToDateTime]
           └─ VBA変換関数またはIConvertible呼び出し
```

### データフロー図

```
[入力]                  [処理]                     [出力]

Byte配列 ──────────────▶ ToBase64String ──────────▶ "SGVsbG8="
[72,101,108,108,111]         │
                             ▼
                    3バイト→4文字変換
                             │
                             ▼
                    パディング追加

"SGVsbG8=" ────────────▶ FromBase64String ─────────▶ Byte配列
                             │
                             ▼
                    4文字→3バイト変換
                             │
                             ▼
                    パディング処理

数値(255) ─────────────▶ ToString(Value, 16) ─────▶ "FF"
                             │
                             ▼
                    基数変換ループ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Convert.cls | `Source/CorLib/System/Convert.cls` | ソース | メインクラス定義 |
| ConvertTests.cls | `Source/Tests/System/ConvertTests.cls` | テスト | 単体テストケース |
| Base64FormattingOptions.cls | `Source/CorLib/System/Base64FormattingOptions.cls` | ソース | Base64フォーマットオプション列挙型 |
| Statics.bas | `Source/CorLib/Statics.bas` | ソース | 静的クラスのグローバル変数定義 |
