# 機能設計書 14-CorMath

## 概要

本ドキュメントは、VBCorLibライブラリにおける拡張数学関数機能「CorMath」クラスの機能設計について記述する。

### 本機能の処理概要

CorMathクラスは、VBAの標準数学関数を補完する拡張数学関数を提供する静的クラスである。.NET FrameworkのSystem.Mathクラスと互換性のあるAPIを持ち、最大/最小値取得、切り上げ/切り捨て、符号判定、数値クランプ、BigMul（大整数乗算）、DivRem（商と剰余の同時計算）などの機能を提供する。

**業務上の目的・背景**：VBAの標準数学関数では提供されていない、あるいは使いにくい機能を補完する。特に、Max/Min関数のオーバーロード、Ceiling/Floor関数、BigMul/DivRemなどは、ビジネスロジックやアルゴリズム実装で頻繁に使用される。

**機能の利用シーン**：
- 数値の範囲制限（クランプ）
- 切り上げ/切り捨て計算（金額計算、日付計算）
- 最大値/最小値の取得（比較処理）
- 大整数演算（暗号化、ハッシュ計算）
- 符号判定（正負判定処理）

**主要な処理内容**：
1. 最大値/最小値取得（Max、Min）
2. 切り上げ/切り捨て（Ceiling、Floor、Truncate）
3. 符号判定（Sign）
4. 数値クランプ（Clamp）
5. 大整数乗算（BigMul）
6. 商と剰余の同時計算（DivRem）
7. シフト演算（ShiftLeft、ShiftRight）

**関連システム・外部連携**：他のVBCorLibクラス（StringBuilder、ArrayList等）の内部で数値計算に使用される。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内の数学関数テスト（CorMathTests）の実行 |

## 機能種別

計算処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Val1, Val2 | Long/Int64/Double/Currency | Yes | Max/Minの比較対象 | 数値型であること |
| Value | Double | Yes | Ceiling/Floor/Truncate/Signの対象値 | 数値型であること |
| Value | Long | Yes | Clampの対象値 | 数値型であること |
| MinValue | Long | Yes | Clampの下限値 | MaxValue以下 |
| MaxValue | Long | Yes | Clampの上限値 | MinValue以上 |
| a, b | Long | Yes | BigMulの乗数 | 任意の整数 |
| a, b | Long | Yes | DivRemの被除数と除数 | bは0以外 |
| Value, Count | Long | Yes | ShiftLeft/ShiftRightの値とシフト量 | 任意の整数 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Max戻り値 | Long/Int64/Double/Currency | 2つの値の大きい方 |
| Min戻り値 | Long/Int64/Double/Currency | 2つの値の小さい方 |
| Ceiling戻り値 | Double | 正の無限大方向への切り上げ |
| Floor戻り値 | Double | 負の無限大方向への切り捨て |
| Truncate戻り値 | Double | ゼロ方向への切り捨て |
| Sign戻り値 | Long | -1（負）、0（ゼロ）、1（正） |
| Clamp戻り値 | Long | 範囲内に制限された値 |
| BigMul戻り値 | Int64 | 64ビットの乗算結果 |
| DivRem戻り値 | Long | 商（剰余はByRef引数で返却） |
| ShiftLeft/ShiftRight戻り値 | Long | シフトされた値 |

### 出力先

呼び出し元への戻り値として返却、またはByRef引数（DivRemのRemainder）への書き込み

## 処理フロー

### 処理シーケンス

```
1. Max呼び出し
   └─ 2つの値を比較
   └─ 大きい方を返却

2. Ceiling呼び出し
   └─ Int関数で整数部取得
   └─ 小数部があれば+1
   └─ 結果を返却

3. Clamp呼び出し
   └─ Value < MinValue なら MinValue を返却
   └─ Value > MaxValue なら MaxValue を返却
   └─ それ以外は Value を返却

4. BigMul呼び出し
   └─ 2つのLong値を乗算
   └─ 64ビットの結果をInt64として返却

5. DivRem呼び出し
   └─ 商を計算（a \ b）
   └─ 剰余を計算（a Mod b）
   └─ 商を返却、剰余はByRef引数にセット
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|Max| C[Val1 > Val2?]
    C -->|Yes| D[Val1を返却]
    C -->|No| E[Val2を返却]
    B -->|Ceiling| F[Int(Value)取得]
    F --> G{小数部あり?}
    G -->|Yes| H[Int(Value)+1を返却]
    G -->|No| I[Int(Value)を返却]
    B -->|Clamp| J{Value < MinValue?}
    J -->|Yes| K[MinValueを返却]
    J -->|No| L{Value > MaxValue?}
    L -->|Yes| M[MaxValueを返却]
    L -->|No| N[Valueを返却]
    B -->|BigMul| O[a * b を64ビットで計算]
    O --> P[Int64として返却]
    B -->|DivRem| Q[商 = a \ b]
    Q --> R[剰余 = a Mod b]
    R --> S[商を返却、剰余はByRef]
    D --> T[終了]
    E --> T
    H --> T
    I --> T
    K --> T
    M --> T
    N --> T
    P --> T
    S --> T
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Ceiling動作 | 正の無限大方向へ丸める（-1.5 → -1） | Ceiling呼び出し時 |
| BR-002 | Floor動作 | 負の無限大方向へ丸める（-1.5 → -2） | Floor呼び出し時 |
| BR-003 | Truncate動作 | ゼロ方向へ丸める（-1.5 → -1、1.5 → 1） | Truncate呼び出し時 |
| BR-004 | Sign戻り値 | 負数は-1、ゼロは0、正数は1を返す | Sign呼び出し時 |
| BR-005 | Clamp範囲 | MinValue ≦ 戻り値 ≦ MaxValue を保証 | Clamp呼び出し時 |
| BR-006 | Clamp引数制約 | MinValue > MaxValueの場合ArgumentException | Clamp呼び出し時 |
| BR-007 | DivRem除数制約 | 除数が0の場合、ゼロ除算エラー | DivRem呼び出し時 |

### 計算ロジック

**Ceiling**:
```
If Value = Int(Value) Then
    Ceiling = Value
Else
    Ceiling = Int(Value) + 1
End If
```

**BigMul**:
```
' 32ビット x 32ビット → 64ビット
Result.LowPart = (a And &HFFFF&) * (b And &HFFFF&)
Result.HighPart = (a \ &H10000) * (b \ &H10000)
' 中間項の加算処理を含む
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentException | ClampのMinValue > MaxValue | 正しい範囲を指定 |
| - | OverflowException | 演算結果がオーバーフロー | 適切な型を使用 |
| - | DivisionByZeroError | DivRemの除数が0 | 0以外の除数を指定 |

### リトライ仕様

計算処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 全メソッド: 数マイクロ秒以内
- インライン可能な単純計算で実装

## セキュリティ考慮事項

- 特になし

## 備考

- CorMathの名称は"Cor"(Core)+"Math"に由来
- VBAのSgn関数と異なり、Signは明示的にLong型を返す

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

CorMathクラスはステートレスな静的クラスである。内部状態を持たず、すべてのメソッドが純粋関数として実装されている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CorMath.cls | `Source/CorLib/System/CorMath.cls` | クラス構造、静的クラスとしての設計 |

#### Step 2: Max/Min関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CorMath.cls | `Source/CorLib/System/CorMath.cls` | Max/Minメソッドのオーバーロード |

**主要処理フロー**:
- **77-100行目**: Max関数群（Long、Int64、Double、Currency版）
- **110-133行目**: Min関数群（Long、Int64、Double、Currency版）

#### Step 3: 丸め関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CorMath.cls | `Source/CorLib/System/CorMath.cls` | Ceiling、Floor、Truncate |

**主要処理フロー**:
- **50-60行目**: Ceilingメソッド - 正の無限大方向への丸め
- **65-70行目**: Floorメソッド - 負の無限大方向への丸め
- **145-150行目**: Truncateメソッド - ゼロ方向への丸め

#### Step 4: その他のユーティリティ関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | CorMath.cls | `Source/CorLib/System/CorMath.cls` | Sign、Clamp、BigMul、DivRem、Shift関数 |

**主要処理フロー**:
- **155-165行目**: Signメソッド - 符号判定
- **170-185行目**: Clampメソッド - 範囲制限
- **190-210行目**: BigMulメソッド - 64ビット乗算
- **215-225行目**: DivRemメソッド - 商と剰余の同時計算
- **230-240行目**: ShiftLeft/ShiftRightメソッド - ビットシフト

### プログラム呼び出し階層図

```
CorMath (静的クラス)
    │
    ├─ Max(Val1, Val2)  [Long/Int64/Double/Currency版]
    │      └─ 単純比較
    │
    ├─ Min(Val1, Val2)  [Long/Int64/Double/Currency版]
    │      └─ 単純比較
    │
    ├─ Ceiling(Value)
    │      └─ Int関数 + 条件分岐
    │
    ├─ Floor(Value)
    │      └─ Int関数
    │
    ├─ Truncate(Value)
    │      └─ Fix関数
    │
    ├─ Sign(Value)
    │      └─ 符号判定
    │
    ├─ Clamp(Value, MinValue, MaxValue)
    │      └─ Min/Max組み合わせ
    │
    ├─ BigMul(a, b)
    │      └─ 64ビット演算
    │
    ├─ DivRem(a, b, Remainder)
    │      └─ 整数除算 + 剰余計算
    │
    └─ ShiftLeft/ShiftRight(Value, Count)
           └─ ビット演算
```

### データフロー図

```
[入力]               [処理]                    [出力]

Val1=10, Val2=20 ───▶ Max ───────────────────▶ 20

Value=-1.5 ─────────▶ Ceiling ────────────────▶ -1

Value=-1.5 ─────────▶ Floor ──────────────────▶ -2

Value=25, Min=0,    ▶ Clamp ──────────────────▶ 10
Max=10

a=100000, b=100000 ─▶ BigMul ─────────────────▶ 10000000000 (Int64)

a=17, b=5 ──────────▶ DivRem ─────────────────▶ 商=3, 剰余=2
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CorMath.cls | `Source/CorLib/System/CorMath.cls` | ソース | メインクラス定義 |
| CorMathTests.cls | `Source/Tests/System/CorMathTests.cls` | テスト | 単体テストケース |
| Statics.bas | `Source/CorLib/Statics.bas` | ソース | 静的クラスのグローバル変数定義 |
| Int64.cls | `Source/CorLib/System/Int64.cls` | ソース | 64ビット整数型（BigMulで使用） |
