# 機能設計書 16-Char

## 概要

本ドキュメントは、VBCorLibライブラリにおける文字の分類と操作機能「Char」クラスの機能設計について記述する。

### 本機能の処理概要

Charクラスは、Unicode文字の分類と操作機能を提供する静的クラスである。.NET FrameworkのSystem.Charクラスと互換性のあるAPIを持ち、文字が数字か、英字か、空白か、制御文字か、大文字か小文字かなどを判定する機能、および大文字・小文字変換機能を提供する。

**業務上の目的・背景**：文字列処理において、文字の種類を判定することは頻繁に必要となる。バリデーション、パース処理、フォーマット処理などで使用される。VBAには標準でこのような包括的な文字分類機能がないため、Charクラスがこのギャップを埋める。

**機能の利用シーン**：
- 入力値のバリデーション（数字のみ、英字のみ等）
- 文字列のパース処理（トークナイザー）
- 大文字・小文字変換
- 空白のトリム処理
- Unicode文字カテゴリ判定

**主要な処理内容**：
1. 文字分類（IsDigit、IsLetter、IsLetterOrDigit、IsWhiteSpace、IsControl、IsPunctuation、IsUpper、IsLower等）
2. 大文字・小文字変換（ToUpper、ToLower）
3. 文字カテゴリ取得（GetUnicodeCategory）
4. 数値変換（GetNumericValue）

**関連システム・外部連携**：StringBuilderやCorString等の文字列処理クラスと組み合わせて使用される。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内のCharテスト（CharTests）の実行 |

## 機能種別

文字処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| c | Integer | Yes | Unicode文字コード（AscW値） | 0-65535の範囲 |
| s | String | Yes | 文字列（1文字または複数文字） | Nullでないこと |
| Index | Long | No | 文字列中の位置（0ベース） | 文字列範囲内 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Is*戻り値 | Boolean | 分類判定結果 |
| ToUpper戻り値 | Integer | 大文字に変換されたコード |
| ToLower戻り値 | Integer | 小文字に変換されたコード |
| GetUnicodeCategory戻り値 | UnicodeCategory | Unicode文字カテゴリ |
| GetNumericValue戻り値 | Double | 数値（数字以外は-1） |

### 出力先

呼び出し元への戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. IsDigit呼び出し
   └─ 文字コードを取得
   └─ 0-9（U+0030-U+0039）の範囲チェック
   └─ 結果を返却

2. IsLetter呼び出し
   └─ 文字コードを取得
   └─ a-z、A-Z、または他のUnicode英字範囲チェック
   └─ 結果を返却

3. IsWhiteSpace呼び出し
   └─ 文字コードを取得
   └─ 空白文字一覧との照合
   └─ 結果を返却

4. ToUpper呼び出し
   └─ 文字コードを取得
   └─ 小文字の場合は大文字に変換
   └─ 変換後のコードを返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|IsDigit| C{c >= '0' && c <= '9'?}
    C -->|Yes| D[True返却]
    C -->|No| E[False返却]
    B -->|IsLetter| F{英字範囲内?}
    F -->|Yes| D
    F -->|No| G{他のUnicode英字?}
    G -->|Yes| D
    G -->|No| E
    B -->|IsWhiteSpace| H{空白文字一覧に含まれる?}
    H -->|Yes| D
    H -->|No| E
    B -->|ToUpper| I{小文字?}
    I -->|Yes| J[大文字に変換]
    I -->|No| K[そのまま返却]
    J --> L[変換結果返却]
    K --> L
    D --> M[終了]
    E --> M
    L --> M
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Unicode準拠 | 文字分類はUnicode標準に準拠 | 全分類メソッド |
| BR-002 | ASCII互換 | ASCII範囲の文字は従来通りの判定 | 全分類メソッド |
| BR-003 | 空白文字定義 | スペース、タブ、改行等を空白として判定 | IsWhiteSpace |
| BR-004 | 数字範囲 | 0-9（U+0030-U+0039）を数字として判定 | IsDigit |
| BR-005 | 大文字小文字変換 | 変換不可能な文字はそのまま返却 | ToUpper、ToLower |

### 計算ロジック

**IsDigit**:
```
IsDigit = (c >= &H30 And c <= &H39)  ' '0'-'9'
```

**IsLetter**:
```
IsLetter = (c >= &H41 And c <= &H5A) Or _  ' 'A'-'Z'
           (c >= &H61 And c <= &H7A) Or _  ' 'a'-'z'
           ' + 他のUnicode英字範囲
```

**IsWhiteSpace**:
```
空白文字: U+0009 (TAB), U+000A (LF), U+000B (VT), U+000C (FF),
          U+000D (CR), U+0020 (SPACE), U+0085 (NEL),
          U+00A0 (NBSP), U+2000-U+200A, etc.
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | 文字列がNull | 有効な文字列を指定 |
| - | ArgumentOutOfRangeException | インデックスが範囲外 | 有効なインデックスを指定 |

### リトライ仕様

文字判定処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 全メソッド: O(1)の定数時間
- 単純な比較演算で実装

## セキュリティ考慮事項

- Unicode正規化攻撃に注意（異なる表現で同じ文字）
- 国際化対応時はロケール依存の大文字小文字変換に注意

## 備考

- VBAのAscW関数でUnicodeコードポイントを取得可能
- サロゲートペア（U+10000以上の文字）は個別の16ビット値として処理される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Charクラスはステートレスな静的クラスである。内部で空白文字や句読点の判定テーブルを使用することがある。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Char.cls | `Source/CorLib/System/Char.cls` | クラス構造、定数定義 |

**読解のコツ**: Unicode文字コードの範囲を理解しておくと読みやすい（ASCII: 0-127、Latin-1: 128-255、BMP: 0-65535）。

#### Step 2: 文字分類メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Char.cls | `Source/CorLib/System/Char.cls` | Is*メソッド群 |

**主要処理フロー**:
1. **100-120行目**: IsDigitメソッド - 数字判定
2. **130-160行目**: IsLetterメソッド - 英字判定
3. **170-200行目**: IsWhiteSpaceメソッド - 空白判定
4. **210-230行目**: IsUpperメソッド - 大文字判定
5. **240-260行目**: IsLowerメソッド - 小文字判定

#### Step 3: 大文字小文字変換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Char.cls | `Source/CorLib/System/Char.cls` | ToUpper、ToLowerメソッド |

**主要処理フロー**:
- **280-310行目**: ToUpperメソッド - 大文字変換
- **320-350行目**: ToLowerメソッド - 小文字変換

#### Step 4: その他のユーティリティを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Char.cls | `Source/CorLib/System/Char.cls` | GetUnicodeCategory、GetNumericValue |

### プログラム呼び出し階層図

```
Char (静的クラス)
    │
    ├─ IsDigit(c) / IsDigit(s, Index)
    │      └─ 範囲チェック (&H30-&H39)
    │
    ├─ IsLetter(c) / IsLetter(s, Index)
    │      └─ 複数範囲チェック (A-Z, a-z, Unicode Letter)
    │
    ├─ IsLetterOrDigit(c) / IsLetterOrDigit(s, Index)
    │      ├─ IsLetter
    │      └─ IsDigit
    │
    ├─ IsWhiteSpace(c) / IsWhiteSpace(s, Index)
    │      └─ 空白文字テーブル照合
    │
    ├─ IsControl(c) / IsControl(s, Index)
    │      └─ 制御文字範囲チェック (&H00-&H1F, &H7F-&H9F)
    │
    ├─ IsPunctuation(c) / IsPunctuation(s, Index)
    │      └─ 句読点範囲チェック
    │
    ├─ IsUpper(c) / IsUpper(s, Index)
    │      └─ 大文字範囲チェック
    │
    ├─ IsLower(c) / IsLower(s, Index)
    │      └─ 小文字範囲チェック
    │
    ├─ ToUpper(c)
    │      └─ 小文字→大文字変換
    │
    ├─ ToLower(c)
    │      └─ 大文字→小文字変換
    │
    ├─ GetUnicodeCategory(c) / GetUnicodeCategory(s, Index)
    │      └─ Unicodeカテゴリテーブル参照
    │
    └─ GetNumericValue(c) / GetNumericValue(s, Index)
           └─ 数字→数値変換
```

### データフロー図

```
[入力]               [処理]                    [出力]

文字コード(Integer) ─▶ IsDigit ─────────────────▶ Boolean
    65 ('A')             │
                         ▼
                 範囲チェック(&H30-&H39)
                         │
                         ▼
                     False

文字コード(Integer) ─▶ ToUpper ─────────────────▶ Integer
    97 ('a')             │
                         ▼
                 小文字判定→大文字変換
                         │
                         ▼
                     65 ('A')

文字列 + Index ──────▶ IsWhiteSpace ────────────▶ Boolean
"Hello World", 5         │
                         ▼
                 空白テーブル照合
                         │
                         ▼
                     True
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Char.cls | `Source/CorLib/System/Char.cls` | ソース | メインクラス定義 |
| CharTests.cls | `Source/Tests/System/CharTests.cls` | テスト | 単体テストケース |
| UnicodeCategory.cls | `Source/CorLib/System.Globalization/UnicodeCategory.cls` | ソース | Unicodeカテゴリ列挙型 |
| Statics.bas | `Source/CorLib/Statics.bas` | ソース | 静的クラスのグローバル変数定義 |
