# 機能設計書 17-StringBuilder

## 概要

本ドキュメントは、VBCorLibライブラリにおける可変長文字列ビルダー機能「StringBuilder」クラスの機能設計について記述する。

### 本機能の処理概要

StringBuilderクラスは、文字列の効率的な連結・操作機能を提供するクラスである。.NET FrameworkのSystem.Text.StringBuilderクラスと互換性のあるAPIを持ち、VBAの標準文字列連結（&演算子）の性能問題を解決する。内部バッファを使用して文字列を管理し、メモリ再割り当てを最小化することで高速な文字列操作を実現する。

**業務上の目的・背景**：VBAの標準文字列連結は、連結のたびに新しい文字列が作成されるため、大量の連結処理では性能が著しく低下する。StringBuilderは内部バッファを使用することで、この問題を解決し、効率的な文字列構築を可能にする。

**機能の利用シーン**：
- ログ出力文字列の構築
- SQL文の動的生成
- HTMLやXMLの動的生成
- 大量データのCSV形式変換
- フォーマット済み文字列の構築

**主要な処理内容**：
1. 文字列追加（Append、AppendLine、AppendFormat）
2. 文字列挿入（Insert、InsertChars）
3. 文字列削除（Remove）
4. 文字列置換（Replace）
5. 文字列取得（ToString）
6. 容量管理（Capacity、Length、EnsureCapacity）

**関連システム・外部連携**：IFormatProvider/ICustomFormatter実装と連携してカスタムフォーマットを実現する。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 主画面 | テストスイート内のStringBuilderテスト（StringBuilderTests）の実行 |

## 機能種別

文字列処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Value | Variant | Yes | Appendの追加文字列/値 | 文字列変換可能な値 |
| StartIndex | Long | No | Appendの開始位置 | 0以上 |
| Count | Long | No | Appendの文字数 | 0以上 |
| Char | Long | Yes | AppendCharの文字コード | 0-65535 |
| Format | String | Yes | AppendFormatのフォーマット文字列 | 有効なフォーマット |
| Args | Variant() | Yes | AppendFormatの引数配列 | フォーマットに対応 |
| Index | Long | Yes | Insertの挿入位置 | 0以上Length以下 |
| OldValue | String | Yes | Replaceの検索文字列 | 空でないこと |
| NewValue | String | Yes | Replaceの置換文字列 | 任意 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ToString戻り値 | String | 構築された文字列 |
| Append戻り値 | StringBuilder | メソッドチェーン用の自身への参照 |
| Length | Long | 現在の文字列長 |
| Capacity | Long | 現在のバッファ容量 |
| Chars(Index) | Integer | 指定位置の文字コード |

### 出力先

呼び出し元への戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. インスタンス生成
   └─ 初期容量（デフォルト128）でバッファ確保
   └─ 初期文字列があればコピー

2. Append呼び出し
   └─ 必要に応じてEnsureCapacityでバッファ拡張
   └─ 内部バッファに文字列追加
   └─ Length更新
   └─ 自身への参照を返却（メソッドチェーン用）

3. AppendFormat呼び出し
   └─ フォーマット文字列解析
   └─ 各プレースホルダーを引数で置換
   └─ 結果をバッファに追加

4. Insert呼び出し
   └─ 挿入位置以降の文字を後方移動
   └─ 挿入位置に新しい文字列をコピー
   └─ Length更新

5. ToString呼び出し
   └─ 内部バッファから文字列を生成
   └─ 文字列を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{メソッド種別}
    B -->|Append| C{バッファ容量足りる?}
    C -->|No| D[EnsureCapacity]
    C -->|Yes| E[バッファに追加]
    D --> E
    E --> F[Length更新]
    F --> G[自身への参照を返却]
    B -->|Insert| H{Index有効?}
    H -->|No| I[ArgumentOutOfRangeException]
    H -->|Yes| J[後方に空間作成]
    J --> K[挿入位置にコピー]
    K --> F
    B -->|Remove| L{範囲有効?}
    L -->|No| M[ArgumentOutOfRangeException]
    L -->|Yes| N[後続文字を前方移動]
    N --> F
    B -->|Replace| O[マッチ箇所を検索]
    O --> P{マッチあり?}
    P -->|No| G
    P -->|Yes| Q[置換処理]
    Q --> G
    B -->|ToString| R[バッファから文字列生成]
    R --> S[文字列を返却]
    G --> T[終了]
    S --> T
    I --> T
    M --> T
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | デフォルト容量 | 初期容量は128文字 | 新規インスタンス生成時 |
| BR-002 | 容量拡張戦略 | 容量不足時は現在の2倍に拡張 | EnsureCapacity時 |
| BR-003 | メソッドチェーン | Append系メソッドは自身への参照を返却 | Append/Insert/Remove/Replace |
| BR-004 | 0ベースインデックス | Charsプロパティ、Insert/Remove等は0ベースインデックス | インデックス指定時 |
| BR-005 | FillCharacter | 長さ拡張時はFillCharacter（デフォルト空白）で埋める | Length設定時 |

### 計算ロジック

**EnsureCapacity**:
```
If RequiredCapacity > mCapacity Then
    NewCapacity = mCapacity * 2
    If NewCapacity < RequiredCapacity Then
        NewCapacity = RequiredCapacity
    End If
    Capacity = NewCapacity
End If
```

**AppendFormat**:
```
フォーマット: "{index[,length][:formatString]}"
例: "{0,-10:N2}" → 引数0を、幅10で左寄せ、数値2桁でフォーマット
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentOutOfRangeException | Indexが負または Length超過 | 有効な範囲を指定 |
| - | ArgumentOutOfRangeException | Countが負 | 0以上の値を指定 |
| - | ArgumentNullException | Charsが未初期化配列 | 初期化済み配列を使用 |
| - | FormatException | フォーマット文字列が無効 | 正しいフォーマットを指定 |
| - | OverflowException | AppendCharのCharが範囲外 | 0-65535の値を指定 |

### リトライ仕様

文字列操作処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- Append: O(1)の償却定数時間（バッファ拡張時を除く）
- Insert: O(n)（n = Length - Index、後方移動のため）
- Remove: O(n)（n = Length - StartIndex - Length、前方移動のため）
- ToString: O(n)（n = Length、文字列生成のため）

## セキュリティ考慮事項

- SQL文生成時はSQLインジェクションに注意（パラメータ化クエリを推奨）
- HTML生成時はXSSに注意（エスケープ処理を推奨）

## 備考

- CopyMemory APIを使用した高速なメモリ操作を実装
- SysAllocStringLen APIを使用した効率的な文字列生成

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

StringBuilderクラスは内部にInteger配列（Unicode文字配列）を持ち、これがバッファとして機能する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | クラス構造、メンバ変数 |

**主要なメンバ変数**:
- **55行目**: `mString() As Integer` - 内部バッファ（文字配列）
- **56行目**: `mStringPtr As Long` - バッファ先頭ポインタ
- **57行目**: `mCapacity As Long` - 現在の容量
- **58行目**: `mLength As Long` - 現在の文字列長
- **59行目**: `mFillChar As Integer` - 埋め文字（デフォルト空白）

#### Step 2: Append系メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | Append、AppendChar、AppendString、AppendLine |

**主要処理フロー**:
1. **71-82行目**: Appendメソッド - 値の型に応じた追加処理
2. **92-118行目**: AppendCharメソッド - 文字の繰り返し追加
3. **197-211行目**: AppendStringメソッド - 高速文字列追加
4. **183-190行目**: AppendLineメソッド - 改行付き追加

#### Step 3: AppendFormat系を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | AppendFormat、AppendFormatArrayEx |

**主要処理フロー**:
- **129-133行目**: AppendFormatメソッド - ParamArrayを使用
- **652-775行目**: AppendFormatArrayExメソッド - フォーマット文字列のパース処理

#### Step 4: 挿入・削除・置換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | Insert、InsertChars、Remove、Replace |

**主要処理フロー**:
- **400-423行目**: Insertメソッド - 文字列挿入
- **433-455行目**: InsertCharsメソッド - 文字配列挿入
- **493-507行目**: Removeメソッド - 範囲削除
- **518-554行目**: Replaceメソッド - 文字列置換

#### Step 5: ヘルパーメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | EnsureCapacity、InsertSpace、FindMatches |

**主要処理フロー**:
- **310-324行目**: EnsureCapacityメソッド - 容量確保
- **925-937行目**: InsertSpaceメソッド - 挿入用空間作成
- **1074-1131行目**: FindMatchesメソッド - 置換用マッチ検索

### プログラム呼び出し階層図

```
StringBuilder (インスタンス)
    │
    ├─ Append(Value, StartIndex, Count)
    │      ├─ AppendCharsValue (配列の場合)
    │      ├─ AppendStringValue (文字列の場合)
    │      └─ AppendString → Object.ToString (その他)
    │
    ├─ AppendChar(Char, Count)
    │      ├─ EnsureCapacity
    │      └─ Fill (繰り返し設定)
    │
    ├─ AppendFormat(Format, Args...)
    │      └─ AppendFormatArrayEx
    │             ├─ ParseArgumentFormatting
    │             ├─ Object.ToString (値のフォーマット)
    │             └─ AppendString
    │
    ├─ AppendLine(s)
    │      ├─ AppendString(s)
    │      └─ AppendString(vbCrLf)
    │
    ├─ Insert(Index, Value, Count)
    │      ├─ InsertSpace
    │      └─ CopyMemory
    │
    ├─ Remove(StartIndex, Length)
    │      └─ CopyMemory (後続を前方移動)
    │
    ├─ Replace(OldValue, NewValue, StartIndex, Count)
    │      ├─ FindMatches
    │      └─ ReplaceEqualLength / ReplaceSmallerLength / ReplaceLargerLength
    │
    └─ ToString(StartIndex, Length)
           └─ SysAllocStringLen
```

### データフロー図

```
[入力]                  [処理]                    [出力]

初期文字列 ─────────────▶ New StringBuilder ──────▶ 内部バッファ初期化

追加文字列 ─────────────▶ Append ─────────────────▶ バッファ拡張＋追加
                             │
                             ▼
                    EnsureCapacity (必要時)
                             │
                             ▼
                        CopyMemory

フォーマット＋引数 ────▶ AppendFormat ────────────▶ フォーマット済み文字列追加
                             │
                             ▼
                    ParseArgumentFormatting
                             │
                             ▼
                        Object.ToString

（なし） ───────────────▶ ToString ────────────────▶ String
                             │
                             ▼
                        SysAllocStringLen
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | ソース | メインクラス定義 |
| StringBuilderTests.cls | `Source/Tests/System.Text/StringBuilderTests.cls` | テスト | 単体テストケース |
| Constructors.cls | `Source/CorLib/Constructors.cls` | ソース | NewStringBuilderコンストラクタ |
| IObject.cls | `Source/CorLib/System/IObject.cls` | ソース | IObjectインターフェース |
| IFormatProvider.cls | `Source/CorLib/System/IFormatProvider.cls` | ソース | フォーマットプロバイダーIF |
| ICustomFormatter.cls | `Source/CorLib/System/ICustomFormatter.cls` | ソース | カスタムフォーマッターIF |
