# 機能設計書: BitArray

## 1. 機能概要

### 1.1 機能名
BitArray（ビット配列）

### 1.2 機能ID
No.23

### 1.3 機能分類
System.Collections - コレクション

### 1.4 概要説明
BitArrayは、ビット値のコンパクトな配列を管理するクラスである。各ビットはブール値を表し、True = 1、False = 0 となる。.NET FrameworkのBitArrayクラスをVB6で実装したもので、ビット単位の論理演算（AND、OR、XOR、NOT）を効率的に実行できる。

### 1.5 主な用途
- ブール値の大量格納（メモリ効率の良い格納）
- ビット単位の論理演算（フラグ操作）
- ビットマスクの管理
- 権限やフラグの集合管理

---

## 2. 機能詳細

### 2.1 提供機能一覧

| No | 機能名 | 説明 |
|----|--------|------|
| 1 | Item | 指定インデックスのビット値を取得/設定 |
| 2 | Length | ビット配列の長さを取得/設定 |
| 3 | Count | ビット数を取得（Lengthと同値、読み取り専用） |
| 4 | AndBits | 他のBitArrayとのビット単位AND演算 |
| 5 | OrBits | 他のBitArrayとのビット単位OR演算 |
| 6 | XorBits | 他のBitArrayとのビット単位XOR演算 |
| 7 | NotBits | ビット単位NOT演算（反転） |
| 8 | SetAll | すべてのビットを指定値に設定 |
| 9 | Clone | 浅いコピーを作成 |
| 10 | CopyTo | 要素を配列にコピー（Long/Byte/Boolean対応） |
| 11 | GetEnumerator | 列挙子を取得 |

### 2.2 処理フロー

#### 2.2.1 ビット取得フロー（Item Get）
```
1. EnsureInited（配列初期化確認）
   ↓
2. インデックス範囲チェック
   ↓
3. GetBitIndex（Long配列インデックスとビット位置を計算）
   ├── Index \ 32 = Long配列インデックス
   └── Index Mod 32 = ビット位置
   ↓
4. ビットマスクでAND演算
   ↓
5. 結果をBooleanで返却
```

#### 2.2.2 ビット設定フロー（Item Let）
```
1. EnsureInited（配列初期化確認）
   ↓
2. インデックス範囲チェック
   ↓
3. GetBitIndex（Long配列インデックスとビット位置を計算）
   ↓
4. Value = True の場合
   │   mBits(Index) = mBits(Index) Or Bit
   ↓
   Value = False の場合
   │   mBits(Index) = mBits(Index) And (Not Bit)
   ↓
5. Version更新
```

### 2.3 データ構造

#### 2.3.1 クラスメンバ変数
```vb
Private mBits()     As Long     ' ビット格納用Long配列（32ビット/要素）
Private mLength     As Long     ' 表現するビット数
Private mUBound     As Long     ' mBits配列の上限インデックス
Private mVersion    As Long     ' バージョン（変更検出用）
```

#### 2.3.2 定数定義
```vb
Private Const BitsPerLong As Long = 32    ' Long型あたりのビット数
Private Const BitsPerByte As Long = 8     ' Byte型あたりのビット数
```

### 2.4 アルゴリズム詳細

#### 2.4.1 ビットインデックス計算
```vb
Private Function GetBitIndex(ByVal Index As Long, ByRef Bit As Long) As Long
    ' Long配列のインデックスを計算
    GetBitIndex = Index \ BitsPerLong  ' 32で割った商
    ' ビット位置を2のべき乗で計算
    Bit = Powers(Index - GetBitIndex * BitsPerLong)
End Function
```

#### 2.4.2 配列サイズ計算
- 必要なLong要素数: `(Length + 31) \ 32`
- 必要なByte要素数: `(Length + 7) \ 8`

#### 2.4.3 論理演算
```vb
' AND演算
For i = 0 To mUBound
    Bits(i) = Bits(i) And mBits(i)
Next

' OR演算
For i = 0 To mUBound
    Bits(i) = Bits(i) Or mBits(i)
Next

' NOT演算
For i = 0 To mUBound
    mBits(i) = Not mBits(i)
Next
```

### 2.5 実装インターフェース

| インターフェース | 説明 |
|------------------|------|
| IObject | 基本オブジェクト機能（Equals, GetHashCode, ToString） |
| ICollection | コレクション基本機能 |
| IEnumerable | 列挙機能 |
| ICloneable | 複製機能 |
| IVersionable | バージョン管理機能 |

---

## 3. インターフェース仕様

### 3.1 パブリックAPI

#### 3.1.1 Item プロパティ
```vb
Public Property Get Item(ByVal Index As Long) As Boolean
Public Property Let Item(ByVal Index As Long, ByVal Value As Boolean)
```
**パラメータ**:
- Index: 0ベースのビットインデックス

**戻り値**: 指定ビットの値（True/False）

**例外**: ArgumentOutOfRangeException: インデックスが範囲外の場合

#### 3.1.2 Length プロパティ
```vb
Public Property Get Length() As Long
Public Property Let Length(ByVal Value As Long)
```
**パラメータ**:
- Value: 新しいビット数

**備考**: 長さを増やした場合、新しいビットは0（False）に初期化

**例外**: ArgumentOutOfRangeException: 負の値が指定された場合

#### 3.1.3 AndBits メソッド
```vb
Public Function AndBits(ByVal Bits As BitArray) As BitArray
```
**パラメータ**:
- Bits: AND演算を行う相手のBitArray

**戻り値**: 自身への参照（メソッドチェーン用）

**例外**:
- ArgumentNullException: BitsがNothingの場合
- ArgumentException: 長さが異なる場合

#### 3.1.4 OrBits メソッド
```vb
Public Function OrBits(ByVal Bits As BitArray) As BitArray
```
**パラメータ**:
- Bits: OR演算を行う相手のBitArray

**戻り値**: 自身への参照（メソッドチェーン用）

#### 3.1.5 XorBits メソッド
```vb
Public Function XorBits(ByVal Bits As BitArray) As BitArray
```
**パラメータ**:
- Bits: XOR演算を行う相手のBitArray

**戻り値**: 自身への参照（メソッドチェーン用）

#### 3.1.6 NotBits メソッド
```vb
Public Function NotBits() As BitArray
```
**戻り値**: 自身への参照（メソッドチェーン用）

**備考**: すべてのビットを反転

#### 3.1.7 SetAll メソッド
```vb
Public Sub SetAll(ByVal Value As Boolean)
```
**パラメータ**:
- Value: すべてのビットに設定する値

#### 3.1.8 CopyTo メソッド
```vb
Public Sub CopyTo(ByRef Arr As Variant, ByVal Index As Long)
```
**パラメータ**:
- Arr: コピー先配列（Long/Byte/Boolean配列に対応）
- Index: コピー開始インデックス

**備考**: 配列の型に応じてコピー方法が異なる

---

## 4. 内部処理詳細

### 4.1 EnsureInited サブルーチン
配列が初期化されていない場合、デフォルトで32ビットに初期化する。

```vb
Private Sub EnsureInited()
    If SAPtr(mBits) = vbNullPtr Then
        ReDim mBits(0)
        mLength = 32
    End If
End Sub
```
**処理概要**（446-451行目）:
1. mBits配列がNullかチェック
2. Nullの場合、要素数1（32ビット）で初期化

### 4.2 GetBitIndex 関数
ビットインデックスからLong配列インデックスとビットマスクを計算する。

```vb
Private Function GetBitIndex(ByVal Index As Long, ByRef Bit As Long) As Long
```
**処理概要**（483-486行目）:
1. Index \ 32 でLong配列のインデックスを取得
2. Powers配列で2のべき乗を取得してビットマスクを生成

### 4.3 MaskOffLastElement サブルーチン
最後の要素で使用していないビットをクリアする。

**処理概要**（453-467行目）:
1. Lengthが要素サイズの倍数でない場合
2. 最後の要素の未使用ビットをマスクでクリア

### 4.4 CopyTo の型別処理
**Long配列の場合**（115-117行目）:
- CorArray.CopyExで直接コピー
- 最後の要素の未使用ビットをマスク

**Byte配列の場合**（119-124行目）:
- Buffer.BlockCopyでバイト単位コピー
- 最後の要素の未使用ビットをマスク

**Boolean配列の場合**（126-138行目）:
- 各ビットを個別にBoolean値として取り出してコピー

---

## 5. 関連コンポーネント

### 5.1 依存クラス

| クラス名 | 用途 |
|----------|------|
| BitArrayEnumerator | BitArray専用の列挙子 |
| Powers | 2のべき乗テーブル（ビットマスク用） |
| CorArray | 配列操作ユーティリティ |
| Buffer | メモリブロックコピー |

### 5.2 シリアライゼーション
PropertyBagを使用した永続化をサポート（492-516行目）:
- mUBound, mLength, mBits配列のUBoundを保存
- mBits配列の各要素を個別に保存（Bits0, Bits1...）

---

## 6. エラー処理

### 6.1 例外一覧

| 例外 | 発生条件 |
|------|----------|
| ArgumentNullException | AndBits/OrBits/XorBitsでBitsがNothingの場合 |
| ArgumentException | 論理演算で配列長が異なる場合、CopyToで対応していない配列型の場合 |
| ArgumentOutOfRangeException | インデックスが範囲外の場合、Lengthに負の値を設定した場合 |

### 6.2 エラーメッセージ
```vb
' 配列長不一致エラー
Arg_ArrayLengthsDiffer = "配列の長さが一致しません"
' サポートされていない配列型
Arg_BitArrayTypeUnsupported = "BitArrayはこの配列型をサポートしていません"
```

---

## 7. パフォーマンス考慮

### 7.1 計算量

| 操作 | 計算量 |
|------|--------|
| Item取得/設定 | O(1) |
| AndBits/OrBits/XorBits | O(n/32) |
| NotBits | O(n/32) |
| SetAll | O(n/32) |
| Clone | O(n/32) |
| CopyTo (Long/Byte) | O(n/32) |
| CopyTo (Boolean) | O(n) |

※ n = ビット数、n/32 = Long配列要素数

### 7.2 メモリ効率
- Boolean配列: 1ビットあたり2バイト（VB6のBoolean）
- BitArray: 1ビットあたり1/32 Long = 4/32 = 0.125バイト
- **メモリ効率は約16倍**

### 7.3 推奨事項
- 大量のフラグを管理する場合はBoolean配列よりBitArrayを推奨
- 論理演算が多い場合はBitArrayの論理演算メソッドを活用
- メソッドチェーンで連続した論理演算が可能

---

## 8. コードリーディングガイド

### 8.1 推奨読解順序

1. **データ構造の理解**
   - `BitArray.cls` 60-63行目: メンバ変数定義
   - 57-58行目: 定数定義（BitsPerLong, BitsPerByte）

2. **初期化処理**
   - `EnsureInited` (446-451行目): デフォルト初期化
   - `Init` (400-403行目): 長さ指定初期化
   - `InitFromBooleans` (405-417行目): Boolean配列からの初期化
   - `InitFromBytes` (419-426行目): Byte配列からの初期化
   - `InitFromLongs` (428-434行目): Long配列からの初期化

3. **基本操作**
   - `Item Get` (205-216行目): ビット取得
   - `Item Let` (225-242行目): ビット設定
   - `Length Let` (261-274行目): 長さ変更

4. **論理演算**
   - `AndBits` (74-84行目): AND演算
   - `OrBits` (312-323行目): OR演算
   - `XorBits` (360-370行目): XOR演算
   - `NotBits` (292-303行目): NOT演算

5. **内部アルゴリズム**
   - `GetBitIndex` (483-486行目): ビットインデックス計算
   - `MaskOffLastElement` (453-467行目): 最終要素マスク処理

### 8.2 プログラム呼び出し階層図

```
BitArray.Item(Get)
  ├── EnsureInited
  └── GetBitIndex
        └── Powers (2のべき乗テーブル参照)

BitArray.Item(Let)
  ├── EnsureInited
  ├── GetBitIndex
  └── ビット演算 (Or/And Not)

BitArray.AndBits
  ├── EnsureInited
  └── ApplyAndTo (内部配列への適用)

BitArray.CopyTo
  ├── ValidateArray
  ├── EnsureInited
  ├── CorArray.CopyEx (Long配列)
  ├── Buffer.BlockCopy (Byte配列)
  └── 個別コピー (Boolean配列)

BitArray.Clone
  └── InitClone
```

### 8.3 データフロー図

```
[ビットインデックス] ──→ [GetBitIndex]
                              │
                    ┌─────────┴─────────┐
                    ↓                   ↓
            [Long配列Index]        [ビットマスク]
                    │                   │
                    └─────────┬─────────┘
                              ↓
                      [mBits(Index)]
                              │
                    ┌─────────┴─────────┐
                    ↓                   ↓
             [And演算(Get)]      [Or/And演算(Set)]
                    │                   │
                    ↓                   ↓
              [Boolean値]          [ビット更新]
```

### 8.4 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| Source/CorLib/System.Collections/BitArray.cls | クラス | メインクラス |
| Source/CorLib/System.Collections/BitArrayEnumerator.cls | クラス | 列挙子 |
| Source/CorLib/System.Collections/BitArrayStatic.cls | クラス | 静的ファクトリ |
| Source/CorLib/System/CorArray.cls | モジュール | 配列操作 |
| Source/CorLib/System/Buffer.cls | クラス | メモリブロック操作 |
| Source/CorLib/System/Powers.bas | モジュール | 2のべき乗テーブル |

### 8.5 読解のコツ
- ビット操作は32ビット単位（Long型）で行われることを意識
- `Index \ 32` はビットインデックスからLong配列インデックスへの変換
- `Powers` は2のべき乗テーブルで、ビットマスク生成に使用
- `SAPtr` はSafeArrayのポインタ取得（配列が初期化されているか確認）
- メソッドチェーン用に自身（Me）を返すパターンに注目

---

## 9. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|------------|----------|
| 2012 | 1.0 | 初期実装（Kelly Ethridge） |

---

## 10. 備考

- .NET FrameworkのSystem.Collections.BitArrayと互換性のあるAPI設計
- デフォルトサイズは32ビット（未初期化時にEnsureInitedで設定）
- Long型配列で内部管理するため、実際のメモリ使用は32の倍数単位
- CopyToは3種類の配列型（Long/Byte/Boolean）に対応
- 論理演算メソッドは自身を返すため、メソッドチェーンが可能
