# 機能設計書: Comparer

## 1. 機能概要

### 1.1 機能名
Comparer（比較クラス）

### 1.2 機能ID
No.24

### 1.3 機能分類
System.Collections - コレクション

### 1.4 概要説明
Comparerは、2つの値を比較して等価性を判定するクラスである。文字列比較は大文字小文字を区別する（ケースセンシティブ）。.NET FrameworkのComparerクラスをVB6で実装したもので、IComparerインターフェースのデフォルト実装を提供する。

### 1.5 主な用途
- コレクションのソート処理における比較ロジックの提供
- 二分探索での要素比較
- カルチャ対応の文字列比較
- IComparable実装オブジェクトの比較

---

## 2. 機能詳細

### 2.1 提供機能一覧

| No | 機能名 | 説明 |
|----|--------|------|
| 1 | Compare | 2つの値を比較し、大小関係を判定 |
| 2 | Equals | オブジェクトの等価性を判定 |
| 3 | GetHashCode | ハッシュコードを取得 |
| 4 | ToString | 文字列表現を取得 |
| 5 | LCID | ロケールIDを取得（Friend） |

### 2.2 処理フロー

#### 2.2.1 Compare処理フロー
```
1. 引数aとbの型を取得
   ↓
2. aの型別処理
   ├── vbString: 両方文字列ならカルチャ比較
   ├── vbNull: bがNull以外なら-1、同じなら0
   ├── vbEmpty: bがNullなら1、Emptyなら0、他は-1
   ├── vbObject/vbDataObject: IComparable実装確認
   ├── vbUserDefinedType: Int64判定
   └── 同一型: 標準比較（<, >, =）
   ↓
3. aで処理完了しない場合、bの型別処理
   ├── vbNull/vbEmpty: 1を返却
   ├── vbObject/vbDataObject: IComparable実装確認
   └── その他: ArgumentException
   ↓
4. 結果を返却（-1, 0, 1）
```

### 2.3 データ構造

#### 2.3.1 クラスメンバ変数
```vb
Private mCulture As CultureInfo    ' カルチャ情報（文字列比較用）
```

### 2.4 比較ロジック詳細

#### 2.4.1 戻り値の意味

| 戻り値 | 意味 |
|--------|------|
| 負の値 | a < b |
| 0 | a = b |
| 正の値 | a > b |

#### 2.4.2 型別比較処理

| 型 | 比較方法 |
|----|----------|
| String | CorString.CompareCulturalV（カルチャ対応比較） |
| Null | Null同士は0、それ以外はNullが小さい |
| Empty | Empty同士は0、Null < Empty < 他の値 |
| Object | IComparable.CompareToを使用 |
| Int64 (UDT) | Int64Compare関数を使用 |
| 同一型 | VB6標準の比較演算子（<, >, =） |

### 2.5 実装インターフェース

| インターフェース | 説明 |
|------------------|------|
| IObject | 基本オブジェクト機能（Equals, GetHashCode, ToString） |
| IComparer | 比較機能（Compare） |

---

## 3. インターフェース仕様

### 3.1 パブリックAPI

#### 3.1.1 Compare メソッド
```vb
Public Function Compare(ByRef a As Variant, ByRef b As Variant) As Long
```
**パラメータ**:
- a: 比較対象1
- b: 比較対象2

**戻り値**:
- 負の整数: aがbより小さい
- 0: aとbが等しい
- 正の整数: aがbより大きい

**例外**:
- ArgumentException: 比較できない型の場合、IComparableを実装していないオブジェクトの場合

**備考**:
- 文字列比較はカルチャに依存
- IComparable実装オブジェクトはCompareToメソッドを使用

#### 3.1.2 Equals メソッド
```vb
Public Function Equals(ByRef Value As Variant) As Boolean
```
**パラメータ**:
- Value: 比較対象

**戻り値**: 同一インスタンスの場合True

#### 3.1.3 GetHashCode メソッド
```vb
Public Function GetHashCode() As Long
```
**戻り値**: インスタンスを識別する擬似一意の数値

#### 3.1.4 ToString メソッド
```vb
Public Function ToString() As String
```
**戻り値**: オブジェクトの文字列表現

---

## 4. 内部処理詳細

### 4.1 Compare メソッドの詳細処理

**処理概要**（114-188行目）:

```vb
Public Function Compare(ByRef a As Variant, ByRef b As Variant) As Long
    ' 1. 型取得
    AVarType = VarType(a)
    BVarType = VarType(b)

    ' 2. aの型別処理
    Select Case AVarType
        Case vbString
            ' 両方文字列の場合、カルチャ比較
            If BVarType = vbString Then
                Compare = CorString.CompareCulturalV(a, b, mCulture, CompareOptions.None)
            End If

        Case vbNull
            ' Null同士は0、それ以外は-1
            If BVarType <> vbNull Then Compare = -1

        Case vbEmpty
            ' Empty vs Null = 1、Empty vs Empty = 0、他は-1

        Case vbObject, vbDataObject
            ' IComparable実装の場合はCompareToを使用
            If TypeOf a Is IComparable Then
                Compare = Comparable.CompareTo(b)
            End If

        Case vbUserDefinedType
            ' Int64型の比較
            If IsInt64(a) And IsInt64(b) Then
                Compare = Int64Compare(...)
            End If

        Case BVarType
            ' 同一型の標準比較
            If a < b Then Compare = -1
            ElseIf a > b Then Compare = 1
    End Select
End Function
```

### 4.2 Init サブルーチン
コンストラクタとして機能する初期化処理。

```vb
Friend Sub Init(ByVal Culture As CultureInfo)
```
**処理概要**（198-203行目）:
1. Cultureパラメータのnullチェック
2. mCultureにカルチャ情報を設定

---

## 5. 関連コンポーネント

### 5.1 依存クラス

| クラス名 | 用途 |
|----------|------|
| CultureInfo | カルチャ情報（文字列比較のロケール） |
| IComparable | 比較可能なオブジェクトのインターフェース |
| CorString | 文字列比較ユーティリティ |
| ComparerStatic | Comparerの静的ファクトリ |

### 5.2 関連クラス

| クラス名 | 関係 |
|----------|------|
| CaseInsensitiveComparer | 大文字小文字を区別しない比較クラス |
| EqualityComparer | 等価性比較クラス |

---

## 6. エラー処理

### 6.1 例外一覧

| 例外 | 発生条件 |
|------|----------|
| ArgumentNullException | Init時にCultureがNothingの場合 |
| ArgumentException | 比較できない型の場合、IComparableを実装していないオブジェクトの場合 |

### 6.2 エラーメッセージ
```vb
' IComparable未実装エラー
Argument_ImplementIComparable = "少なくとも一方のオブジェクトがIComparableを実装している必要があります"
' 無効な値型エラー
Argument_InvalidValueType = "'{0}'は比較可能な型ではありません"
```

---

## 7. パフォーマンス考慮

### 7.1 計算量

| 操作 | 計算量 |
|------|--------|
| Compare（数値/Boolean） | O(1) |
| Compare（文字列） | O(n)（nは文字列長） |
| Compare（IComparable） | 実装依存 |

### 7.2 推奨事項
- カルチャを意識した文字列比較が必要な場合に使用
- 大文字小文字を区別しない比較はCaseInsensitiveComparerを使用
- パフォーマンスが重要な場合はカスタムIComparer実装を検討

---

## 8. コードリーディングガイド

### 8.1 推奨読解順序

1. **データ構造の理解**
   - `Comparer.cls` 55行目: メンバ変数定義（mCulture）

2. **インターフェース定義**
   - 52-53行目: IObject, IComparerのImplements宣言

3. **初期化処理**
   - `Init` (198-203行目): カルチャ設定

4. **メイン処理**
   - `Compare` (114-188行目): 比較ロジック本体
   - 122-169行目: aの型別処理
   - 171-187行目: bの型別処理

5. **インターフェース実装**
   - `IComparer_Compare` (209-211行目): IComparer実装

### 8.2 プログラム呼び出し階層図

```
Comparer.Compare
  ├── VarType (型判定)
  ├── [vbString] CorString.CompareCulturalV
  │               └── CultureInfo.LCID
  ├── [vbObject] IComparable.CompareTo
  ├── [vbUserDefinedType] Int64Compare
  │                        └── CallFunc_T_T_Long
  └── [同一型] VB標準比較演算子

IComparer.Compare
  └── Comparer.Compare (委譲)
```

### 8.3 データフロー図

```
[引数a, b] ──→ [VarType取得]
                    │
         ┌─────────┼─────────┐
         ↓         ↓         ↓
    [vbString] [vbObject] [同一型]
         │         │         │
         ↓         ↓         ↓
[CompareCulturalV] [CompareTo] [<,>,=]
         │         │         │
         └─────────┼─────────┘
                   ↓
            [結果: -1, 0, 1]
```

### 8.4 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| Source/CorLib/System.Collections/Comparer.cls | クラス | メインクラス |
| Source/CorLib/System.Collections/ComparerStatic.cls | クラス | 静的ファクトリ |
| Source/CorLib/System.Collections/IComparer.cls | インターフェース | 比較契約 |
| Source/CorLib/System.Collections/CaseInsensitiveComparer.cls | クラス | 大文字小文字非区別比較 |
| Source/CorLib/System.Globalization/CultureInfo.cls | クラス | カルチャ情報 |
| Source/CorLib/System/IComparable.cls | インターフェース | 比較可能オブジェクト契約 |

### 8.5 読解のコツ
- Select Case文で型別の比較処理を行うパターンを理解
- IComparableを実装したオブジェクトはCompareToに委譲
- Null/Emptyの順序関係を意識（Null < Empty < 他の値）
- vbUserDefinedTypeの特別処理はInt64型のための対応
- mCultureがNothingの場合はCurrentCultureが使用される（ComparerStatic経由）

---

## 9. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|------------|----------|
| 2014 | 1.0 | 初期実装（Kelly Ethridge） |

---

## 10. 備考

- .NET FrameworkのSystem.Collections.Comparerと互換性のあるAPI設計
- Thread.CurrentCultureを使用したカルチャ対応比較が可能
- 大文字小文字を区別する（CaseSensitive）比較がデフォルト
- CaseInsensitiveComparerとの使い分けが重要
- Int64（Currency型のラッパー）の比較にも対応
