# 機能設計書: EqualityComparer

## 1. 機能概要

### 1.1 機能名
EqualityComparer（等価比較クラス）

### 1.2 機能ID
No.25

### 1.3 機能分類
System.Collections - コレクション

### 1.4 概要説明
EqualityComparerは、2つの値の等価性を判定し、オブジェクトのハッシュコードを取得するクラスである。IEqualityComparerインターフェースのデフォルト実装を提供し、主にHashtableなどのハッシュベースコレクションで使用される。

### 1.5 主な用途
- Hashtableでのキー比較
- ハッシュベースコレクションでのキーのハッシュコード計算
- オブジェクトの等価性判定
- カスタムコレクションでの等価比較ロジックの提供

---

## 2. 機能詳細

### 2.1 提供機能一覧

| No | 機能名 | 説明 |
|----|--------|------|
| 1 | Equals | 2つの値の等価性を判定 |
| 2 | GetHashCode | オブジェクトのハッシュコードを取得 |

### 2.2 処理フロー

#### 2.2.1 Equals処理フロー
```
1. IEqualityComparer_Equalsが呼ばれる
   ↓
2. Object.Equalsに委譲
   ↓
3. 等価性判定結果を返却（Boolean）
```

#### 2.2.2 GetHashCode処理フロー
```
1. IEqualityComparer_GetHashCodeが呼ばれる
   ↓
2. Object.GetHashCodeに委譲
   ↓
3. ハッシュコードを返却（Long）
```

### 2.3 データ構造

#### 2.3.1 クラスメンバ変数
```vb
' メンバ変数なし（状態を持たないシンプルな実装）
```

### 2.4 実装インターフェース

| インターフェース | 説明 |
|------------------|------|
| IEqualityComparer | 等価比較機能（Equals, GetHashCode） |

---

## 3. インターフェース仕様

### 3.1 IEqualityComparer実装

#### 3.1.1 Equals メソッド
```vb
Private Function IEqualityComparer_Equals(X As Variant, Y As Variant) As Boolean
    IEqualityComparer_Equals = Object.Equals(X, Y)
End Function
```
**パラメータ**:
- X: 比較対象1
- Y: 比較対象2

**戻り値**: 等価であればTrue、そうでなければFalse

**備考**: Object.Equalsに委譲するため、以下の比較が行われる
- 両方がNothing/Null/Emptyの場合は等価
- オブジェクトの場合はIObject.Equalsまたは参照比較
- プリミティブ型は値比較

#### 3.1.2 GetHashCode メソッド
```vb
Private Function IEqualityComparer_GetHashCode(Obj As Variant) As Long
    IEqualityComparer_GetHashCode = Object.GetHashCode(Obj)
End Function
```
**パラメータ**:
- Obj: ハッシュコードを取得するオブジェクト

**戻り値**: オブジェクトのハッシュコード（Long型）

**備考**: Object.GetHashCodeに委譲するため、以下の処理が行われる
- オブジェクトの場合はIObject.GetHashCodeまたはObjPtrベースのハッシュ
- 文字列の場合は文字列ハッシュ
- 数値の場合は値ベースのハッシュ

---

## 4. 内部処理詳細

### 4.1 Object.Equalsへの委譲
EqualityComparerはObject.Equalsスタティックメソッドに処理を委譲する。
Object.Equalsは以下のロジックで等価性を判定する：

1. 参照が同一の場合はTrue
2. どちらかがNothingの場合はFalse
3. IObjectを実装している場合はIObject.Equalsを使用
4. プリミティブ型は値比較

### 4.2 Object.GetHashCodeへの委譲
EqualityComparerはObject.GetHashCodeスタティックメソッドに処理を委譲する。
Object.GetHashCodeは以下のロジックでハッシュコードを計算する：

1. オブジェクトがIObjectを実装している場合はIObject.GetHashCodeを使用
2. 文字列の場合は文字列ハッシュアルゴリズム
3. 数値の場合は値をそのままハッシュコードとして使用
4. オブジェクトの場合はObjPtrをベースにしたハッシュ

---

## 5. 関連コンポーネント

### 5.1 依存クラス

| クラス名 | 用途 |
|----------|------|
| Object | Equals, GetHashCodeのスタティックメソッドを提供 |
| IEqualityComparer | 実装するインターフェース |

### 5.2 使用されるクラス

| クラス名 | 使用方法 |
|----------|----------|
| Hashtable | キーの等価比較とハッシュコード計算 |
| Dictionary | キーの等価比較とハッシュコード計算 |

---

## 6. エラー処理

### 6.1 例外一覧

このクラスは例外をスローしない。すべての処理はObject.Equals/GetHashCodeに委譲され、それらのメソッドが例外処理を行う。

---

## 7. パフォーマンス考慮

### 7.1 計算量

| 操作 | 計算量 |
|------|--------|
| Equals | O(1)（プリミティブ）/ O(n)（文字列）/ 実装依存（オブジェクト） |
| GetHashCode | O(1)（プリミティブ）/ O(n)（文字列）/ 実装依存（オブジェクト） |

### 7.2 推奨事項
- デフォルトの等価比較で十分な場合はこのクラスを使用
- カスタムの等価比較が必要な場合はIEqualityComparerを実装したカスタムクラスを作成
- パフォーマンスが重要な場合は、キーとなるオブジェクトにIObjectを実装

---

## 8. コードリーディングガイド

### 8.1 推奨読解順序

1. **インターフェース定義**
   - `EqualityComparer.cls` 38行目: IEqualityComparerのImplements宣言

2. **メイン処理**
   - `IEqualityComparer_Equals` (44-46行目): 等価比較
   - `IEqualityComparer_GetHashCode` (48-50行目): ハッシュコード取得

3. **依存先の理解**
   - `Object.cls` のEquals/GetHashCodeメソッド（別ファイル）

### 8.2 プログラム呼び出し階層図

```
Hashtable.Add/Item
  └── IEqualityComparer.Equals
        └── EqualityComparer.IEqualityComparer_Equals
              └── Object.Equals
                    ├── IObject.Equals (オブジェクトの場合)
                    └── VB標準比較 (プリミティブの場合)

Hashtable.Add/Item
  └── IEqualityComparer.GetHashCode
        └── EqualityComparer.IEqualityComparer_GetHashCode
              └── Object.GetHashCode
                    ├── IObject.GetHashCode (オブジェクトの場合)
                    └── 値/文字列ハッシュ (プリミティブの場合)
```

### 8.3 データフロー図

```
[Hashtable]
    │
    ↓
[IEqualityComparer]
    │
    ├── Equals(X, Y) ────→ [Object.Equals] ────→ [Boolean]
    │
    └── GetHashCode(Obj) ─→ [Object.GetHashCode] ─→ [Long]
```

### 8.4 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| Source/CorLib/System.Collections/EqualityComparer.cls | クラス | メインクラス |
| Source/CorLib/System.Collections/IEqualityComparer.cls | インターフェース | 等価比較契約 |
| Source/CorLib/System/ObjectStatic.cls | クラス | Equals/GetHashCodeのスタティック実装 |
| Source/CorLib/System/IObject.cls | インターフェース | オブジェクト基本機能契約 |

### 8.5 読解のコツ
- このクラスは非常にシンプル（約51行）で、状態を持たない
- すべての処理はObject.Equals/GetHashCodeに委譲される
- VB_Exposed = Falseのため、ライブラリ外部からは直接インスタンス化できない
- Hashtableなどのコレクション内部でデフォルトの比較クラスとして使用される

---

## 9. 変更履歴

| 日付 | バージョン | 変更内容 |
|------|------------|----------|
| 2015 | 1.0 | 初期実装（Kelly Ethridge） |

---

## 10. 備考

- .NET FrameworkのEqualityComparerと互換性のある設計
- 内部使用クラス（VB_Exposed = False）
- 状態を持たないステートレスな実装
- Hashtableでデフォルトの等価比較クラスとして使用される
- カスタムの等価比較が必要な場合はIEqualityComparerを直接実装
