# 機能設計書 36-FileInfo

## 概要

本ドキュメントは、VBCorLibライブラリにおけるFileInfoクラスの機能設計について記述する。FileInfoクラスは、特定のファイルに関する情報の取得・操作を行うインスタンスクラスである。

### 本機能の処理概要

**業務上の目的・背景**：アプリケーション開発において、特定のファイルに対して複数の操作を行う場面では、毎回パスを指定するよりもオブジェクトとして保持する方が便利である。FileInfoクラスは、ファイル情報をオブジェクトとしてカプセル化し、ファイルの存在確認、属性操作、タイムスタンプ管理、読み書き操作を一貫したインターフェースで提供する。

**機能の利用シーン**：特定のファイルに対する複数の操作、ファイル情報のキャッシュとリフレッシュ、ファイルのコピー・移動・削除、FileStreamやStreamReaderの取得など、オブジェクト指向的なファイル操作が必要な場面で利用される。

**主要な処理内容**：
1. ファイル情報取得（Attributes、Length、Extension、Name、FullName、DirectoryName）
2. タイムスタンプ操作（CreationTime、LastAccessTime、LastWriteTime）- ローカル/UTC両対応
3. ファイル操作（CopyTo、MoveTo、Delete）
4. ストリーム取得（Create、OpenFile、OpenRead、OpenWrite、OpenText、AppendText、CreateText）
5. ディレクトリ情報取得（Directory）
6. 情報リフレッシュ（Refresh）
7. シリアライズ対応（Class_ReadProperties、Class_WriteProperties）

**関連システム・外部連携**：Fileクラス、DirectoryInfoクラス、FileStreamクラス、Pathクラスと連携。FileSystemInfoインターフェースを実装。

**権限による制御**：ファイルシステムのアクセス権限に依存。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないライブラリクラスである |

## 機能種別

ファイル情報オブジェクト

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| FileName | String | Yes | 対象ファイルのパス | 有効なファイルパスであること |
| DestFileName | String | Yes | コピー先/移動先パス | 空文字列の場合はArgumentException |
| OverWrite | Boolean | No | 上書き許可フラグ（デフォルト: False） | - |
| Mode | FileMode | Yes | ファイルオープンモード | - |
| Access | FileAccess | No | アクセスモード（デフォルト: ReadWriteAccess） | - |
| Share | FileShare | No | 共有モード（デフォルト: None） | - |
| Value | Variant | Yes | 属性・タイムスタンプ設定値 | 適切な型であること |

### 入力データソース

コンストラクタで指定されるファイルパス

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Attributes | FileAttributes | ファイル属性（ReadOnly、Hidden等） |
| Length | Currency | ファイルサイズ（バイト） |
| Exists | Boolean | ファイル存在確認結果 |
| Name | String | ファイル名（拡張子含む） |
| FullName | String | フルパス |
| DirectoryName | String | ディレクトリ部分のパス |
| Extension | String | 拡張子 |
| Directory | DirectoryInfo | ディレクトリ情報オブジェクト |
| CreationTime | CorDateTime | 作成日時 |
| LastAccessTime | CorDateTime | 最終アクセス日時 |
| LastWriteTime | CorDateTime | 最終更新日時 |
| FileStream | FileStream | Create、OpenFile等で返されるストリーム |
| StreamReader | StreamReader | OpenTextで返されるリーダー |
| StreamWriter | StreamWriter | AppendText、CreateTextで返されるライター |
| FileInfo | FileInfo | CopyToで返されるコピー先のFileInfo |

### 出力先

呼び出し元のプログラムに戻り値として返される

## 処理フロー

### 処理シーケンス

```
1. FileInfo生成
   └─ パスをフルパスに変換して保持
2. プロパティアクセス
   └─ ダーティフラグをチェック
   └─ ダーティならRefreshを呼び出し
   └─ キャッシュされた値を返す
3. 書き込み操作（属性・タイムスタンプ設定）
   └─ Fileクラスを通じて設定
   └─ ダーティフラグをセット
4. ファイル操作
   └─ Fileクラスの対応メソッドを呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[FileInfo生成] --> B[パスをフルパスに変換]
    B --> C[プロパティアクセス]
    C --> D{ダーティ?}
    D -->|Yes| E[Refresh呼び出し]
    D -->|No| F[キャッシュ値を返す]
    E --> F
    F --> G{書き込み操作?}
    G -->|Yes| H[File.Set系メソッド呼び出し]
    G -->|No| I[終了]
    H --> J[ダーティフラグセット]
    J --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 遅延ロード | ファイル情報はアクセス時に初めて取得 | プロパティ初回アクセス時 |
| BR-002 | ダーティフラグ管理 | 書き込み後は情報を再取得する必要あり | 書き込み操作後 |
| BR-003 | ファイルサイズ形式 | Currency型で10000倍の値として保持 | Length取得時 |
| BR-004 | 存在確認の遅延 | ファイルの存在は操作実行時まで確認されない | コンストラクタ呼び出し時 |
| BR-005 | MoveTo後のパス更新 | MoveToでパス情報も新しいパスに更新 | MoveTo実行時 |

### 計算ロジック

**ファイルサイズの取得**:
```vb
' WIN32_FILE_ATTRIBUTE_DATA構造体からサイズを取得
Ptr = VarPtr(mFileSize)
MemLong(Ptr) = Data.nFileSizeLow
MemLong(Ptr + 4) = Data.nFileSizeHigh
mFileSize = 10000@ * mFileSize  ' Currencyに変換
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| Argument_EmptyFileName | ArgumentException | ファイル名が空文字列 | 有効なファイル名を指定 |
| ERROR_FILE_NOT_FOUND | FileNotFoundException | ファイルが見つからない | ファイルの存在を確認 |

### リトライ仕様

リトライ処理は実装されていない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- ダーティフラグによりファイルシステムへのアクセスを最小化
- Refreshを呼び出すまでキャッシュされた値を使用

## セキュリティ考慮事項

- ファイルシステムのアクセス権限に依存
- シリアライズ時にパス情報が保存される

## 備考

- .NET FrameworkのSystem.IO.FileInfoクラスと高い互換性を持つ
- FileSystemInfoインターフェースを実装しているため多態的に利用可能
- シリアライズ（PropertyBag）に対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 59-67行目のプライベート変数宣言を確認 |

**読解のコツ**: mOriginalFileName（元のパス）、mFullName（フルパス）、mName（ファイル名）、mIOResult（IO結果/ダーティフラグ）が主要な状態変数。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 519-523行目のInit関数でコンストラクタ処理を確認 |
| 2-2 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 525-527行目のClass_Initialize |

**主要処理フロー**:
1. **520-522行目**: パスを保持しフルパスに変換
2. **526行目**: MarkDirtyでダーティフラグをセット

#### Step 3: プロパティアクセスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 79-85行目のAttributesプロパティ |
| 3-2 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 329-335行目のLengthプロパティ |
| 3-3 | FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | 351-370行目のRefreshメソッド |

**主要処理フロー**:
- **80-82行目**: IsDirtyチェックとRefresh呼び出し
- **354行目**: File.GetFileDataで属性データ取得
- **359-361行目**: 属性とタイムスタンプのキャッシュ

### プログラム呼び出し階層図

```
FileInfo
    │
    ├─ Init (コンストラクタ)
    │      └─ Path.GetFullPath
    │
    ├─ Refresh
    │      └─ File.GetFileData
    │             └─ GetFileAttributesExW (Win API)
    │
    ├─ Attributes (Get)
    │      ├─ IsDirty チェック
    │      └─ Refresh（必要時）
    │
    ├─ Attributes (Let)
    │      ├─ SetFileAttributesW (Win API)
    │      └─ MarkDirty
    │
    ├─ CreationTime / LastAccessTime / LastWriteTime
    │      ├─ IsDirty チェック
    │      ├─ Refresh（必要時）
    │      └─ ToLocalTime / ToUniversalTime
    │
    ├─ CopyTo
    │      ├─ File.Copy
    │      └─ NewFileInfo
    │
    ├─ MoveTo
    │      ├─ Path.GetFullPath
    │      ├─ MoveFileW (Win API)
    │      └─ MarkDirty
    │
    ├─ Delete
    │      └─ DeleteFileW (Win API)
    │
    ├─ Create / OpenFile / OpenRead / OpenWrite
    │      └─ File.Create / File.OpenFile / File.OpenRead / File.OpenWrite
    │
    ├─ OpenText / AppendText / CreateText
    │      └─ File.OpenText / File.AppendText / File.CreateText
    │
    └─ Directory
           └─ NewDirectoryInfo
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ファイルパス ────────▶ Init
                           │
                           ▼
                     フルパスに変換
                           │
                           ▼
                     プロパティアクセス
                           │
                           ▼
                     必要ならRefresh
                           │
                           ▼
                     キャッシュ値 ─────────▶ プロパティ値
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FileInfo.cls | `Source/CorLib/System.IO/FileInfo.cls` | ソース | FileInfoクラス本体 |
| File.cls | `Source/CorLib/System.IO/File.cls` | ソース | 静的ファイル操作 |
| DirectoryInfo.cls | `Source/CorLib/System.IO/DirectoryInfo.cls` | ソース | ディレクトリ情報 |
| Path.cls | `Source/CorLib/System.IO/Path.cls` | ソース | パス操作 |
| FileSystemInfo.cls | `Source/CorLib/System.IO/FileSystemInfo.cls` | ソース | 実装インターフェース |
| FileStream.cls | `Source/CorLib/System.IO/FileStream.cls` | ソース | ファイルストリーム |
