# 機能設計書 38-DriveInfo

## 概要

本ドキュメントは、VBCorLibライブラリにおけるDriveInfoクラスの機能設計について記述する。DriveInfoクラスは、特定のドライブに関する情報（容量、フォーマット、ボリュームラベル等）の取得・設定を行うインスタンスクラスである。

### 本機能の処理概要

**業務上の目的・背景**：アプリケーション開発において、ディスク空き容量の確認、ドライブ種別の判定、ボリューム情報の取得などが必要となる場面は多い。DriveInfoクラスは、ドライブ情報をオブジェクトとしてカプセル化し、.NET Frameworkと互換性のあるAPIでドライブ情報へのアクセスを提供する。

**機能の利用シーン**：ディスク空き容量の確認、ドライブの種類判定（ローカル、ネットワーク、CD-ROM等）、ボリュームラベルの取得・設定、ドライブフォーマットの確認、シリアル番号の取得など、ドライブ情報が必要な場面で利用される。

**主要な処理内容**：
1. ドライブ名取得（Name）
2. ルートディレクトリ取得（RootDirectory）
3. ドライブフォーマット取得（DriveFormat）- NTFS、FAT32等
4. ボリュームラベル取得・設定（VolumeLabel）
5. ドライブ種別取得（DriveType）
6. 準備状態確認（IsReady）
7. シリアル番号取得（SerialNumber）
8. 容量情報取得（AvailableFreeSpace、TotalFreeSpace、TotalSize）

**関連システム・外部連携**：DirectoryInfo、Pathクラスと連携。Windows APIを直接呼び出してドライブ情報を取得。

**権限による制御**：ボリュームラベル設定には管理者権限が必要な場合がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないライブラリクラスである |

## 機能種別

ドライブ情報オブジェクト

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Name | String | Yes | ドライブ名（"c"、"c:\"、"\\\\Server\\Share\\"形式） | 空文字列の場合はArgumentException、無効なドライブ名の場合はArgumentException |
| Value | String | Yes | 新しいボリュームラベル | - |

### 入力データソース

コンストラクタで指定されるドライブ名

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Name | String | ドライブ名（"C:\"形式） |
| RootDirectory | DirectoryInfo | ルートディレクトリ情報 |
| DriveFormat | String | ドライブフォーマット（"NTFS"、"FAT32"等） |
| VolumeLabel | String | ボリュームラベル |
| DriveType | DriveType | ドライブ種別（Unknown、NoRootDirectory、Removable、Fixed、Network、CDRom、Ram） |
| IsReady | Boolean | ドライブの準備状態 |
| SerialNumber | Long | ボリュームシリアル番号 |
| AvailableFreeSpace | Currency | 現在のアカウントで利用可能な空き容量 |
| TotalFreeSpace | Currency | ドライブ全体の空き容量 |
| TotalSize | Currency | ドライブ全体の容量 |

### 出力先

呼び出し元のプログラムに戻り値として返される

## 処理フロー

### 処理シーケンス

```
1. DriveInfo生成
   └─ ドライブ名の検証と正規化
2. プロパティアクセス
   └─ 各プロパティで対応するWindows APIを呼び出し
   └─ 結果を整形して返す
3. ボリュームラベル設定
   └─ SetVolumeLabelWを呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[DriveInfo生成] --> B{名前検証}
    B -->|空文字列| C[ArgumentException]
    B -->|有効| D[パス形式判定]
    D -->|1文字| E[": ドライブ名に変換]
    D -->|UNCパス| F[共有名を正規化]
    D -->|ボリューム| G[ルートパスを取得]
    E --> H[mNameに格納]
    F --> H
    G --> H
    H --> I[プロパティアクセス]
    I --> J{API呼び出し}
    J -->|GetVolumeInformation| K[フォーマット/ラベル取得]
    J -->|GetDiskFreeSpaceEx| L[容量情報取得]
    J -->|GetDriveType| M[種別取得]
    K --> N[結果を返す]
    L --> N
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | ドライブ名正規化 | 1文字の場合は":\"を付加して正規化 | コンストラクタ |
| BR-002 | UNCパス対応 | "\\\\Server\\Share\\"形式のパスを受け付ける | コンストラクタ |
| BR-003 | 容量の単位 | Currency型で10000倍の値として返却 | 容量プロパティ |
| BR-004 | IsReady判定 | Directory.Existsで準備状態を判定 | IsReady取得時 |
| BR-005 | ドライブ文字制限 | a-z、A-Zのみ有効 | ボリュームパス時 |

### 計算ロジック

**容量の取得**:
```vb
If GetDiskFreeSpaceEx(mName, AvailableFreeSpace, 0@, 0@) = BOOL_FALSE Then
    IOError.WinIOError Err.LastDllError
End If
AvailableFreeSpace = 10000@ * AvailableFreeSpace  ' Currencyに変換
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentException | ArgumentException | 空文字列のドライブ名 | 有効なドライブ名を指定 |
| ArgumentException | ArgumentException | 無効なドライブ文字 | a-zまたはA-Zを使用 |
| ArgumentException | ArgumentException | 不正なパス形式 | ルートディレクトリ、ドライブ文字、共有名形式を使用 |
| Win32Error | IOException | Windows API呼び出し失敗 | ドライブの存在・準備状態を確認 |

### リトライ仕様

リトライ処理は実装されていない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- 各プロパティアクセス時にWindows APIを呼び出すため、頻繁なアクセスは避ける
- IsReadyでドライブの準備状態を事前確認することを推奨

## セキュリティ考慮事項

- VolumeLabelの設定には適切な権限が必要
- ネットワークドライブのアクセス権限に注意

## 備考

- .NET FrameworkのSystem.IO.DriveInfoクラスと高い互換性を持つ
- シリアライズ（PropertyBag）に対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 54-58行目のプライベート変数と定数を確認 |

**読解のコツ**: mName（ドライブ名）が唯一の状態変数。PATHTYPE_VOLUMEとPATHTYPE_SHAREでパス種別を識別。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 221-223行目のInit関数 |
| 2-2 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 229-249行目のSetName関数 |

**主要処理フロー**:
1. **230-231行目**: 空文字列チェック
2. **233行目**: 1文字の場合は":\"を付加
3. **236-248行目**: パス種別判定と正規化

#### Step 3: 情報取得を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 84-93行目のDriveFormatプロパティ |
| 3-2 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 100-109行目のVolumeLabelプロパティ |
| 3-3 | DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | 157-163行目のAvailableFreeSpaceプロパティ |

**主要処理フロー**:
- **88行目**: GetVolumeInformationでフォーマット取得
- **92行目**: SysAllocStringで文字列を生成
- **158-159行目**: GetDiskFreeSpaceExで容量取得

### プログラム呼び出し階層図

```
DriveInfo
    │
    ├─ Init (コンストラクタ)
    │      └─ SetName
    │             ├─ 空文字列チェック
    │             ├─ GetPathRootAndType
    │             └─ パス正規化
    │
    ├─ Name
    │      └─ mName を返す
    │
    ├─ RootDirectory
    │      └─ NewDirectoryInfo(mName)
    │
    ├─ DriveFormat
    │      └─ GetVolumeInformation (Win API)
    │
    ├─ VolumeLabel (Get)
    │      └─ GetVolumeInformation (Win API)
    │
    ├─ VolumeLabel (Let)
    │      └─ SetVolumeLabel (Win API)
    │
    ├─ DriveType
    │      └─ GetDriveType (Win API)
    │
    ├─ IsReady
    │      └─ Directory.Exists
    │
    ├─ SerialNumber
    │      └─ GetVolumeInformation (Win API)
    │
    ├─ AvailableFreeSpace
    │      └─ GetDiskFreeSpaceEx (Win API)
    │
    ├─ TotalFreeSpace
    │      └─ GetDiskFreeSpaceEx (Win API)
    │
    └─ TotalSize
           └─ GetDiskFreeSpaceEx (Win API)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ドライブ名 ─────────▶ SetName
                           │
                           ▼
                     パス正規化
                           │
                           ▼
                     プロパティアクセス
                           │
                           ▼
                     Windows API呼び出し
                           │
                           ▼
                     結果整形 ─────────▶ プロパティ値
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| DriveInfo.cls | `Source/CorLib/System.IO/DriveInfo.cls` | ソース | DriveInfoクラス本体 |
| DriveInfoStatic.cls | `Source/CorLib/System.IO/DriveInfoStatic.cls` | ソース | 静的メソッド（GetDrives等） |
| DirectoryInfo.cls | `Source/CorLib/System.IO/DirectoryInfo.cls` | ソース | RootDirectoryで使用 |
| Directory.cls | `Source/CorLib/System.IO/Directory.cls` | ソース | IsReadyで使用 |
| Path.cls | `Source/CorLib/System.IO/Path.cls` | ソース | パスルート取得で使用 |
