# 機能設計書 39-Path

## 概要

本ドキュメントは、VBCorLibライブラリにおけるPathクラスの機能設計について記述する。Pathクラスは、ファイルパスやディレクトリパスの操作（結合、分解、正規化等）を行う静的メソッドを提供するユーティリティクラスである。

### 本機能の処理概要

**業務上の目的・背景**：アプリケーション開発において、ファイルパスの操作は非常に頻繁に必要となる処理である。Pathクラスは、.NET Frameworkと互換性のあるパス操作APIを提供し、パスの結合、ディレクトリ名やファイル名の抽出、拡張子の操作、相対パスから絶対パスへの変換などを安全かつ統一的に行えるようにする。

**機能の利用シーン**：パスの結合（ディレクトリ+ファイル名）、ファイル名や拡張子の抽出、拡張子の変更、相対パスから絶対パスへの変換、一時ファイルパスの生成、パスの検証など、パス文字列の操作が必要なすべての場面で利用される。

**主要な処理内容**：
1. パス結合（Combine、CombineArray）
2. パス分解（GetDirectoryName、GetFileName、GetFileNameWithoutExtension、GetExtension）
3. パス判定（IsPathRooted、HasExtension）
4. パス変換（GetFullPath、GetPathRoot、ChangeExtension）
5. 一時パス取得（GetTempFileName、GetTempPath）
6. 不正文字取得（GetInvalidPathChars、GetInvalidFileNameChars）
7. セパレータ定数（DirectorySeparator、AltDirectorySeparator、VolumeSeparator、PathSeparator）

**関連システム・外部連携**：File、Directory、FileInfo、DirectoryInfoなど、すべてのファイルシステム関連クラスで内部的に使用。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないライブラリクラスである |

## 機能種別

パス操作ユーティリティ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Path | String | Yes | 操作対象のパス | 不正文字チェック |
| Path1 | String | Yes | 結合するパスの1番目 | 不正文字チェック |
| Path2 | String | Yes | 結合するパスの2番目 | 不正文字チェック |
| Paths | Variant() | No | 追加で結合するパス | - |
| Extension | String | No | 新しい拡張子 | - |

### 入力データソース

プログラムから直接渡されるパス文字列

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| String | String | 各種パス操作の結果文字列 |
| Boolean | Boolean | IsPathRooted、HasExtensionの判定結果 |
| Integer() | Integer() | GetInvalidPathChars、GetInvalidFileNameCharsの不正文字配列 |

### 出力先

呼び出し元のプログラムに戻り値として返される

## 処理フロー

### 処理シーケンス

```
1. パス入力
   └─ 不正文字のチェック
2. パス操作実行
   └─ 文字列処理またはWindows API呼び出し
3. 結果の返却
```

### フローチャート

```mermaid
flowchart TD
    A[パス入力] --> B{不正文字?}
    B -->|Yes| C[ArgumentException]
    B -->|No| D{操作種別}
    D -->|Combine| E[パス結合処理]
    D -->|GetDirectoryName| F[ディレクトリ名抽出]
    D -->|GetFileName| G[ファイル名抽出]
    D -->|GetFullPath| H[正規化処理]
    D -->|GetTempFileName| I[Windows API呼び出し]
    E --> J[結果を返す]
    F --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | パス結合時のセパレータ | 必要に応じてディレクトリセパレータを挿入 | Combine実行時 |
| BR-002 | ルートパスの優先 | 後のパスがルートパスの場合、前のパスは無視 | Combine実行時 |
| BR-003 | 不正文字の定義 | 制御文字、<>"|等がパスで使用禁止 | パス操作時 |
| BR-004 | 追加の不正文字（ファイル名） | パスに加え、:\*?\\/がファイル名で使用禁止 | ファイル名操作時 |
| BR-005 | セパレータの正規化 | AltDirectorySeparator(/)はDirectorySeparator(\)に変換 | GetFullPath実行時 |

### 計算ロジック

**パス結合処理**:
```vb
If IsPathRooted(Path2) Then
    CombinePaths = Path2  ' 後のパスがルートなら前を無視
Else
    If IsSeparator(LastChar(Path1)) Then
        CombinePaths = Path1 & Path2
    Else
        CombinePaths = Path1 & "\" & Path2
    End If
End If
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| Argument_InvalidPathChars | ArgumentException | 不正な文字を含むパス | 不正文字を除去 |
| Argument_EmptyPath | ArgumentException | GetDirectoryNameに空文字列 | 有効なパスを指定 |
| Argument_PathUriFormatNotSupported | ArgumentException | http:やfile:形式のパス | ローカルパス形式を使用 |
| Arg_PathIllegalUNC | ArgumentException | 不正なUNCパス形式 | 正しいUNC形式を使用 |
| Argument_PathFormatNotSupported | NotSupportedException | サポートされないパス形式 | 標準的なパス形式を使用 |

### リトライ仕様

リトライ処理は実装されていない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- 文字列処理が中心のため、大量の呼び出しでもオーバーヘッドは小さい
- GetFullPathは内部で複雑な正規化処理を行うため比較的負荷が高い

## セキュリティ考慮事項

- パストラバーサル攻撃に注意（GetFullPathで正規化推奨）
- 不正文字チェックにより基本的な入力検証を実施

## 備考

- .NET FrameworkのSystem.IO.Pathクラスと高い互換性を持つ
- 静的クラスとして設計されており、インスタンス化不要
- Windows固有のパス形式（ドライブレター、UNC等）に対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 定数とデータ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 44-47行目のプライベート配列変数を確認 |
| 1-2 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 503-507行目のClass_Initializeで配列初期化を確認 |
| 1-3 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 61-169行目のセパレータ定数プロパティを確認 |

**読解のコツ**: mInvalidPathChars、mInvalidFileNameChars、mTrimEndCharsが不正文字や空白トリム用の配列。セパレータ定数はWindows標準（\=92、/=47、:=58、;=59）。

#### Step 2: 基本操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 261-273行目のCombine関数 |
| 2-2 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 635-654行目のCombinePaths内部関数 |
| 2-3 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 216-230行目のIsPathRooted関数 |

**主要処理フロー**:
- **266行目**: 最初の2つのパスを結合
- **268-270行目**: ParamArray引数を順次結合
- **636-638行目**: 後のパスがルートならそのまま返す

#### Step 3: 分解・抽出操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 308-333行目のGetDirectoryName関数 |
| 3-2 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 359-370行目のGetFileName関数 |
| 3-3 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 342-350行目のGetExtension関数 |

**主要処理フロー**:
- **317行目**: NormalizePathで正規化
- **321行目**: InStrRevで最後のセパレータを検索
- **347-349行目**: ピリオド位置から末尾まで抽出

#### Step 4: 正規化処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 399-408行目のGetFullPath関数 |
| 4-2 | Path.cls | `Source/CorLib/System.IO/Path.cls` | 679-710行目のNormalizePath内部関数 |

**主要処理フロー**:
- **402行目**: NormalizePathでパスを正規化
- **691行目**: PrepareForUNCPathでUNCパス処理
- **693行目**: PrepareForVolumeLabelでボリュームラベル処理

### プログラム呼び出し階層図

```
Path (静的クラス)
    │
    ├─ セパレータ定数
    │      ├─ DirectorySeparator / DirectorySeparatorChar
    │      ├─ AltDirectorySeparator / AltDirectorySeparatorChar
    │      ├─ VolumeSeparator / VolumeSeparatorChar
    │      └─ PathSeparator / PathSeparatorChar
    │
    ├─ Combine / CombineArray
    │      ├─ CheckValidPathChars
    │      ├─ CombinePaths (内部)
    │      │      ├─ IsPathRooted
    │      │      └─ IsSeparator
    │      └─ Object.ToString
    │
    ├─ GetDirectoryName
    │      ├─ CheckValidPathChars
    │      ├─ NormalizePath (内部)
    │      └─ GetRootLength (内部)
    │
    ├─ GetFileName
    │      ├─ CheckValidPathChars
    │      └─ GetDirectoryCharIndex (内部)
    │
    ├─ GetExtension
    │      └─ GetExtensionCharIndex (内部)
    │
    ├─ GetFileNameWithoutExtension
    │      ├─ GetFileName
    │      └─ InStrRev
    │
    ├─ GetFullPath
    │      ├─ NormalizePath (内部)
    │      │      ├─ CleanPath (内部)
    │      │      ├─ PrepareForUNCPath (内部)
    │      │      ├─ PrepareForVolumeLabel (内部)
    │      │      ├─ ProcessPreSigCharSegment (内部)
    │      │      ├─ ProcessSigCharSegment (内部)
    │      │      └─ ValidateUNCPath (内部)
    │      └─ PathHelper.GetFullPathName
    │
    ├─ GetPathRoot
    │      ├─ NormalizePath (内部)
    │      └─ GetRootLength (内部)
    │
    ├─ ChangeExtension
    │      ├─ CheckValidPathChars
    │      └─ GetExtensionCharIndex (内部)
    │
    ├─ IsPathRooted
    │      ├─ CheckValidPathChars
    │      └─ FirstTwoChars (内部)
    │
    ├─ HasExtension
    │      ├─ CheckValidPathChars
    │      └─ GetExtensionCharIndex (内部)
    │
    ├─ GetTempFileName
    │      ├─ GetTempPath
    │      └─ GetTempFileNameW (Win API)
    │
    ├─ GetTempPath
    │      ├─ GetTempPathW (Win API)
    │      └─ GetFullPath
    │
    └─ GetInvalidPathChars / GetInvalidFileNameChars
           └─ 配列を返す
```

### データフロー図

```
[入力]                    [処理]                         [出力]

パス文字列 ─────────▶ 不正文字チェック
                           │
                           ▼
                     文字列操作/API呼び出し
                           │
                           ▼
                     結果文字列 ─────────▶ 呼び出し元
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Path.cls | `Source/CorLib/System.IO/Path.cls` | ソース | Pathクラス本体 |
| PathHelper.cls | `Source/CorLib/System.IO/PathHelper.cls` | ソース | パス正規化ヘルパー |
| CorString.cls | `Source/CorLib/System/CorString.cls` | ソース | 文字列操作 |
| File.cls | `Source/CorLib/System.IO/File.cls` | ソース | Pathを内部利用 |
| Directory.cls | `Source/CorLib/System.IO/Directory.cls` | ソース | Pathを内部利用 |
