# 機能設計書 42-Encoding

## 概要

本ドキュメントは、VBCorLibライブラリにおけるテキストエンコーディング基底クラス「Encoding」の機能設計について記述する。

### 本機能の処理概要

Encodingクラスは、Unicode文字とバイト配列間の変換を行うエンコーディングプロバイダのインターフェースを定義する抽象基底クラスである。具体的な実装（UTF8Encoding、ASCIIEncoding等）はこのインターフェースを継承し、各エンコーディング固有の変換ロジックを提供する。

**業務上の目的・背景**：テキストデータのファイル保存、ネットワーク送受信、異なるシステム間でのデータ交換において、文字エンコーディングの変換は不可欠である。.NET FrameworkのSystem.Text.Encodingクラスと互換性のあるAPIを提供し、VBA/VB6環境でも一貫したエンコーディング処理を可能にする。

**機能の利用シーン**：
- テキストファイルの読み書き（UTF-8、Shift-JIS等）
- Webリクエスト/レスポンスの文字コード変換
- データベースとの文字列データ交換
- メール送受信時のエンコーディング処理
- 異なるシステム間でのデータ連携

**主要な処理内容**：
1. 文字列/文字配列からバイト配列へのエンコード（GetBytes、GetBytesEx）
2. バイト配列から文字列/文字配列へのデコード（GetChars、GetCharsEx、GetString）
3. エンコード/デコードに必要なバイト数/文字数の計算（GetByteCount、GetCharCount）
4. BOM（Byte Order Mark）の取得（GetPreamble）
5. Encoder/Decoderオブジェクトの取得（ストリーム処理用）
6. フォールバック処理の設定（変換不可能な文字の処理）

**関連システム・外部連携**：EncodingStaticクラスを通じて、各種エンコーディングインスタンスを取得する。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | Base64 Encoding | 参照 | Base64変換時のエンコーディング選択 |
| 6 | Real-Time Decryption | 参照 | 暗号化テキストのエンコーディング処理 |
| 7 | File Encryption | 参照 | ファイル暗号化時のテキストエンコーディング |

## 機能種別

データ変換処理 / ユーティリティ機能

## 入力仕様

### 入力パラメータ（GetBytesメソッド）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Chars | Integer() または String | Yes | エンコード対象の文字列/文字配列 | 初期化済み配列または非空文字列 |
| Index | Long | No | 開始インデックス | 有効な配列範囲内 |
| Count | Long | No | 処理文字数 | 0以上、配列範囲内 |

### 入力パラメータ（GetCharsメソッド）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Bytes | Byte() | Yes | デコード対象のバイト配列 | 初期化済み配列 |
| Index | Long | No | 開始インデックス | 有効な配列範囲内 |
| Count | Long | No | 処理バイト数 | 0以上、配列範囲内 |

### 入力データソース

プログラムからの直接呼び出し（パラメータ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| GetBytes戻り値 | Byte() | エンコードされたバイト配列 |
| GetChars戻り値 | Integer() | デコードされた文字配列 |
| GetString戻り値 | String | デコードされた文字列 |
| GetByteCount戻り値 | Long | エンコードに必要なバイト数 |
| GetCharCount戻り値 | Long | デコードで生成される文字数 |
| GetPreamble戻り値 | Byte() | BOM（バイトオーダーマーク） |

### 出力先

呼び出し元への戻り値として返却、またはByRef引数への直接書き込み

## 処理フロー

### 処理シーケンス

```
1. エンコーディングインスタンス取得
   └─ Encoding.GetEncoding(コードページ) または直接インスタンス化
2. エンコード処理
   └─ GetByteCount: 必要バイト数計算
   └─ GetBytes/GetBytesEx: 文字→バイト変換
3. デコード処理
   └─ GetCharCount: 生成文字数計算
   └─ GetChars/GetCharsEx/GetString: バイト→文字変換
4. ストリーム処理
   └─ GetEncoder: Encoderオブジェクト取得（状態保持）
   └─ GetDecoder: Decoderオブジェクト取得（状態保持）
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{処理種別}
    B -->|エンコード| C[GetByteCount呼び出し]
    C --> D[出力バッファ確保]
    D --> E[GetBytes/GetBytesEx呼び出し]
    E --> F{フォールバック発生?}
    F -->|Yes| G[EncoderFallback処理]
    G --> H[代替文字/例外]
    F -->|No| I[バイト配列返却]
    H --> I
    B -->|デコード| J[GetCharCount呼び出し]
    J --> K[出力バッファ確保]
    K --> L[GetChars/GetString呼び出し]
    L --> M{フォールバック発生?}
    M -->|Yes| N[DecoderFallback処理]
    N --> O[代替文字/例外]
    M -->|No| P[文字列/文字配列返却]
    O --> P
    I --> Q[終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | BOM出力 | GetPreambleは対象エンコーディングのBOMを返す（非対応の場合は空配列） | BOM対応エンコーディング |
| BR-002 | フォールバック処理 | 変換不能文字はEncoderFallback/DecoderFallbackで処理 | 変換不能文字検出時 |
| BR-003 | 読み取り専用判定 | IsReadOnlyがTrueの場合、フォールバック設定変更不可 | フォールバック設定時 |
| BR-004 | Clone可変性 | Cloneで作成されたインスタンスは常に書き込み可能 | Clone時 |
| BR-005 | コードページ一意性 | 各エンコーディングは一意のCodePageを持つ | インスタンス生成時 |

### 計算ロジック

```
GetMaxByteCount = 1文字あたりの最大バイト数 × CharCount
GetMaxCharCount = ByteCount / 1文字あたりの最小バイト数
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | Bytes/Chars配列が未初期化 | 配列を初期化してから呼び出し |
| - | ArgumentOutOfRangeException | Index/Countが配列範囲外 | 有効な範囲を指定 |
| - | EncoderFallbackException | エンコード不能文字（ExceptionFallback設定時） | フォールバック設定を変更またはデータ修正 |
| - | DecoderFallbackException | デコード不能バイト列（ExceptionFallback設定時） | フォールバック設定を変更またはデータ修正 |
| - | ArgumentException | サポートされないコードページ | サポートされるコードページを使用 |

### リトライ仕様

データ変換処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 数KB程度のテキスト変換: 数ミリ秒以内
- 大規模テキスト（数MB）の変換: GetEncoder/GetDecoderによるストリーム処理を推奨

## セキュリティ考慮事項

- 不正なバイト列のデコード時、フォールバック設定により動作が異なる
- ExceptionFallbackを使用すると、不正データ検出時に例外発生
- ReplacementFallbackを使用すると、不正データは代替文字に置換される

## 備考

- Encodingクラスはインターフェース定義であり、直接インスタンス化はできない
- 具体的な実装はUTF8Encoding、ASCIIEncoding等の派生クラスで提供される
- EncodingStaticクラス（Encoding静的メンバ相当）を通じてインスタンスを取得する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェース定義を理解する

Encodingクラスはインターフェースとして機能し、各メソッドは空の実装となっている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Encoding.cls | `Source/CorLib/System.Text/Encoding.cls` | インターフェース全体構造、各メソッド/プロパティの定義 |

**読解のコツ**: 38-49行目のクラスコメントで概要を把握。各メソッドは空実装（End Function/End Property）だが、コメントで仕様が詳細に記載されている。

#### Step 2: 静的メソッド（ファクトリ）を理解する

EncodingStaticクラスがエンコーディングインスタンスの取得を担当する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | EncodingStatic.cls | `Source/CorLib/System.Text/EncodingStatic.cls` | 各エンコーディングの取得プロパティ、GetEncodingメソッド |

**主要処理フロー**:
1. **80-86行目**: Defaultプロパティ - システムデフォルトエンコーディング取得
2. **93-99行目**: BigEndianUnicodeプロパティ - UTF-16 BE取得
3. **106-112行目**: Unicodeプロパティ - UTF-16 LE取得
4. **119-125行目**: ASCIIプロパティ - ASCII取得
5. **132-138行目**: UTF7プロパティ - UTF-7取得
6. **145-151行目**: UTF8プロパティ - UTF-8取得
7. **158-164行目**: UTF32プロパティ - UTF-32取得
8. **225-265行目**: GetEncodingメソッド - コードページ/名前からエンコーディング取得

#### Step 3: 具体的な実装を理解する

代表的な実装クラスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UTF8Encoding.cls | `Source/CorLib/System.Text/UTF8Encoding.cls` | UTF-8の具体的な実装 |
| 3-2 | ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | ASCIIの具体的な実装 |

### プログラム呼び出し階層図

```
Encoding.GetEncoding(CodePage) [EncodingStatic]
    │
    ├─ CodePage判定
    │      ├─ 1200 → Unicode (UTF-16 LE)
    │      ├─ 1201 → BigEndianUnicode (UTF-16 BE)
    │      ├─ 20127 → ASCII
    │      ├─ 65000 → UTF7
    │      ├─ 65001 → UTF8
    │      ├─ 12000 → UTF32
    │      └─ その他 → CodePageEncoding
    │
    └─ Encodingインスタンス返却

Encoding.GetBytes(Chars)
    │
    ├─ GetByteCount(Chars) → 必要バイト数計算
    │
    └─ GetBytesEx(Chars, 0, Count, Bytes, 0)
           │
           ├─ 文字→バイト変換ループ
           │      └─ エンコード不能時: EncoderFallback
           │
           └─ バイト配列返却

Encoding.GetString(Bytes)
    │
    ├─ GetCharCount(Bytes) → 生成文字数計算
    │
    └─ GetChars(Bytes)
           │
           ├─ バイト→文字変換ループ
           │      └─ デコード不能時: DecoderFallback
           │
           └─ 文字列構築・返却
```

### データフロー図

```
[入力]                    [処理]                         [出力]

文字列/文字配列 ─────────▶ GetBytes ──────────────────▶ バイト配列
                              │
                              ├─ GetByteCount計算
                              ├─ 文字→バイト変換
                              └─ EncoderFallback処理

バイト配列 ────────────────▶ GetChars/GetString ───────▶ 文字配列/文字列
                              │
                              ├─ GetCharCount計算
                              ├─ バイト→文字変換
                              └─ DecoderFallback処理

（なし） ──────────────────▶ GetPreamble ──────────────▶ BOM (Byte[])

CharCount ────────────────▶ GetMaxByteCount ──────────▶ 最大バイト数
ByteCount ────────────────▶ GetMaxCharCount ──────────▶ 最大文字数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Encoding.cls | `Source/CorLib/System.Text/Encoding.cls` | ソース | インターフェース定義（368行） |
| EncodingStatic.cls | `Source/CorLib/System.Text/EncodingStatic.cls` | ソース | 静的メソッド/ファクトリ（316行） |
| UTF8Encoding.cls | `Source/CorLib/System.Text/UTF8Encoding.cls` | ソース | UTF-8実装 |
| UTF7Encoding.cls | `Source/CorLib/System.Text/UTF7Encoding.cls` | ソース | UTF-7実装 |
| UTF32Encoding.cls | `Source/CorLib/System.Text/UTF32Encoding.cls` | ソース | UTF-32実装 |
| UnicodeEncoding.cls | `Source/CorLib/System.Text/UnicodeEncoding.cls` | ソース | UTF-16実装 |
| ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | ソース | ASCII実装 |
| EncoderFallback.cls | `Source/CorLib/System.Text/EncoderFallback.cls` | ソース | エンコードフォールバック |
| DecoderFallback.cls | `Source/CorLib/System.Text/DecoderFallback.cls` | ソース | デコードフォールバック |
| Encoder.cls | `Source/CorLib/System.Text/Encoder.cls` | ソース | ストリーム用エンコーダ |
| Decoder.cls | `Source/CorLib/System.Text/Decoder.cls` | ソース | ストリーム用デコーダ |
| EncodingTable.cls | `Source/CorLib/System.Text/EncodingTable.cls` | ソース | コードページ/名前対応表 |
| EncodingInfo.cls | `Source/CorLib/System.Text/EncodingInfo.cls` | ソース | エンコーディング情報 |
