# 機能設計書 47-ASCIIEncoding

## 概要

本ドキュメントは、VBCorLibライブラリにおけるASCIIエンコーディング機能「ASCIIEncoding」クラスの機能設計について記述する。

### 本機能の処理概要

ASCIIEncodingクラスは、Unicode文字をASCII形式（7ビット、0x00-0x7F）のバイト配列に変換（エンコード）、およびASCII形式のバイト配列をUnicode文字に変換（デコード）する機能を提供する。ASCII範囲外の文字はフォールバック処理される。

**業務上の目的・背景**：ASCIIは最も基本的で広くサポートされる文字エンコーディング。レガシーシステム、プロトコル、ファイルフォーマットとの互換性維持に必須。

**機能の利用シーン**：
- レガシーシステムとのデータ連携
- テキストプロトコル（HTTP、SMTP等）のヘッダー処理
- 英語圏向けテキストファイルの処理
- ASCII限定のファイルフォーマット（一部のCSV等）

**主要な処理内容**：
1. Unicode文字（0x00-0x7F）からASCIIバイト配列へのエンコード
2. ASCIIバイト配列からUnicode文字へのデコード
3. ASCII範囲外文字のフォールバック処理
4. 最適化された高速エンコード/デコード

**関連システム・外部連携**：EncodingStaticクラスを通じて取得可能。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 4 | Base64 Encoding | 参照 | Base64変換時のASCIIエンコーディング |

## 機能種別

データ変換処理 / テキストエンコーディング

## 入力仕様

### 入力パラメータ（エンコード処理）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Chars | Integer() または String | Yes | エンコード対象の文字配列/文字列 | 初期化済み配列または有効な文字列 |
| Index | Long | No | 開始インデックス（0ベース） | 有効な配列範囲内 |
| Count | Long | No | 処理文字数 | 0以上、配列範囲内 |

### 入力パラメータ（デコード処理）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Bytes | Byte() | Yes | デコード対象のバイト配列 | 初期化済み配列 |
| Index | Long | No | 開始インデックス（0ベース） | 有効な配列範囲内 |
| Count | Long | No | 処理バイト数 | 0以上、配列範囲内 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| GetBytes戻り値 | Byte() | ASCIIエンコードされたバイト配列（1バイト/文字） |
| GetChars戻り値 | Integer() | デコードされたUnicode文字配列 |
| GetString戻り値 | String | デコードされたUnicode文字列 |
| GetByteCount戻り値 | Long | エンコードに必要なバイト数 |
| GetCharCount戻り値 | Long | デコードで生成される文字数 |
| GetPreamble戻り値 | Byte() | 常に空配列（ASCIIにBOMはない） |

### ASCIIエンコーディング規則

```
有効な文字範囲: U+0000 - U+007F (7ビット)
1文字 = 1バイト

文字 → バイト変換:
  0x00-0x7F: そのまま出力
  0x80以上: フォールバック処理

バイト → 文字変換:
  0x00-0x7F: そのまま出力
  0x80-0xFF: フォールバック処理
```

## 処理フロー

### 処理シーケンス

```
1. インスタンス生成
   └─ フォールバック設定の初期化（デフォルト: "?"置換）
2. エンコード処理 (GetBytes)
   └─ GetByteCount: 必要バイト数計算
   └─ GetBytesCore: 文字→ASCIIバイト変換
       ├─ ASCII文字 (< 0x80): 直接出力
       └─ 非ASCII文字: EncoderFallback処理
3. デコード処理 (GetString/GetChars)
   └─ GetCharCount: 生成文字数計算
   └─ GetCharsCore: ASCIIバイト→文字変換
       ├─ 有効バイト (< 0x80): 直接出力
       └─ 無効バイト (>= 0x80): DecoderFallback処理
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{処理種別}
    B -->|エンコード| C[文字ループ開始]
    C --> D{ASCII範囲内?}
    D -->|Yes| E[1バイト出力]
    D -->|No| F[EncoderFallback処理]
    F --> G[代替文字出力]
    E --> H{次の文字?}
    G --> H
    H -->|Yes| C
    H -->|No| I[バイト配列返却]
    B -->|デコード| J[バイトループ開始]
    J --> K{有効ASCII?}
    K -->|Yes| L[文字出力]
    K -->|No| M[DecoderFallback処理]
    M --> N[代替文字出力]
    L --> O{次のバイト?}
    N --> O
    O -->|Yes| J
    O -->|No| P[文字列返却]
    I --> Q[終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | シングルバイト | 1文字は常に1バイトでエンコードされる | 常時 |
| BR-002 | ASCII範囲制限 | 0x00-0x7Fの文字のみ直接エンコード可能 | エンコード時 |
| BR-003 | 非ASCII処理 | 0x80以上の文字はEncoderFallbackで処理 | エンコード時 |
| BR-004 | 無効バイト処理 | 0x80以上のバイトはDecoderFallbackで処理 | デコード時 |
| BR-005 | BOMなし | GetPreambleは常に空配列を返す | GetPreamble呼び出し時 |
| BR-006 | コードページ | CodePageは常に20127を返す | CodePage取得時 |

### 計算ロジック

```
GetMaxByteCount(CharCount) = CharCount * EncoderFallback.MaxCharCount
GetMaxCharCount(ByteCount) = ByteCount * DecoderFallback.MaxCharCount
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentNullException | Bytes/Chars配列が未初期化 | 配列を初期化してから呼び出し |
| - | ArgumentOutOfRangeException | Index/Countが配列範囲外 | 有効な範囲を指定 |
| - | ArgumentOutOfRangeException | CharCount/ByteCountが負の値 | 0以上の値を指定 |
| - | EncoderFallbackException | 非ASCII文字（ExceptionFallback設定時） | フォールバック設定変更またはデータ修正 |
| - | DecoderFallbackException | 無効バイト（ExceptionFallback設定時） | フォールバック設定変更またはデータ修正 |
| - | ArgumentException | 出力バッファが小さすぎる | 十分なサイズのバッファを確保 |
| - | InvalidOperationException | 読み取り専用インスタンスのフォールバック設定変更 | Cloneで書き込み可能なコピーを作成 |

### リトライ仕様

データ変換処理であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- シングルバイトエンコーディングのため非常に高速
- 最適化された高速ループによる処理
- 数KB程度のテキスト変換: 数ミリ秒以内

## セキュリティ考慮事項

- 非ASCII文字が含まれる場合、情報の欠落が発生する可能性
- フォールバック設定により非ASCII文字の処理方法を制御可能
- ExceptionFallbackを使用すると、非ASCII文字検出時に例外発生

## 備考

- ASCIIはUS-ASCII、ANSI X3.4とも呼ばれる
- 英語圏以外の文字を扱う場合はUTF-8等の使用を推奨
- .NET FrameworkのASCIIEncodingと高い互換性を持つ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造とクラス定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | クラス定義、プライベート変数（58-61行目） |

**読解のコツ**:
- 55行目: CODE_PAGE定数（20127）
- 58-61行目: mIsReadOnly、mEncoderFallback、mDecoderFallback

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | 公開メソッド（GetBytes、GetChars、GetString等） |

**主要処理フロー**:
1. **244-259行目**: GetByteCountメソッド - エンコードに必要なバイト数計算
2. **273-293行目**: GetBytesメソッド - 文字→バイト配列変換
3. **321-349行目**: GetCharCountメソッド - デコードで生成される文字数計算
4. **363-372行目**: GetCharsメソッド - バイト→文字配列変換
5. **406-426行目**: GetStringメソッド - バイト→文字列変換
6. **433-435行目**: GetPreambleメソッド - 空配列を返す

#### Step 3: コア処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | GetBytesCore/GetCharsCore（内部処理） |

**主要処理フロー**:
- **550-650行目**: GetByteCountCore - エンコードバイト数計算
  - 高速ループでASCII文字をスキップ
  - 非ASCII文字のフォールバック計算
- **652-797行目**: GetBytesCore - 文字→バイト変換の実装
  - ASCII文字は直接コピー
  - 非ASCII文字はEncoderFallback処理
- **799-868行目**: GetCharCountCore - デコード文字数計算
- **870-988行目**: GetCharsCore - バイト→文字変換の実装
  - 有効ASCIIバイトは直接コピー
  - 無効バイトはDecoderFallback処理

### プログラム呼び出し階層図

```
ASCIIEncoding
    │
    ├─ GetBytes(Chars)
    │      ├─ AsChars(): 文字配列変換
    │      ├─ ValidateGetBytes(): バリデーション
    │      ├─ GetByteCountCore(): バイト数計算
    │      │      ├─ ASCII高速ループ (< 0x80)
    │      │      └─ フォールバック計算
    │      └─ GetBytesCore(): エンコード実行
    │             ├─ ASCII文字: 直接コピー
    │             └─ FallbackUnknownChar(): フォールバック
    │
    ├─ GetString(Bytes)
    │      ├─ ValidateGetChars(): バリデーション
    │      ├─ GetCharCountCore(): 文字数計算
    │      │      ├─ 有効バイトカウント
    │      │      └─ フォールバック計算
    │      └─ GetCharsCore(): デコード実行
    │             ├─ 有効ASCII: 直接コピー
    │             └─ InsertFallbackChars(): フォールバック
    │
    ├─ GetEncoder()
    │      └─ ASCIIEncoder.Init(Me)
    │
    └─ GetDecoder()
           └─ ASCIIDecoder.Init(Me)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

文字列 ──────────────────▶ GetBytes ──────────────────▶ ASCIIバイト配列
  │                           │                          (1バイト/文字)
  ├─ ASCII文字 (< 0x80)       ├─ 直接1バイト出力
  └─ 非ASCII文字              └─ フォールバック("?"等)

ASCIIバイト配列 ──────────▶ GetString ─────────────────▶ 文字列
  │                           │
  ├─ 有効ASCII (< 0x80)       ├─ 直接文字出力
  └─ 無効バイト (>= 0x80)     └─ フォールバック

（なし） ──────────────────▶ GetPreamble ──────────────▶ 空配列
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ASCIIEncoding.cls | `Source/CorLib/System.Text/ASCIIEncoding.cls` | ソース | メインクラス定義（988行） |
| ASCIIEncoder.cls | `Source/CorLib/System.Text/ASCIIEncoder.cls` | ソース | ストリーム用エンコーダ |
| ASCIIDecoder.cls | `Source/CorLib/System.Text/ASCIIDecoder.cls` | ソース | ストリーム用デコーダ |
| Encoding.cls | `Source/CorLib/System.Text/Encoding.cls` | ソース | インターフェース定義 |
| EncodingStatic.cls | `Source/CorLib/System.Text/EncodingStatic.cls` | ソース | ASCIIプロパティ（119-125行目） |
