# 機能設計書 48-StringBuilder

## 概要

本ドキュメントは、VBCorLibライブラリにおける文字列構築機能「StringBuilder」クラスの機能設計について記述する。

### 本機能の処理概要

StringBuilderクラスは、可変長の文字列を効率的に構築・操作するためのクラスである。VB6/VBAの標準文字列連結（&演算子）は連結のたびにメモリ再確保が発生し非効率だが、StringBuilderは内部バッファを使用してメモリ再確保を最小限に抑え、高速な文字列操作を実現する。

**業務上の目的・背景**：大量の文字列連結が必要な場面（ログ出力、レポート生成、HTML/XML構築等）で、標準の文字列連結より大幅に高速な処理を提供。.NET FrameworkのSystem.Text.StringBuilderと互換性のあるAPIを持つ。

**機能の利用シーン**：
- 大量のテキストデータの連結
- 動的なHTML/XML生成
- ログメッセージの構築
- CSVデータの生成
- フォーマット文字列の構築

**主要な処理内容**：
1. 文字列/文字の追加（Append、AppendChar、AppendLine）
2. 書式付き文字列の追加（AppendFormat）
3. 文字列の挿入（Insert、InsertChars）
4. 文字列の削除（Remove）
5. 文字列の置換（Replace）
6. 容量管理（Capacity、EnsureCapacity）

**関連システム・外部連携**：IFormatProvider、ICustomFormatterインターフェースを使用した書式設定をサポート。

**権限による制御**：本機能には権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面からの直接操作を持たない |

## 機能種別

データ操作処理 / ユーティリティ機能

## 入力仕様

### コンストラクタパラメータ

| パラメータ名 | 型 | 必須 | 説明 | デフォルト値 |
|-------------|-----|-----|------|-------------|
| s | String | No | 初期文字列 | 空文字列 |
| StartIndex | Long | No | 初期文字列の開始インデックス | 0 |
| Length | Long | No | 初期文字列の長さ | 文字列全体 |
| Capacity | Long | No | 初期容量 | 128 |

### 主要メソッドのパラメータ

| メソッド名 | パラメータ | 型 | 説明 |
|-----------|-----------|-----|------|
| Append | Value | Variant | 追加する値（文字列/整数配列/オブジェクト） |
| AppendChar | Char | Long | 追加する文字コード |
| AppendChar | Count | Long | 追加回数（デフォルト1） |
| AppendFormat | Format | String | 書式文字列 |
| AppendFormat | Args | ParamArray | 書式に埋め込む値 |
| Insert | Index | Long | 挿入位置 |
| Insert | Value | Variant | 挿入する値 |
| Remove | StartIndex | Long | 削除開始位置 |
| Remove | Length | Long | 削除する文字数 |
| Replace | OldValue | String | 置換対象文字列 |
| Replace | NewValue | String | 置換後文字列 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ToString戻り値 | String | 構築された文字列 |
| Length | Long | 現在の文字列長 |
| Capacity | Long | 現在の容量（文字数） |
| Chars | Integer | 指定位置の文字コード |

## 処理フロー

### 処理シーケンス

```
1. インスタンス生成
   └─ 内部バッファ確保（デフォルト128文字）
   └─ 初期文字列がある場合はコピー
2. 文字列追加 (Append系)
   └─ 必要に応じてバッファ拡張（EnsureCapacity）
   └─ 文字列をバッファ末尾にコピー
   └─ Length更新
3. 文字列挿入 (Insert系)
   └─ 必要に応じてバッファ拡張
   └─ 挿入位置以降を後方シフト
   └─ 新文字列を挿入位置にコピー
   └─ Length更新
4. 文字列削除 (Remove)
   └─ 削除位置以降を前方シフト
   └─ Length更新
5. 文字列置換 (Replace)
   └─ マッチ箇所を検索
   └─ 置換後サイズに応じてバッファ処理
   └─ 置換実行
6. 文字列取得 (ToString)
   └─ バッファからString生成・返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{操作種別}
    B -->|Append| C{バッファ足りる?}
    C -->|No| D[EnsureCapacity]
    C -->|Yes| E[末尾にコピー]
    D --> E
    E --> F[Length更新]
    B -->|Insert| G{バッファ足りる?}
    G -->|No| H[EnsureCapacity]
    G -->|Yes| I[後方シフト]
    H --> I
    I --> J[挿入位置にコピー]
    J --> K[Length更新]
    B -->|Remove| L[前方シフト]
    L --> M[Length更新]
    B -->|Replace| N[マッチ検索]
    N --> O{マッチあり?}
    O -->|No| P[変更なし]
    O -->|Yes| Q{サイズ変化?}
    Q -->|等しい| R[上書き]
    Q -->|小さい| S[縮小置換]
    Q -->|大きい| T[拡大置換]
    R --> U[Length更新]
    S --> U
    T --> U
    B -->|ToString| V[バッファからString生成]
    F --> W[終了]
    K --> W
    M --> W
    P --> W
    U --> W
    V --> W
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | バッファ自動拡張 | 容量不足時は現在容量の2倍に拡張（不足なら必要量に拡張） | Append/Insert時 |
| BR-002 | 0ベースインデックス | 全てのインデックスは0ベース | 常時 |
| BR-003 | フルーエントインターフェース | 多くのメソッドはMe（自身）を返し、メソッドチェーンが可能 | Append/Insert/Remove/Replace等 |
| BR-004 | デフォルト容量 | 初期容量は128文字 | 新規作成時 |
| BR-005 | FillCharacter | Length拡張時のパディング文字（デフォルトはスペース） | Length設定時 |
| BR-006 | 書式文字列 | {index[,width][:formatString]}形式をサポート | AppendFormat時 |

### 計算ロジック

```
EnsureCapacity:
  新容量 = Max(RequiredCapacity, 現在容量 * 2)

Replace後のLength:
  新Length = 現在Length - (OldValue長 - NewValue長) * マッチ数
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ArgumentOutOfRangeException | Indexが範囲外 | 有効なインデックスを指定 |
| - | ArgumentOutOfRangeException | Lengthが負の値 | 0以上の値を指定 |
| - | ArgumentOutOfRangeException | Countが負の値 | 0以上の値を指定 |
| - | ArgumentNullException | 必須配列がNull | 配列を初期化してから呼び出し |
| - | FormatException | 書式文字列が不正 | 正しい書式文字列を指定 |
| - | InvalidCastException | FillCharacterに不正な型 | 文字列/数値を指定 |

### リトライ仕様

メモリ内操作であるため、リトライは不要。

## トランザクション仕様

本機能はトランザクション処理を行わない。

## パフォーマンス要件

- 連続Appendは標準文字列連結の数十倍〜数百倍高速
- EnsureCapacityによる事前容量確保でさらに高速化
- 内部バッファはInteger配列（CopyMemoryで高速コピー）

## セキュリティ考慮事項

- メモリ内に機密情報を保持する場合は、使用後にClearを呼び出すことを推奨
- 外部入力を直接AppendFormatに渡す場合は、書式文字列インジェクションに注意

## 備考

- .NET FrameworkのStringBuilderと高い互換性を持つ
- AppendFormatはICustomFormatterによるカスタム書式設定をサポート
- VB6/VBAの内部文字表現（UTF-16）をそのまま使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造とクラス定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | クラス定義、プライベート変数（52-59行目） |

**読解のコツ**:
- 52-53行目: DefaultFillChar（スペース）、DefaultCapacity（128）の定数
- 55-59行目: mString（内部バッファ）、mCapacity（容量）、mLength（長さ）、mFillChar（パディング文字）

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | 公開メソッド（Append、Insert、Remove等） |

**主要処理フロー**:
1. **71-82行目**: Appendメソッド - 値の型に応じて適切なAppendを呼び出し
2. **92-118行目**: AppendCharメソッド - 文字を指定回数追加
3. **129-133行目**: AppendFormatメソッド - 書式付き文字列追加
4. **183-190行目**: AppendLineメソッド - 文字列+改行を追加
5. **197-211行目**: AppendStringメソッド - 高速文字列追加
6. **400-423行目**: Insertメソッド - 文字列挿入
7. **493-507行目**: Removeメソッド - 文字列削除
8. **518-554行目**: Replaceメソッド - 文字列置換
9. **563-586行目**: ToStringメソッド - 文字列取得

#### Step 3: バッファ管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | Capacity、EnsureCapacity |

**主要処理フロー**:
- **245-247行目**: Capacityプロパティ（Get）- 現在の容量取得
- **255-271行目**: Capacityプロパティ（Let）- 容量設定とバッファ再確保
- **310-324行目**: EnsureCapacityメソッド - バッファ拡張（現在の2倍または要求量）

#### Step 4: 書式設定処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | AppendFormatArrayEx（652-775行目） |

**主要処理フロー**:
- 672-761行目: 書式文字列のパース（{index,width:format}形式）
- 948-994行目: ParseArgumentFormatting - 引数書式の解析
- 1029-1067行目: ParseNumber - 数値のパース

### プログラム呼び出し階層図

```
StringBuilder
    │
    ├─ Append(Value)
    │      ├─ vbIntegerArray: AppendCharsValue()
    │      ├─ vbString: AppendStringValue()
    │      └─ その他: Object.ToString() → AppendString()
    │
    ├─ AppendString(s)
    │      ├─ EnsureCapacity(): バッファ拡張
    │      └─ CopyMemory(): 高速コピー
    │
    ├─ AppendFormat(Format, Args)
    │      └─ AppendFormatArrayEx()
    │             ├─ ParseArgumentFormatting(): 書式解析
    │             ├─ ICustomFormatter.Format(): カスタム書式
    │             ├─ Object.ToString(): 値の文字列化
    │             └─ AppendString(): 結果追加
    │
    ├─ Insert(Index, Value)
    │      ├─ InsertSpace(): スペース確保
    │      └─ CopyMemory(): 値コピー
    │
    ├─ Remove(StartIndex, Length)
    │      └─ CopyMemory(): 前方シフト
    │
    ├─ Replace(OldValue, NewValue)
    │      ├─ FindMatches(): マッチ検索
    │      ├─ ReplaceEqualLength(): 同長置換
    │      ├─ ReplaceSmallerLength(): 縮小置換
    │      └─ ReplaceLargerLength(): 拡大置換
    │
    └─ ToString()
           └─ SysAllocStringLen(): String生成
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Value ───────────────────▶ Append ────────────────────▶ mString更新
  │                           │
  ├─ EnsureCapacity           │
  └─ CopyMemory              mLength++

Format, Args ────────────▶ AppendFormat ──────────────▶ mString更新
  │                           │
  ├─ ParseArgumentFormatting  │
  ├─ ICustomFormatter.Format  │
  └─ Object.ToString         mLength++

Index, Value ────────────▶ Insert ────────────────────▶ mString更新
  │                           │
  ├─ InsertSpace (後方シフト)  │
  └─ CopyMemory              mLength++

StartIndex, Length ───────▶ Remove ────────────────────▶ mString更新
  │                           │
  └─ CopyMemory (前方シフト)  mLength--

OldValue, NewValue ───────▶ Replace ───────────────────▶ mString更新
  │                           │
  ├─ FindMatches              │
  └─ ReplaceMatches          mLength更新

（なし） ──────────────────▶ ToString ─────────────────▶ String
                              │
                              └─ SysAllocStringLen
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| StringBuilder.cls | `Source/CorLib/System.Text/StringBuilder.cls` | ソース | メインクラス定義（1216行） |
| Constructors.cls | `Source/CorLib/Constructors.cls` | ソース | NewStringBuilderファクトリメソッド |
| ObjectStatic.cls | `Source/CorLib/ObjectStatic.cls` | ソース | Object.ToString（値の文字列化） |
| IFormatProvider.cls | `Source/CorLib/System/IFormatProvider.cls` | インターフェース | 書式プロバイダ |
| ICustomFormatter.cls | `Source/CorLib/System/ICustomFormatter.cls` | インターフェース | カスタム書式 |
| CultureInfo.cls | `Source/CorLib/System.Globalization/CultureInfo.cls` | ソース | カルチャ情報 |
