# 機能設計書 61-HMACSHA1

## 概要

本ドキュメントは、VBCorLibライブラリにおけるHMACSHA1クラスの機能設計を記述する。HMACSHA1は、SHA-1ハッシュアルゴリズムを使用したHash-based Message Authentication Code（HMAC）の実装を提供するクラスである。

### 本機能の処理概要

HMACSHA1クラスは、メッセージの認証と完全性検証のための暗号学的ハッシュ関数を提供する。秘密鍵とメッセージを組み合わせて固定長のハッシュ値（160ビット/20バイト）を生成し、通信の改竄検知や認証に使用される。

**業務上の目的・背景**：セキュアな通信において、送信されたメッセージが途中で改竄されていないことを検証する必要がある。HMACSHA1は、送信者と受信者が共有する秘密鍵を使用して、メッセージの認証コードを生成することで、この要件を満たす。APIの認証、データの完全性検証、セッショントークンの生成など、様々なセキュリティシナリオで使用される。

**機能の利用シーン**：
- Webサービス間のAPI認証（OAuth署名など）
- メッセージの完全性検証
- セッショントークンの生成
- デジタル署名の作成
- パスワードベースの鍵導出（PBKDF2で内部使用）

**主要な処理内容**：
1. 秘密鍵の設定と内部パディング処理（ipad/opad生成）
2. 入力データのブロック単位での処理（TransformBlock）
3. 最終ブロックの処理とハッシュ値の算出（TransformFinalBlock）
4. バイト配列またはストリームからの一括ハッシュ計算（ComputeHash）
5. ハッシュアルゴリズムの初期化とリセット（Initialize）

**関連システム・外部連携**：
- SHA1CryptoServiceProvider / SHA1Managed: 内部で使用するハッシュアルゴリズム
- HMACBase: HMAC処理の基盤クラス
- CryptoStream: ストリーム暗号化との連携

**権限による制御**：特になし。ただし、Windowsの暗号化サービスプロバイダー（CSP）へのアクセスが必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | テスト実行画面 | テストスイート内のHMACSHA1テストの実行 |

## 機能種別

計算処理 / 暗号化処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Key | Byte() | No | HMAC計算に使用する秘密鍵 | Nullの場合は64バイトのランダム鍵を自動生成 |
| Source | Variant | Yes | ハッシュ計算対象（Byte配列またはStream） | Byte配列またはStreamオブジェクトであること |
| Index | Long | No | Byte配列の開始インデックス | 0以上かつ配列範囲内 |
| Count | Long | No | 処理するバイト数 | 0以上かつ配列範囲内 |
| InputBuffer | Byte() | Yes | TransformBlockの入力データ | 有効なByte配列であること |
| InputOffset | Long | Yes | 入力バッファの開始位置 | 0以上かつ配列範囲内 |
| InputCount | Long | Yes | 処理するバイト数 | 0以上かつ配列範囲内 |
| UseManagedSha1 | Boolean | No | マネージドSHA1実装を使用するか | - |

### 入力データソース

- アプリケーションから直接渡されるバイト配列
- Streamインターフェースを実装したオブジェクト（FileStream、MemoryStreamなど）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Hash | Byte()(20バイト) | 計算されたHMAC-SHA1ハッシュ値（160ビット） |
| HashSize | Long | ハッシュサイズ（常に160ビット） |
| TransformBlock戻り値 | Long | 処理されたバイト数 |
| TransformFinalBlock戻り値 | Byte() | 処理された入力データのコピー |

### 出力先

- 呼び出し元アプリケーションへの戻り値として返却
- OutputBufferパラメータへのデータコピー（TransformBlock使用時）

## 処理フロー

### 処理シーケンス

```
1. インスタンス初期化（Class_Initialize）
   └─ HMACBaseオブジェクトをSHA1CryptoServiceProviderで初期化（ブロックサイズ64バイト）

2. 鍵の設定（Key Letプロパティ / Init）
   └─ 鍵を設定し、必要に応じてパディングまたはハッシュ化

3. ハッシュ計算開始（StartHash - HMACBase内部）
   └─ 鍵をipad（0x36でXOR）とopad（0x5CでXOR）で拡張
   └─ ipadを最初にハッシュ処理

4. データ処理（TransformBlock / ComputeHash）
   └─ 入力データをブロック単位で処理
   └─ 内部SHA1アルゴリズムにデータを渡す

5. 最終処理（TransformFinalBlock / EndHash）
   └─ 残りのデータを処理
   └─ opadと中間ハッシュを結合して最終ハッシュを計算

6. ハッシュ取得（Hash Get）
   └─ 20バイトのハッシュ値を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[インスタンス生成]
    B --> C{鍵が設定済み?}
    C -->|No| D[ランダム鍵生成]
    C -->|Yes| E{鍵長チェック}
    D --> E
    E -->|< 64バイト| F[ゼロパディング]
    E -->|> 64バイト| G[SHA1でハッシュ化]
    E -->|= 64バイト| H[そのまま使用]
    F --> I[ipad/opad生成]
    G --> I
    H --> I
    I --> J[入力データ処理]
    J --> K{全データ処理完了?}
    K -->|No| J
    K -->|Yes| L[最終ブロック処理]
    L --> M[opadと中間ハッシュ結合]
    M --> N[最終ハッシュ計算]
    N --> O[ハッシュ値返却]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-61-01 | 鍵長正規化 | 64バイト未満の鍵はゼロパディング、64バイト超の鍵はSHA1でハッシュ化して64バイトに | 鍵設定時 |
| BR-61-02 | 自動鍵生成 | 鍵が未設定の場合、64バイトのランダム鍵を生成 | GetKeyまたはハッシュ計算開始時 |
| BR-61-03 | ハッシュ開始後の鍵変更禁止 | ハッシュ計算開始後は鍵を変更できない | Key設定時にmHashStartedがTrueの場合 |
| BR-61-04 | 再利用可能 | CanReuseTransformは常にTrue | - |

### 計算ロジック

**HMAC-SHA1アルゴリズム（RFC 2104準拠）**:
```
HMAC(K, M) = SHA1((K XOR opad) || SHA1((K XOR ipad) || M))

ここで:
- K: 秘密鍵（64バイトに正規化）
- M: メッセージ
- ipad: 0x36を64回繰り返したバイト列
- opad: 0x5Cを64回繰り返したバイト列
- ||: 連結演算子
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| CryptographicException | 暗号例外 | ハッシュ開始後に鍵を変更しようとした場合 | Initializeを呼び出してリセット後に鍵を設定 |
| ArgumentNullException | 引数Null例外 | 鍵にNull配列を設定した場合 | 有効なByte配列を渡す |
| InvalidOperationException | 無効操作例外 | TransformFinalBlock前にHashプロパティにアクセス | TransformFinalBlockを先に呼び出す |
| ObjectDisposedException | 破棄済み例外 | Clear後にメソッドを呼び出した場合 | 新しいインスタンスを作成 |

### リトライ仕様

エラー発生時のリトライは行わない。エラーは即座に呼び出し元に伝播する。

## トランザクション仕様

本機能はトランザクション管理を行わない（メモリ内処理のみ）。

## パフォーマンス要件

- 大量データ処理時はストリーム入力を使用することを推奨
- CanTransformMultipleBlocksがTrueのため、複数ブロックの一括処理が可能
- 鍵の再設定を避け、Initializeでリセットすることでパフォーマンス向上

## セキュリティ考慮事項

- 秘密鍵は安全に管理し、ログ出力や永続化を避ける
- SHA-1は衝突攻撃に対して脆弱性があるため、新規開発ではHMACSHA256以上を推奨
- 鍵長は最低でも160ビット（20バイト）以上を推奨
- Clear()メソッドで使用後に鍵情報を消去する

## 備考

- .NET FrameworkのSystem.Security.Cryptography.HMACSHA1クラスと互換性のあるAPI設計
- FIPS 198-1（HMAC）およびFIPS 180-4（SHA-1）に準拠
- UseManagedSha1オプションにより、マネージドまたはCSP実装を選択可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、HMACの基本構造とインターフェースを理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | HMAC.cls | `Source/CorLib/System.Security.Cryptography/HMAC.cls` | HMACインターフェースの定義、必要なプロパティ・メソッドの確認 |
| 1-2 | HashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/HashAlgorithm.cls` | ハッシュアルゴリズムの基本インターフェース |
| 1-3 | ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | 暗号変換インターフェース |

**読解のコツ**: VB6のクラスモジュールでは`Implements`キーワードでインターフェースを実装する。インターフェースメソッドは`InterfaceName_MethodName`の形式で実装される。

#### Step 2: エントリーポイントを理解する

HMACSHA1クラスの初期化と主要メソッドを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HMACSHA1.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA1.cls` | メインクラスの構造と公開API |

**主要処理フロー**:
1. **247-249行目**: Class_Initialize - HMACBaseをSHA1CryptoServiceProviderで初期化、ブロックサイズ64バイト
2. **251-257行目**: Init - 鍵とマネージド実装オプションの初期化
3. **166-168行目**: ComputeHash - 一括ハッシュ計算の委譲
4. **190-192行目**: TransformBlock - ブロック単位処理の委譲
5. **204-206行目**: TransformFinalBlock - 最終ブロック処理の委譲
6. **117-119行目**: Hash Get - 計算済みハッシュの取得

#### Step 3: 基盤クラスを理解する

HMACの実際の処理ロジックはHMACBaseクラスに実装されている。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | HMAC処理の核心部分 |

**主要処理フロー**:
- **99-110行目**: SetKey - 鍵の設定とバリデーション
- **245-251行目**: StartHash - ipadの処理開始
- **239-243行目**: EndHash - opadと中間ハッシュの結合
- **253-281行目**: ExpandKey - 鍵の正規化（パディング/ハッシュ化）とipad/opad生成

#### Step 4: 依存するハッシュアルゴリズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | SHA1CryptoServiceProvider.cls | `Source/CorLib/System.Security.Cryptography/SHA1CryptoServiceProvider.cls` | CSPベースのSHA1実装 |
| 4-2 | SHA1Managed.cls | `Source/CorLib/System.Security.Cryptography/SHA1Managed.cls` | マネージドSHA1実装 |

### プログラム呼び出し階層図

```
HMACSHA1
    │
    ├─ HMACBase
    │      ├─ SetKey() → 鍵のバリデーションと保存
    │      ├─ ExpandKey() → ipad/opad生成
    │      │      └─ HashAlgorithm.ComputeHash() [鍵が64バイト超の場合]
    │      ├─ StartHash() → ipadをハッシュ処理
    │      │      └─ HashAlgorithm.TransformBlock()
    │      ├─ ComputeHash() → データのハッシュ計算
    │      │      └─ HashAlgorithm.ComputeHash()
    │      ├─ TransformBlock() → ブロック処理
    │      │      └─ HashAlgorithm.TransformBlock()
    │      ├─ TransformFinalBlock() → 最終処理
    │      │      └─ HashAlgorithm.TransformFinalBlock()
    │      └─ EndHash() → opadと中間ハッシュ結合
    │             ├─ HashAlgorithm.TransformBlock()
    │             └─ HashAlgorithm.TransformFinalBlock()
    │
    └─ SHA1CryptoServiceProvider / SHA1Managed
           └─ CryptoAPI (Windows暗号化サービス)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

秘密鍵(Key) ──────────────┐
                          ▼
                    ┌─────────────┐
                    │ ExpandKey   │
                    │ ・パディング │
                    │ ・ipad生成  │
                    │ ・opad生成  │
                    └─────────────┘
                          │
                          ▼
メッセージ ──────────▶ ┌─────────────┐
(Source)              │ StartHash   │
                      │ SHA1(ipad)  │
                      └─────────────┘
                          │
                          ▼
                    ┌─────────────┐
                    │TransformBloc│
                    │ SHA1(data)  │
                    └─────────────┘
                          │
                          ▼
                    ┌─────────────┐
                    │ EndHash     │
                    │ SHA1(opad|| │ ──────▶ HMAC値(20バイト)
                    │  中間Hash)  │
                    └─────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| HMACSHA1.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA1.cls` | ソース | メインクラス |
| HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | ソース | HMAC処理基盤 |
| HMAC.cls | `Source/CorLib/System.Security.Cryptography/HMAC.cls` | ソース | HMACインターフェース |
| HashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/HashAlgorithm.cls` | ソース | ハッシュアルゴリズムインターフェース |
| KeyedHashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/KeyedHashAlgorithm.cls` | ソース | 鍵付きハッシュインターフェース |
| ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | ソース | 暗号変換インターフェース |
| SHA1CryptoServiceProvider.cls | `Source/CorLib/System.Security.Cryptography/SHA1CryptoServiceProvider.cls` | ソース | CSPベースSHA1実装 |
| SHA1Managed.cls | `Source/CorLib/System.Security.Cryptography/SHA1Managed.cls` | ソース | マネージドSHA1実装 |
| CryptoHelper.cls | `Source/CorLib/System.Security.Cryptography/CryptoHelper.cls` | ソース | 暗号処理ヘルパー |
