# 機能設計書 62-HMACSHA256

## 概要

本ドキュメントは、VBCorLibライブラリにおけるHMACSHA256クラスの機能設計を記述する。HMACSHA256は、SHA-256ハッシュアルゴリズムを使用したHash-based Message Authentication Code（HMAC）の実装を提供するクラスである。

### 本機能の処理概要

HMACSHA256クラスは、メッセージの認証と完全性検証のための暗号学的ハッシュ関数を提供する。秘密鍵とメッセージを組み合わせて固定長のハッシュ値（256ビット/32バイト）を生成し、通信の改竄検知や認証に使用される。SHA-1より強力なセキュリティを提供する。

**業務上の目的・背景**：現代のセキュリティ要件では、SHA-1の脆弱性を考慮し、より強力なハッシュアルゴリズムが求められる。HMACSHA256は、256ビットのハッシュ長を持ち、現在の暗号学的標準で推奨されるセキュリティレベルを提供する。JWT（JSON Web Token）の署名、APIの認証、データの完全性検証など、幅広いセキュリティシナリオで使用される。

**機能の利用シーン**：
- JWT（JSON Web Token）のHS256署名
- RESTful API認証（AWS Signature Version 4など）
- メッセージの完全性検証
- セキュアなセッション管理
- キー導出関数（KDF）の基盤

**主要な処理内容**：
1. 秘密鍵の設定と内部パディング処理（ipad/opad生成）
2. 入力データのブロック単位での処理（TransformBlock）
3. 最終ブロックの処理とハッシュ値の算出（TransformFinalBlock）
4. バイト配列またはストリームからの一括ハッシュ計算（ComputeHash）
5. ハッシュアルゴリズムの初期化とリセット（Initialize）

**関連システム・外部連携**：
- SHA256Managed: 内部で使用するハッシュアルゴリズム（マネージド実装）
- HMACBase: HMAC処理の基盤クラス
- CryptoStream: ストリーム暗号化との連携

**権限による制御**：特になし。マネージド実装のみ使用するため、CSPへのアクセスは不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに記載なし |

## 機能種別

計算処理 / 暗号化処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Key | Byte() | No | HMAC計算に使用する秘密鍵 | Nullの場合は64バイトのランダム鍵を自動生成 |
| Source | Variant | Yes | ハッシュ計算対象（Byte配列またはStream） | Byte配列またはStreamオブジェクトであること |
| Index | Long | No | Byte配列の開始インデックス | 0以上かつ配列範囲内 |
| Count | Long | No | 処理するバイト数 | 0以上かつ配列範囲内 |
| InputBuffer | Byte() | Yes | TransformBlockの入力データ | 有効なByte配列であること |
| InputOffset | Long | Yes | 入力バッファの開始位置 | 0以上かつ配列範囲内 |
| InputCount | Long | Yes | 処理するバイト数 | 0以上かつ配列範囲内 |

### 入力データソース

- アプリケーションから直接渡されるバイト配列
- Streamインターフェースを実装したオブジェクト（FileStream、MemoryStreamなど）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Hash | Byte()(32バイト) | 計算されたHMAC-SHA256ハッシュ値（256ビット） |
| HashSize | Long | ハッシュサイズ（常に256ビット） |
| TransformBlock戻り値 | Long | 処理されたバイト数 |
| TransformFinalBlock戻り値 | Byte() | 処理された入力データのコピー |

### 出力先

- 呼び出し元アプリケーションへの戻り値として返却
- OutputBufferパラメータへのデータコピー（TransformBlock使用時）

## 処理フロー

### 処理シーケンス

```
1. インスタンス初期化（Class_Initialize）
   └─ HMACBaseオブジェクトをSHA256Managedで初期化（ブロックサイズ64バイト）

2. 鍵の設定（Key Letプロパティ / Init）
   └─ 鍵を設定し、必要に応じてパディングまたはハッシュ化

3. ハッシュ計算開始（StartHash - HMACBase内部）
   └─ 鍵をipad（0x36でXOR）とopad（0x5CでXOR）で拡張
   └─ ipadを最初にハッシュ処理

4. データ処理（TransformBlock / ComputeHash）
   └─ 入力データをブロック単位で処理
   └─ 内部SHA256アルゴリズムにデータを渡す

5. 最終処理（TransformFinalBlock / EndHash）
   └─ 残りのデータを処理
   └─ opadと中間ハッシュを結合して最終ハッシュを計算

6. ハッシュ取得（Hash Get）
   └─ 32バイトのハッシュ値を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[インスタンス生成]
    B --> C{鍵が設定済み?}
    C -->|No| D[ランダム鍵生成]
    C -->|Yes| E{鍵長チェック}
    D --> E
    E -->|< 64バイト| F[ゼロパディング]
    E -->|> 64バイト| G[SHA256でハッシュ化]
    E -->|= 64バイト| H[そのまま使用]
    F --> I[ipad/opad生成]
    G --> I
    H --> I
    I --> J[入力データ処理]
    J --> K{全データ処理完了?}
    K -->|No| J
    K -->|Yes| L[最終ブロック処理]
    L --> M[opadと中間ハッシュ結合]
    M --> N[最終ハッシュ計算]
    N --> O[ハッシュ値返却]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-62-01 | 鍵長正規化 | 64バイト未満の鍵はゼロパディング、64バイト超の鍵はSHA256でハッシュ化して64バイトに | 鍵設定時 |
| BR-62-02 | 自動鍵生成 | 鍵が未設定の場合、64バイトのランダム鍵を生成 | GetKeyまたはハッシュ計算開始時 |
| BR-62-03 | ハッシュ開始後の鍵変更禁止 | ハッシュ計算開始後は鍵を変更できない | Key設定時にmHashStartedがTrueの場合 |
| BR-62-04 | 再利用可能 | CanReuseTransformは常にTrue | - |

### 計算ロジック

**HMAC-SHA256アルゴリズム（RFC 2104準拠）**:
```
HMAC(K, M) = SHA256((K XOR opad) || SHA256((K XOR ipad) || M))

ここで:
- K: 秘密鍵（64バイトに正規化）
- M: メッセージ
- ipad: 0x36を64回繰り返したバイト列
- opad: 0x5Cを64回繰り返したバイト列
- ||: 連結演算子
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| CryptographicException | 暗号例外 | ハッシュ開始後に鍵を変更しようとした場合 | Initializeを呼び出してリセット後に鍵を設定 |
| ArgumentNullException | 引数Null例外 | 鍵にNull配列を設定した場合 | 有効なByte配列を渡す |
| InvalidOperationException | 無効操作例外 | TransformFinalBlock前にHashプロパティにアクセス | TransformFinalBlockを先に呼び出す |
| ObjectDisposedException | 破棄済み例外 | Clear後にメソッドを呼び出した場合 | 新しいインスタンスを作成 |

### リトライ仕様

エラー発生時のリトライは行わない。エラーは即座に呼び出し元に伝播する。

## トランザクション仕様

本機能はトランザクション管理を行わない（メモリ内処理のみ）。

## パフォーマンス要件

- 大量データ処理時はストリーム入力を使用することを推奨
- CanTransformMultipleBlocksがTrueのため、複数ブロックの一括処理が可能
- 鍵の再設定を避け、Initializeでリセットすることでパフォーマンス向上
- SHA256ManagedはマネージドコードのためCSPベースより若干遅い可能性がある

## セキュリティ考慮事項

- 秘密鍵は安全に管理し、ログ出力や永続化を避ける
- SHA-256は現在の暗号学的標準で推奨されるセキュリティレベル
- 鍵長は最低でも256ビット（32バイト）以上を推奨
- Clear()メソッドで使用後に鍵情報を消去する
- タイミング攻撃を防ぐため、ハッシュ比較には定数時間比較を使用すること

## 備考

- .NET FrameworkのSystem.Security.Cryptography.HMACSHA256クラスと互換性のあるAPI設計
- FIPS 198-1（HMAC）およびFIPS 180-4（SHA-256）に準拠
- マネージド実装のみ提供（SHA256Managedを使用）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

HMACSHA1と同様の構造だが、SHA256を使用する点が異なる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | HMAC.cls | `Source/CorLib/System.Security.Cryptography/HMAC.cls` | HMACインターフェースの定義 |
| 1-2 | HashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/HashAlgorithm.cls` | ハッシュアルゴリズムの基本インターフェース |

**読解のコツ**: HMACSHA256はHMACSHA1とほぼ同じ構造を持ち、主な違いはハッシュサイズとコンストラクタで使用するハッシュアルゴリズムのみ。

#### Step 2: エントリーポイントを理解する

HMACSHA256クラスの初期化と主要メソッドを確認。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HMACSHA256.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA256.cls` | メインクラスの構造と公開API |

**主要処理フロー**:
1. **247-249行目**: Class_Initialize - HMACBaseをSHA256Managedで初期化、ブロックサイズ64バイト
2. **251-253行目**: Init - 鍵の初期化（HMACSHA1より簡潔）
3. **126-128行目**: HashSize = 256 - SHA256の出力サイズ
4. **166-168行目**: ComputeHash - 一括ハッシュ計算の委譲

#### Step 3: 基盤クラスを理解する

HMACの実際の処理ロジックはHMACBaseクラスに実装されている（HMACSHA1と共通）。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | HMAC処理の核心部分 |

#### Step 4: 依存するハッシュアルゴリズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | SHA256Managed.cls | `Source/CorLib/System.Security.Cryptography/SHA256Managed.cls` | マネージドSHA256実装 |

### プログラム呼び出し階層図

```
HMACSHA256
    │
    ├─ HMACBase
    │      ├─ SetKey() → 鍵のバリデーションと保存
    │      ├─ ExpandKey() → ipad/opad生成
    │      │      └─ HashAlgorithm.ComputeHash() [鍵が64バイト超の場合]
    │      ├─ StartHash() → ipadをハッシュ処理
    │      │      └─ HashAlgorithm.TransformBlock()
    │      ├─ ComputeHash() → データのハッシュ計算
    │      │      └─ HashAlgorithm.ComputeHash()
    │      ├─ TransformBlock() → ブロック処理
    │      │      └─ HashAlgorithm.TransformBlock()
    │      ├─ TransformFinalBlock() → 最終処理
    │      │      └─ HashAlgorithm.TransformFinalBlock()
    │      └─ EndHash() → opadと中間ハッシュ結合
    │             ├─ HashAlgorithm.TransformBlock()
    │             └─ HashAlgorithm.TransformFinalBlock()
    │
    └─ SHA256Managed
           └─ マネージドSHA-256実装
```

### データフロー図

```
[入力]                      [処理]                           [出力]

秘密鍵(Key) ──────────────┐
                          ▼
                    ┌─────────────┐
                    │ ExpandKey   │
                    │ ・パディング │
                    │ ・ipad生成  │
                    │ ・opad生成  │
                    └─────────────┘
                          │
                          ▼
メッセージ ──────────▶ ┌─────────────┐
(Source)              │ StartHash   │
                      │SHA256(ipad) │
                      └─────────────┘
                          │
                          ▼
                    ┌─────────────┐
                    │TransformBloc│
                    │ SHA256(data)│
                    └─────────────┘
                          │
                          ▼
                    ┌─────────────┐
                    │ EndHash     │
                    │SHA256(opad||│ ──────▶ HMAC値(32バイト)
                    │  中間Hash)  │
                    └─────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| HMACSHA256.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA256.cls` | ソース | メインクラス |
| HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | ソース | HMAC処理基盤 |
| HMAC.cls | `Source/CorLib/System.Security.Cryptography/HMAC.cls` | ソース | HMACインターフェース |
| HashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/HashAlgorithm.cls` | ソース | ハッシュアルゴリズムインターフェース |
| KeyedHashAlgorithm.cls | `Source/CorLib/System.Security.Cryptography/KeyedHashAlgorithm.cls` | ソース | 鍵付きハッシュインターフェース |
| ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | ソース | 暗号変換インターフェース |
| SHA256Managed.cls | `Source/CorLib/System.Security.Cryptography/SHA256Managed.cls` | ソース | マネージドSHA256実装 |
| CryptoHelper.cls | `Source/CorLib/System.Security.Cryptography/CryptoHelper.cls` | ソース | 暗号処理ヘルパー |
