# 機能設計書 63-HMACSHA384

## 概要

本ドキュメントは、VBCorLibライブラリにおけるHMACSHA384クラスの機能設計を記述する。HMACSHA384は、SHA-384ハッシュアルゴリズムを使用したHash-based Message Authentication Code（HMAC）の実装を提供するクラスである。

### 本機能の処理概要

HMACSHA384クラスは、メッセージの認証と完全性検証のための暗号学的ハッシュ関数を提供する。秘密鍵とメッセージを組み合わせて固定長のハッシュ値（384ビット/48バイト）を生成し、通信の改竄検知や認証に使用される。SHA-512のトランケート版として、高いセキュリティレベルを提供する。

**業務上の目的・背景**：高度なセキュリティが要求されるアプリケーションでは、より長いハッシュ長を持つアルゴリズムが必要となる。HMACSHA384は、384ビットのハッシュ長を持ち、SHA-256より高いセキュリティマージンを提供する。特に、長期保存データの完全性検証や、政府・金融機関向けのセキュリティ要件を満たすために使用される。

**機能の利用シーン**：
- 高セキュリティ要件のAPI認証
- 長期保存データの完全性検証
- 政府・金融機関向けアプリケーション
- JWT（JSON Web Token）のHS384署名
- TLS/SSLハンドシェイクでの使用

**主要な処理内容**：
1. 秘密鍵の設定と内部パディング処理（ipad/opad生成）
2. 入力データのブロック単位での処理（TransformBlock）
3. 最終ブロックの処理とハッシュ値の算出（TransformFinalBlock）
4. バイト配列またはストリームからの一括ハッシュ計算（ComputeHash）
5. レガシーHMAC値生成モードのサポート（ProduceLegacyHmacValues）

**関連システム・外部連携**：
- SHA384Managed: 内部で使用するハッシュアルゴリズム（マネージド実装）
- HMACBase: HMAC処理の基盤クラス
- CryptoStream: ストリーム暗号化との連携

**権限による制御**：特になし。マネージド実装のみ使用するため、CSPへのアクセスは不要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに記載なし |

## 機能種別

計算処理 / 暗号化処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Key | Byte() | No | HMAC計算に使用する秘密鍵 | Nullの場合は128バイトのランダム鍵を自動生成 |
| ProduceLegacyHmacValues | Boolean | No | レガシーモード（ブロックサイズ64バイト）を使用するか | デフォルトはFalse |
| Source | Variant | Yes | ハッシュ計算対象（Byte配列またはStream） | Byte配列またはStreamオブジェクトであること |
| Index | Long | No | Byte配列の開始インデックス | 0以上かつ配列範囲内 |
| Count | Long | No | 処理するバイト数 | 0以上かつ配列範囲内 |

### 入力データソース

- アプリケーションから直接渡されるバイト配列
- Streamインターフェースを実装したオブジェクト（FileStream、MemoryStreamなど）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Hash | Byte()(48バイト) | 計算されたHMAC-SHA384ハッシュ値（384ビット） |
| HashSize | Long | ハッシュサイズ（常に384ビット） |
| TransformBlock戻り値 | Long | 処理されたバイト数 |
| TransformFinalBlock戻り値 | Byte() | 処理された入力データのコピー |

### 出力先

- 呼び出し元アプリケーションへの戻り値として返却
- OutputBufferパラメータへのデータコピー（TransformBlock使用時）

## 処理フロー

### 処理シーケンス

```
1. インスタンス初期化（Class_Initialize）
   └─ InitBaseを呼び出し、HMACBaseをSHA384Managedで初期化
   └─ ブロックサイズはProduceLegacyHmacValuesにより64または128バイト

2. レガシーモード設定（ProduceLegacyHmacValues Let）
   └─ 値が変更された場合、InitBaseを再実行

3. 鍵の設定（Key Letプロパティ / Init）
   └─ 鍵を設定し、必要に応じてパディングまたはハッシュ化

4. ハッシュ計算開始（StartHash - HMACBase内部）
   └─ 鍵をipad（0x36でXOR）とopad（0x5CでXOR）で拡張
   └─ ipadを最初にハッシュ処理

5. データ処理（TransformBlock / ComputeHash）
   └─ 入力データをブロック単位で処理
   └─ 内部SHA384アルゴリズムにデータを渡す

6. 最終処理（TransformFinalBlock / EndHash）
   └─ 残りのデータを処理
   └─ opadと中間ハッシュを結合して最終ハッシュを計算

7. ハッシュ取得（Hash Get）
   └─ 48バイトのハッシュ値を返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[インスタンス生成]
    B --> C{ProduceLegacyHmacValues?}
    C -->|True| D[ブロックサイズ=64]
    C -->|False| E[ブロックサイズ=128]
    D --> F[HMACBase初期化]
    E --> F
    F --> G{鍵が設定済み?}
    G -->|No| H[ランダム鍵生成]
    G -->|Yes| I{鍵長チェック}
    H --> I
    I -->|< ブロックサイズ| J[ゼロパディング]
    I -->|> ブロックサイズ| K[SHA384でハッシュ化]
    I -->|= ブロックサイズ| L[そのまま使用]
    J --> M[ipad/opad生成]
    K --> M
    L --> M
    M --> N[入力データ処理]
    N --> O{全データ処理完了?}
    O -->|No| N
    O -->|Yes| P[最終ブロック処理]
    P --> Q[opadと中間ハッシュ結合]
    Q --> R[最終ハッシュ計算]
    R --> S[ハッシュ値返却]
    S --> T[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-63-01 | 鍵長正規化（標準） | 128バイト未満の鍵はゼロパディング、128バイト超の鍵はSHA384でハッシュ化 | ProduceLegacyHmacValues = False |
| BR-63-02 | 鍵長正規化（レガシー） | 64バイト未満の鍵はゼロパディング、64バイト超の鍵はSHA384でハッシュ化 | ProduceLegacyHmacValues = True |
| BR-63-03 | レガシーモード切替 | ProduceLegacyHmacValues変更時はHMACBaseを再初期化 | プロパティ変更時 |
| BR-63-04 | 再利用可能 | CanReuseTransformは常にTrue | - |

### 計算ロジック

**HMAC-SHA384アルゴリズム（RFC 2104準拠）**:
```
HMAC(K, M) = SHA384((K XOR opad) || SHA384((K XOR ipad) || M))

ここで:
- K: 秘密鍵（128バイトに正規化、レガシーモードでは64バイト）
- M: メッセージ
- ipad: 0x36をブロックサイズ分繰り返したバイト列
- opad: 0x5Cをブロックサイズ分繰り返したバイト列
- ||: 連結演算子
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| CryptographicException | 暗号例外 | ハッシュ開始後に鍵を変更しようとした場合 | Initializeを呼び出してリセット後に鍵を設定 |
| ArgumentNullException | 引数Null例外 | 鍵にNull配列を設定した場合 | 有効なByte配列を渡す |
| InvalidOperationException | 無効操作例外 | TransformFinalBlock前にHashプロパティにアクセス | TransformFinalBlockを先に呼び出す |
| ObjectDisposedException | 破棄済み例外 | Clear後にメソッドを呼び出した場合 | 新しいインスタンスを作成 |

### リトライ仕様

エラー発生時のリトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない（メモリ内処理のみ）。

## パフォーマンス要件

- SHA384はSHA512のトランケート版のため、SHA512と同等の計算コスト
- 大量データ処理時はストリーム入力を使用することを推奨
- 標準モード（128バイトブロック）はレガシーモード（64バイトブロック）より若干効率的

## セキュリティ考慮事項

- 秘密鍵は安全に管理し、ログ出力や永続化を避ける
- SHA-384は現在の暗号学的標準で高いセキュリティレベルを提供
- 新規開発ではProduceLegacyHmacValues = False（デフォルト）を推奨
- 鍵長は最低でも256ビット（32バイト）以上を推奨
- Clear()メソッドで使用後に鍵情報を消去する

## 備考

- .NET FrameworkのSystem.Security.Cryptography.HMACSHA384クラスと互換性のあるAPI設計
- ProduceLegacyHmacValuesは.NET Framework 2.0以前との互換性のために提供
- FIPS 198-1（HMAC）およびFIPS 180-4（SHA-384）に準拠

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: HMACSHA384固有の特徴を理解する

HMACSHA384はProduceLegacyHmacValuesプロパティを持つ点が他のHMACクラスと異なる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | HMACSHA384.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA384.cls` | メインクラス、ProduceLegacyHmacValuesの動作 |

**主要処理フロー**:
1. **54行目**: mProduceLegacyHmacValues - レガシーモードフラグ
2. **57-66行目**: ProduceLegacyHmacValues Get/Let - モード切替時にInitBase再実行
3. **271-279行目**: InitBase - ブロックサイズ決定ロジック（64 or 128）
4. **263-266行目**: Class_Initialize - InitBaseを呼び出し

#### Step 2: ブロックサイズの違いを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | KeySizeパラメータの使用方法 |

**読解のコツ**: HMACBase.Init()のKeySizeパラメータがブロックサイズとして使用され、ipad/opadのサイズとして使われる。

### プログラム呼び出し階層図

```
HMACSHA384
    │
    ├─ InitBase()
    │      └─ HMACBase.Init(SHA384Managed, KeySize, "SHA384")
    │             └─ KeySize = 64 (Legacy) or 128 (Standard)
    │
    ├─ ProduceLegacyHmacValues Let
    │      └─ 値変更時 → InitBase() 再実行
    │
    └─ HMACBase
           ├─ ExpandKey() → ブロックサイズに基づくipad/opad生成
           └─ (以下、他のHMACクラスと同様)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

ProduceLegacyHmacValues
        │
        ▼
  ┌─────────────┐
  │ InitBase    │
  │ KeySize決定 │
  │  64 or 128  │
  └─────────────┘
        │
        ▼
秘密鍵(Key) ──────────▶ ExpandKey
                       ・KeySizeにパディング
                       ・ipad/opad生成
        │
        ▼
メッセージ ──────────▶ HMAC計算 ──────▶ HMAC値(48バイト)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| HMACSHA384.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA384.cls` | ソース | メインクラス |
| HMACBase.cls | `Source/CorLib/System.Security.Cryptography/HMACBase.cls` | ソース | HMAC処理基盤 |
| SHA384Managed.cls | `Source/CorLib/System.Security.Cryptography/SHA384Managed.cls` | ソース | マネージドSHA384実装 |
| HMAC.cls | `Source/CorLib/System.Security.Cryptography/HMAC.cls` | ソース | HMACインターフェース |
