# 機能設計書 69-Rfc2898DeriveBytes

## 概要

本ドキュメントは、VBCorLibライブラリにおけるRfc2898DeriveBytesクラスの機能設計を記述する。Rfc2898DeriveBytesは、RFC 2898で定義されたPBKDF2（Password-Based Key Derivation Function 2）アルゴリズムを実装し、パスワードから暗号鍵を導出する機能を提供するクラスである。

### 本機能の処理概要

Rfc2898DeriveBytesクラスは、パスワードとソルトを入力として、指定された反復回数のHMAC-SHA1演算を行い、任意の長さの擬似乱数バイト列（暗号鍵）を生成する。これにより、人間が記憶可能なパスワードから、暗号学的に安全な鍵を導出できる。

**業務上の目的・背景**：パスワードベースの暗号化では、ユーザーが覚えやすいパスワードから強力な暗号鍵を生成する必要がある。PBKDF2は、辞書攻撃やブルートフォース攻撃への耐性を持つ鍵導出関数として広く使用されている。

**機能の利用シーン**：
- パスワードからの暗号鍵生成（AES鍵など）
- パスワードハッシュの生成（認証用）
- マスター鍵からの派生鍵生成
- セキュアなキーストレッチング
- WPA2-PSK等の鍵導出

**主要な処理内容**：
1. パスワードとソルトでHMACSHA1を初期化
2. 指定された反復回数でHMAC演算を繰り返し
3. 任意の長さの鍵バイト列を生成（GetBytes）
4. 状態のリセット（Reset）

**関連システム・外部連携**：
- HMACSHA1: 内部で使用するHMACアルゴリズム
- Encoding.UTF8: 文字列パスワードのエンコーディング
- CryptoHelper: ソルト生成用乱数

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに記載なし |

## 機能種別

計算処理 / 暗号化処理 / 鍵導出

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Password | Variant | Yes | パスワード（String または Byte配列） | StringまたはByte配列であること |
| Salt | Variant | Yes | ソルト（Byte配列 または サイズ指定の整数） | Byte配列は8バイト以上、整数は0以上 |
| Iterations | Long | No | 反復回数 | 1以上（デフォルトは未指定時のみ） |
| cb | Long | Yes | GetBytesで取得するバイト数 | 0以上 |

### 入力データソース

- アプリケーションから提供されるパスワード文字列またはバイト配列
- ソルト値（バイト配列または自動生成サイズ指定）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| GetBytes戻り値 | Byte() | 指定バイト数の導出鍵 |
| IterationCount | Long | 現在の反復回数 |
| Salt | Byte() | 現在のソルト値 |

### 出力先

- GetBytesの戻り値として導出鍵を返却
- プロパティとして反復回数とソルトを返却

## 処理フロー

### 処理シーケンス

```
1. インスタンス初期化（Init）
   └─ パスワードでHMACSHA1を初期化（UTF8エンコードまたはByte配列）
   └─ ソルトを設定（Byte配列または指定サイズで自動生成）
   └─ 反復回数を設定

2. 鍵導出（GetBytes）
   └─ 必要なブロック数を計算
   └─ 各ブロックについて:
       └─ NextBlockBytesでPBKDF2ブロック計算
       └─ ソルト + カウンタをHMAC処理
       └─ 反復回数分HMAC演算を繰り返し
       └─ 各反復結果をXORで累積
   └─ 余剰バイトをバッファに保持（次回呼び出し用）

3. 状態リセット（Reset）
   └─ HMACSHA1をInitialize
   └─ ブロックカウンタをリセット
   └─ バッファをクリア
```

### フローチャート

```mermaid
flowchart TD
    A[Init] --> B[パスワードでHMACSHA1作成]
    B --> C[ソルト設定]
    C --> D[反復回数設定]
    D --> E[GetBytes呼び出し]
    E --> F{バッファに残りあり?}
    F -->|Yes| G[バッファから取得]
    F -->|No| H[新規ブロック計算]
    G --> I{必要量取得済み?}
    H --> J[NextBlockBytes]
    J --> K[Salt + Counter ハッシュ]
    K --> L[反復HMAC演算]
    L --> M[XOR累積]
    M --> I
    I -->|No| H
    I -->|Yes| N[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-69-01 | ソルト最小長 | ソルトは8バイト以上必要 | Salt設定時 |
| BR-69-02 | 反復回数下限 | 反復回数は1以上 | IterationCount設定時 |
| BR-69-03 | 状態自動リセット | ソルトまたは反復回数変更時は自動リセット | プロパティ変更時 |
| BR-69-04 | UTF8エンコーディング | 文字列パスワードはUTF-8でエンコード | Init時にString渡し |
| BR-69-05 | ブロックサイズ | 1ブロック = 20バイト（HMACSHA1出力） | 常時 |

### 計算ロジック

**PBKDF2アルゴリズム（RFC 2898準拠）**:
```
DK = T1 || T2 || ... || Tn

Ti = F(Password, Salt, Iterations, i)

F(Password, Salt, c, i) = U1 XOR U2 XOR ... XOR Uc

U1 = PRF(Password, Salt || INT(i))
U2 = PRF(Password, U1)
...
Uc = PRF(Password, Uc-1)

ここで:
- PRF: HMAC-SHA1
- c: 反復回数（Iterations）
- i: ブロック番号（1から開始）
- INT(i): iのビッグエンディアン4バイト表現
- ||: 連結演算子
- XOR: 排他的論理和
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentOutOfRangeException | 引数範囲外例外 | IterationCountが1未満 | 1以上の値を設定 |
| ArgumentOutOfRangeException | 引数範囲外例外 | cbが0未満 | 0以上の値を指定 |
| ArgumentOutOfRangeException | 引数範囲外例外 | Saltサイズ指定が負数 | 0以上の値を指定 |
| ArgumentException | 引数例外 | Saltが8バイト未満 | 8バイト以上のソルトを設定 |
| ArgumentNullException | 引数Null例外 | Saltが未初期化配列 | 有効なByte配列を設定 |
| ArgumentException | 引数例外 | パスワードがStringでもByte配列でもない | 有効な型を指定 |

### リトライ仕様

エラー発生時のリトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- 反復回数が多いほど処理時間が増加（セキュリティとのトレードオフ）
- 推奨反復回数: 10,000以上（2024年時点）
- GetBytesは連続呼び出しをサポート（状態を保持）
- 大量のバイト数が必要な場合は一度に取得することを推奨

## セキュリティ考慮事項

- 反復回数は十分に大きな値（10,000以上）を推奨
- ソルトは各パスワードに対してユニークであるべき
- ソルトは暗号論的に安全な乱数で生成すること
- パスワードと導出鍵はログに出力しない
- HMACSHA1はSHA-1ベースだが、PBKDF2での使用は現在も安全と考えられている
- より高いセキュリティにはArgon2やscryptの検討を推奨

## 備考

- .NET FrameworkのSystem.Security.Cryptography.Rfc2898DeriveBytesクラスと互換性のあるAPI設計
- RFC 2898（PKCS #5 v2.0）のPBKDF2に準拠
- HMACSHA1を内部で使用（20バイトブロック）
- 連続したGetBytes呼び出しで連続した鍵材料を取得可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: クラスの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Rfc2898DeriveBytes.cls | `Source/CorLib/System.Security.Cryptography/Rfc2898DeriveBytes.cls` | メインクラスの構造 |

**主要処理フロー**:
1. **46行目**: BlockSize = 20 - HMACSHA1の出力サイズ
2. **48-55行目**: メンバ変数 - mHasher, mIterations, mSalt, mBlockCount, mBuffer, mBytesBuffered
3. **128-157行目**: GetBytes - 鍵導出のメインロジック
4. **248-270行目**: NextBlockBytes - PBKDF2の1ブロック計算
5. **209-228行目**: Init - パスワードとソルトの初期化

#### Step 2: 鍵導出アルゴリズムを理解する

**GetBytes（128-157行目）の詳細**:
1. **134行目**: cbが負数ならエラー
2. **137行目**: 必要ブロック数を計算
3. **140-142行目**: 残りバッファがあれば先にコピー
4. **144-147行目**: 各ブロックについてNextBlockBytesを呼び出し
5. **149-153行目**: 余剰をバッファに保持

**NextBlockBytes（248-270行目）の詳細**:
1. **249行目**: ブロックカウンタをインクリメント
2. **251行目**: Salt || INT(blockCount) をハッシュ
3. **253-254行目**: 初回HMAC（U1）
4. **260-266行目**: 反復HMAC（U2...Uc）とXOR累積

#### Step 3: 依存コンポーネントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | HMACSHA1.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA1.cls` | 内部で使用するHMAC |

### プログラム呼び出し階層図

```
Rfc2898DeriveBytes
    │
    ├─ Init(Password, Salt, Iterations)
    │      ├─ InitHasher(Password)
    │      │      ├─ Encoding.UTF8.GetBytes() [Stringの場合]
    │      │      └─ Cor.NewHMACSHA1(Key)
    │      └─ InitSalt(Salt)
    │             └─ CryptoHelper.GetRandomBytes() [サイズ指定の場合]
    │
    ├─ GetBytes(cb)
    │      ├─ バッファ残り確認
    │      └─ NextBlockBytes() × 必要ブロック数
    │             ├─ mHasher.TransformBlock(Salt)
    │             ├─ mHasher.ComputeHash(BlockCounter)
    │             └─ 反復 × mIterations
    │                    └─ mHasher.ComputeHash(previousResult)
    │
    └─ Reset()
           └─ mHasher.Initialize()
```

### データフロー図

```
[入力]                      [処理]                           [出力]

Password ──────────────────┐
                          ▼
                    ┌─────────────┐
                    │ HMACSHA1    │
                    │ 初期化      │
                    └─────────────┘
                          │
Salt ─────────────────────▼
                    ┌─────────────┐
Iterations ───────▶│ PBKDF2      │
                    │ ブロック計算 │
cb ───────────────▶│ (反復HMAC)  │ ──────▶ 導出鍵(cbバイト)
                    └─────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Rfc2898DeriveBytes.cls | `Source/CorLib/System.Security.Cryptography/Rfc2898DeriveBytes.cls` | ソース | メインクラス |
| HMACSHA1.cls | `Source/CorLib/System.Security.Cryptography/HMACSHA1.cls` | ソース | 内部HMAC実装 |
| CryptoHelper.cls | `Source/CorLib/System.Security.Cryptography/CryptoHelper.cls` | ソース | ソルト生成ヘルパー |
| Encoding.cls | `Source/CorLib/System.Text/Encoding.cls` | ソース | UTF-8エンコーディング |
