# 機能設計書 71-ToBase64Transform

## 概要

本ドキュメントは、VBCorLibライブラリにおけるToBase64Transformクラスの機能設計を記載する。ToBase64Transformは、バイナリデータをBase64エンコード形式に変換するための暗号変換クラスである。

### 本機能の処理概要

ToBase64Transformは、バイト配列データをBase64エンコードされたバイト配列に変換するクラスである。ICryptoTransformインターフェースを実装しており、CryptoStreamと組み合わせてストリーミング方式でのデータ変換を実現する。

**業務上の目的・背景**：バイナリデータをテキスト形式で安全に転送・保存する必要がある場合、Base64エンコーディングは標準的な手法として広く使用されている。メール添付ファイル、XMLやJSONへのバイナリデータ埋め込み、データベースへのバイナリ保存など、様々な業務シーンでBase64変換が必要となる。本クラスは、.NET FrameworkのSystem.Security.Cryptography.ToBase64Transformクラスと互換性のあるVB6実装を提供し、既存の.NETアプリケーションとの相互運用性を確保する。

**機能の利用シーン**：ファイルのBase64エンコード、暗号化データの文字列化、Web API通信でのバイナリデータ送信、電子メールの添付ファイル処理などで利用される。特にCryptoStreamと組み合わせることで、大量データのストリーミング変換が可能となる。

**主要な処理内容**：
1. 3バイトの入力データを4バイトのBase64エンコードデータに変換
2. 24ビット単位でのビット操作による変換処理
3. データ長が3の倍数でない場合のパディング処理（'='文字による埋め込み）
4. ICryptoTransformインターフェースによるストリーム変換対応

**関連システム・外部連携**：CryptoStreamクラスと連携してストリーミング変換を実現する。Convert.ToBase64Stringメソッドは本クラスを内部的に使用する可能性がある。

**権限による制御**：特になし。変換処理は入力データに対して実行され、権限チェックは行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないユーティリティクラス |

## 機能種別

データ変換処理（Base64エンコーディング）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| InputBuffer | Byte() | Yes | 変換対象のバイト配列 | 配列が初期化されていること |
| InputOffset | Long | Yes | 入力配列の開始位置 | 配列の下限以上であること |
| InputCount | Long | Yes | 変換するバイト数 | 0以上、TransformBlockでは3以下、TransformFinalBlockでは0-3 |
| OutputBuffer | Byte() | Yes (TransformBlockのみ) | 出力先のバイト配列 | 配列が初期化されていること |
| OutputOffset | Long | Yes (TransformBlockのみ) | 出力配列の開始位置 | 配列の下限以上、4バイト以上の空きがあること |

### 入力データソース

プログラムから渡されるバイト配列データ。ファイル、メモリ、ネットワークストリームなど任意のソースからのバイナリデータを受け付ける。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TransformBlock戻り値 | Long | 変換されたバイト数（常に4） |
| TransformFinalBlock戻り値 | Byte() | Base64エンコードされたバイト配列（0-4バイト） |
| OutputBuffer | Byte() | Base64文字のASCIIコードが格納される |

### 出力先

引数で指定されたOutputBuffer配列、またはTransformFinalBlockの場合は新規作成されたバイト配列。

## 処理フロー

### 処理シーケンス

```
1. 入力パラメータのバリデーション
   └─ 配列の初期化確認、オフセット・サイズの範囲チェック
2. ビット操作による変換処理
   └─ 24ビット（3バイト）を6ビット×4に分割
3. Base64文字テーブル参照
   └─ 6ビット値（0-63）を対応するASCII文字に変換
4. 出力バッファへの書き込み
   └─ 4バイトのBase64文字コードを格納
5. パディング処理（TransformFinalBlockのみ）
   └─ 入力が1-2バイトの場合、'='でパディング
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{入力バイト数}
    B -->|0バイト| C[空配列を返却]
    B -->|1バイト| D[2文字変換 + ==パディング]
    B -->|2バイト| E[3文字変換 + =パディング]
    B -->|3バイト| F[4文字変換]
    D --> G[変換処理]
    E --> G
    F --> G
    G --> H[ビットシフト処理]
    H --> I[Base64テーブル参照]
    I --> J[出力バッファに書き込み]
    J --> K[終了]
    C --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 入力ブロックサイズ | 入力は3バイト単位で処理される | TransformBlock呼び出し時 |
| BR-02 | 出力ブロックサイズ | 出力は常に4バイト | 変換実行時 |
| BR-03 | パディング規則 | 入力が3の倍数でない場合は'='でパディング | TransformFinalBlock時 |
| BR-04 | 再利用可能 | インスタンスは変換後も再利用可能 | CanReuseTransform = True |
| BR-05 | 単一ブロック処理 | 一度に複数ブロックの変換は不可 | CanTransformMultipleBlocks = False |

### 計算ロジック

```
Base64エンコーディング計算式：
- 出力[0] = Base64文字[(入力[0] AND 0xFC) >> 2]
- 出力[1] = Base64文字[((入力[0] AND 0x03) << 4) OR ((入力[1] AND 0xF0) >> 4)]
- 出力[2] = Base64文字[((入力[1] AND 0x0F) << 2) OR ((入力[2] AND 0xC0) >> 6)]
- 出力[3] = Base64文字[入力[2] AND 0x3F]

Base64文字テーブル（64文字）：
A-Z (0-25), a-z (26-51), 0-9 (52-61), + (62), / (63)
パディング文字: = (ASCII 61)
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentOutOfRange | 引数範囲外 | InputOffsetが配列下限未満 | 有効なオフセット値を指定 |
| ArgumentOutOfRange | 引数範囲外 | InputCountが負数 | 0以上の値を指定 |
| Argument | 引数エラー | オフセット+サイズが配列範囲外 | 配列サイズに収まる値を指定 |
| ArgumentOutOfRange | 引数範囲外 | TransformFinalBlockでInputCount > 3 | 3以下の値を指定 |

### リトライ仕様

変換処理は同期的に実行され、リトライ機構は不要。

## トランザクション仕様

本機能はトランザクション管理を必要としない。

## パフォーマンス要件

- InputBlockSize: 3バイト
- OutputBlockSize: 4バイト
- 変換効率: 入力3バイトに対して出力4バイト（約33%の増加）

## セキュリティ考慮事項

- Base64エンコーディングは暗号化ではなく、データの秘匿性は提供しない
- 機密データを扱う場合は、Base64変換前に適切な暗号化を施すこと
- 入力データのバリデーションは呼び出し元の責任

## 備考

- .NET FrameworkのSystem.Security.Cryptography.ToBase64Transformクラスと互換性のある実装
- MITライセンスで提供（Copyright (c) 2015 Kelly Ethridge）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ICryptoTransformインターフェースの役割を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | 暗号変換インターフェースの定義、プロパティとメソッドの仕様 |

**読解のコツ**: VB6のImplementsキーワードによるインターフェース実装パターンを理解すること。

#### Step 2: エントリーポイントを理解する

処理の起点となるクラスとメソッドを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ToBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/ToBase64Transform.cls` | クラス全体の構造、公開プロパティとメソッド |

**主要処理フロー**:
1. **行57-59**: CanReuseTransformプロパティ - 常にTrueを返す
2. **行67-69**: CanTransformMultipleBlocksプロパティ - 常にFalseを返す
3. **行80-82**: InputBlockSizeプロパティ - 常に3を返す
4. **行92-94**: OutputBlockSizeプロパティ - 常に4を返す
5. **行116-134**: TransformBlockメソッド - メイン変換処理
6. **行148-178**: TransformFinalBlockメソッド - 最終ブロック変換

#### Step 3: 変換ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ToBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/ToBase64Transform.cls` | TransformBlockCoreサブルーチンのビット演算 |

**主要処理フロー**:
- **行220-225**: TransformBlockCore - 24ビットを6ビット×4に分割するビット演算
- **行159-162**: 1バイト入力時のパディング処理
- **行165-168**: 2バイト入力時のパディング処理

### プログラム呼び出し階層図

```
CryptoStream / ユーザーコード
    │
    ├─ TransformBlock(InputBuffer, InputOffset, InputCount, OutputBuffer, OutputOffset)
    │      ├─ ValidateArray(InputBuffer)
    │      ├─ ValidateArray(OutputBuffer)
    │      ├─ 引数範囲チェック
    │      └─ TransformBlockCore(InputBuffer, InputOffset, OutputBuffer, OutputOffset)
    │             └─ Base64Bytes配列参照
    │
    └─ TransformFinalBlock(InputBuffer, InputOffset, InputCount)
           ├─ ValidateArrayRange(InputBuffer, InputOffset, InputCount)
           ├─ Case 0: 空配列を返却
           ├─ Case 1: 2文字変換 + "=="
           ├─ Case 2: 3文字変換 + "="
           └─ Case 3: TransformBlockCore呼び出し
```

### データフロー図

```
[入力]                    [処理]                         [出力]

バイト配列 ───▶ ビットシフト処理 ───▶ Base64バイト配列
(3バイト)       24bit → 6bit×4         (4バイト)
                     │
                     ▼
              Base64Bytes配列参照
              (64文字テーブル)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ToBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/ToBase64Transform.cls` | ソース | Base64エンコード変換クラス |
| ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | ソース | 暗号変換インターフェース定義 |
| CryptoStream.cls | `Source/CorLib/System.Security.Cryptography/CryptoStream.cls` | ソース | ストリーミング暗号変換クラス |
| modCryptographyHelpers.bas | `Source/CorLib/System.Security.Cryptography/modCryptographyHelpers.bas` | ソース | Base64Bytes配列等の共通定義 |
| IObject.cls | `Source/CorLib/System/IObject.cls` | ソース | オブジェクト基底インターフェース |
