# 機能設計書 72-FromBase64Transform

## 概要

本ドキュメントは、VBCorLibライブラリにおけるFromBase64Transformクラスの機能設計を記載する。FromBase64Transformは、Base64エンコード形式のデータを元のバイナリデータにデコードするための暗号変換クラスである。

### 本機能の処理概要

FromBase64Transformは、Base64エンコードされたバイト配列を元のバイナリデータに変換するクラスである。ICryptoTransformインターフェースを実装しており、CryptoStreamと組み合わせてストリーミング方式でのデータ変換を実現する。空白文字の処理モードを設定でき、柔軟なデコード処理が可能である。

**業務上の目的・背景**：Base64エンコードされたデータを元のバイナリ形式に復元する必要がある場合、本クラスが使用される。メール添付ファイルの復元、XMLやJSONからのバイナリデータ抽出、データベースからのバイナリデータ読み取りなど、様々な業務シーンでBase64デコードが必要となる。本クラスは、.NET FrameworkのSystem.Security.Cryptography.FromBase64Transformクラスと互換性のあるVB6実装を提供し、既存の.NETアプリケーションとの相互運用性を確保する。

**機能の利用シーン**：Base64エンコードされたファイルのデコード、暗号化データの復元前処理、Web API通信でのバイナリデータ受信、電子メールの添付ファイル処理などで利用される。特にCryptoStreamと組み合わせることで、大量データのストリーミング変換が可能となる。

**主要な処理内容**：
1. 4バイト（4文字）のBase64入力データを3バイトのバイナリデータにデコード
2. 6ビット×4を24ビット（3バイト）に再構成するビット操作
3. パディング文字'='の処理（出力バイト数の調整）
4. 空白文字の処理（IgnoreWhiteSpacesモードまたはDoNotIgnoreWhiteSpacesモード）
5. ICryptoTransformインターフェースによるストリーム変換対応

**関連システム・外部連携**：CryptoStreamクラスと連携してストリーミング変換を実現する。Convert.FromBase64Stringメソッドは本クラスを内部的に使用する可能性がある。

**権限による制御**：特になし。変換処理は入力データに対して実行され、権限チェックは行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面を持たないユーティリティクラス |

## 機能種別

データ変換処理（Base64デコーディング）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| InputBuffer | Byte() | Yes | Base64エンコードされたバイト配列 | 配列が初期化されていること |
| InputOffset | Long | Yes | 入力配列の開始位置 | 配列の下限以上であること |
| InputCount | Long | Yes | 変換するバイト数 | 0以上であること |
| OutputBuffer | Byte() | Yes (TransformBlockのみ) | 出力先のバイト配列 | 配列が初期化されていること |
| OutputOffset | Long | Yes (TransformBlockのみ) | 出力配列の開始位置 | 配列の下限以上であること |
| WhiteSpaces | FromBase64TransformMode | No | 空白文字の処理モード | IgnoreWhiteSpaces(0)またはDoNotIgnoreWhiteSpaces(1) |

### 入力データソース

プログラムから渡されるBase64エンコードされたバイト配列データ。ファイル、メモリ、ネットワークストリームなど任意のソースからのBase64データを受け付ける。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| TransformBlock戻り値 | Long | 変換されたバイト数（0-3） |
| TransformFinalBlock戻り値 | Byte() | デコードされたバイト配列 |
| OutputBuffer | Byte() | 元のバイナリデータが格納される |

### 出力先

引数で指定されたOutputBuffer配列、またはTransformFinalBlockの場合は新規作成されたバイト配列。

## 処理フロー

### 処理シーケンス

```
1. 入力パラメータのバリデーション
   └─ 配列の初期化確認、オフセット・サイズの範囲チェック
2. 空白文字のチェック
   └─ モードに応じてスキップまたは例外スロー
3. Base64文字の検証
   └─ 有効な文字かどうかをチェック
4. ビット蓄積処理
   └─ 6ビットずつ24ビットまで蓄積
5. 24ビット蓄積完了時のデコード
   └─ パディング数に応じて1-3バイトを出力
6. 出力バッファへの書き込み
   └─ デコードされたバイトを格納
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[入力文字を読み取り]
    B --> C{空白文字?}
    C -->|Yes| D{IgnoreWhiteSpaces?}
    D -->|Yes| B
    D -->|No| E[例外スロー]
    C -->|No| F{終端文字'='?}
    F -->|Yes| G[termCount増加]
    F -->|No| H[Base64→6ビット変換]
    G --> I[6ビット蓄積]
    H --> I
    I --> J{24ビット蓄積?}
    J -->|No| K{次の入力あり?}
    K -->|Yes| B
    K -->|No| L[終了]
    J -->|Yes| M[3-termCountバイト出力]
    M --> N[状態リセット]
    N --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 入力ブロックサイズ | 入力は1バイト単位で処理される | InputBlockSize = 1 |
| BR-02 | 出力ブロックサイズ | 出力は最大3バイト | OutputBlockSize = 3 |
| BR-03 | パディング処理 | '='文字は終端を示し、出力バイト数を減らす | デコード時 |
| BR-04 | 空白文字処理 | デフォルトで空白文字を無視 | IgnoreWhiteSpacesモード |
| BR-05 | 再利用可能 | インスタンスは変換後も再利用可能 | CanReuseTransform = True |
| BR-06 | 単一ブロック処理 | 一度に複数ブロックの変換は不可 | CanTransformMultipleBlocks = False |
| BR-07 | 終端文字制限 | 連続する'='は最大2個まで | termCount <= 2 |

### 計算ロジック

```
Base64デコーディング計算式（4文字→3バイト）：
- 24ビット値 = (Base64値[0] << 18) | (Base64値[1] << 12) | (Base64値[2] << 6) | Base64値[3]
- 出力[0] = (24ビット値 AND 0xFF0000) >> 16
- 出力[1] = (24ビット値 AND 0x00FF00) >> 8
- 出力[2] = 24ビット値 AND 0x0000FF

パディング処理：
- termCount=0: 3バイト出力
- termCount=1: 2バイト出力
- termCount=2: 1バイト出力

Base64文字→値変換：
A-Z → 0-25, a-z → 26-51, 0-9 → 52-61, + → 62, / → 63

空白文字（ASCII）：32(空白), 9(タブ), 10(LF), 11(VT), 12(FF), 13(CR), 133, 160
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentOutOfRange | 引数範囲外 | InputOffsetが配列下限未満 | 有効なオフセット値を指定 |
| ArgumentOutOfRange | 引数範囲外 | InputCountが負数 | 0以上の値を指定 |
| Argument | 引数エラー | オフセット+サイズが配列範囲外 | 配列サイズに収まる値を指定 |
| Format | 書式エラー | 無効なBase64文字が検出された | 正しいBase64文字列を入力 |
| Format | 書式エラー | 終端文字'='の後に通常文字がある | 正しいBase64文字列を入力 |
| Format | 書式エラー | 終端文字'='が3個以上連続 | 正しいBase64文字列を入力 |
| Format | 書式エラー | DoNotIgnoreWhiteSpacesモードで空白文字検出 | 空白を除去するか、モードを変更 |
| Argument | 引数エラー | 出力バッファが小さすぎる | 十分なサイズのバッファを指定 |

### リトライ仕様

変換処理は同期的に実行され、リトライ機構は不要。

## トランザクション仕様

本機能はトランザクション管理を必要としない。

## パフォーマンス要件

- InputBlockSize: 1バイト（文字単位で処理）
- OutputBlockSize: 3バイト（最大出力）
- 変換効率: 入力4バイトに対して出力3バイト（約25%の削減）

## セキュリティ考慮事項

- Base64デコーディングは暗号化ではなく、データの秘匿性は提供しない
- 入力データの検証により、不正なBase64文字列による攻撃を防止
- 出力データのバリデーションは呼び出し元の責任

## 備考

- .NET FrameworkのSystem.Security.Cryptography.FromBase64Transformクラスと互換性のある実装
- MITライセンスで提供（Copyright (c) 2015 Kelly Ethridge）
- FromBase64TransformMode列挙型で空白文字の処理モードを制御可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、変換モードの列挙型とクラスの状態変数を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | FromBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/FromBase64Transform.cls` | FromBase64TransformMode列挙型の定義（行61-64） |
| 1-2 | FromBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/FromBase64Transform.cls` | メンバ変数（mIgnoreWhiteSpaces, mBits, mBitCount, mTermCount）の役割 |

**読解のコツ**: mBitsに6ビットずつ蓄積し、24ビット（4文字分）になったら出力する状態機械パターンを理解すること。

#### Step 2: エントリーポイントを理解する

処理の起点となるクラスとメソッドを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FromBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/FromBase64Transform.cls` | クラス初期化（行273-279）でのデフォルトモード設定 |

**主要処理フロー**:
1. **行79-81**: CanReuseTransformプロパティ - 常にTrueを返す
2. **行88-90**: CanTransformMultipleBlocksプロパティ - 常にFalseを返す
3. **行97-99**: InputBlockSizeプロパティ - 常に1を返す
4. **行106-108**: OutputBlockSizeプロパティ - 常に3を返す
5. **行127-198**: TransformBlockメソッド - メイン変換処理
6. **行208-232**: TransformFinalBlockメソッド - 最終ブロック変換

#### Step 3: 変換ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FromBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/FromBase64Transform.cls` | TransformBlockのメインループ（行146-195） |

**主要処理フロー**:
- **行148-149**: 入力バイトの読み取り
- **行149-164**: 空白文字チェックとBase64文字検証
- **行165-168**: 6ビットシフトと蓄積
- **行169-190**: 24ビット蓄積時のデコード出力
- **行176-186**: パディング数に応じた出力バイト数の調整
- **行289-296**: CanProcessChar - 空白文字処理ヘルパー
- **行299-303**: Reset - 状態リセット

### プログラム呼び出し階層図

```
CryptoStream / ユーザーコード
    │
    ├─ Cor.NewFromBase64Transform(WhiteSpaces)
    │      └─ Init(WhiteSpaces) - モード設定
    │
    ├─ TransformBlock(InputBuffer, InputOffset, InputCount, OutputBuffer, OutputOffset)
    │      ├─ ValidateArray(InputBuffer)
    │      ├─ ValidateArray(OutputBuffer)
    │      ├─ 引数範囲チェック
    │      └─ メインループ
    │             ├─ CanProcessChar(b) - 空白文字チェック
    │             ├─ Base64CharToBits(b) - 文字→6ビット変換
    │             ├─ ビット蓄積（mBits, mBitCount）
    │             ├─ 出力バッファ書き込み
    │             └─ Reset() - 状態リセット
    │
    └─ TransformFinalBlock(InputBuffer, InputOffset, InputCount)
           ├─ ValidateArrayRange(InputBuffer, InputOffset, InputCount)
           ├─ TransformBlock呼び出し
           └─ Reset() - 最終リセット
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Base64バイト配列 ───▶ 文字検証・空白スキップ
(1バイト単位)              │
                           ▼
                     Base64CharToBits
                     (文字→6ビット変換)
                           │
                           ▼
                     ビット蓄積（mBits）
                     6ビット × 4 = 24ビット
                           │
                           ▼
                     24ビット→3バイト分割 ───▶ バイナリ配列
                     (パディング考慮)           (1-3バイト)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FromBase64Transform.cls | `Source/CorLib/System.Security.Cryptography/FromBase64Transform.cls` | ソース | Base64デコード変換クラス |
| ICryptoTransform.cls | `Source/CorLib/System.Security.Cryptography/ICryptoTransform.cls` | ソース | 暗号変換インターフェース定義 |
| CryptoStream.cls | `Source/CorLib/System.Security.Cryptography/CryptoStream.cls` | ソース | ストリーミング暗号変換クラス |
| modCryptographyHelpers.bas | `Source/CorLib/System.Security.Cryptography/modCryptographyHelpers.bas` | ソース | Base64CharToBits関数等の共通定義 |
| IObject.cls | `Source/CorLib/System/IObject.cls` | ソース | オブジェクト基底インターフェース |
| Constructors.cls | `Source/CorLib/System/Constructors.cls` | ソース | NewFromBase64Transformファクトリメソッド |
