# 機能設計書 84-BigInteger

## 概要

本ドキュメントは、VBCorLibライブラリにおけるBigInteger（任意精度整数）機能の設計を記述する。

### 本機能の処理概要

BigIntegerは、メモリ容量のみに制限される任意精度の符号付き整数演算機能を提供するクラスである。VB6の標準型（Long、Currency、Decimal）では表現できない大きな整数値の計算を可能にする。

**業務上の目的・背景**：暗号処理、科学計算、金融計算など、非常に大きな整数を扱う必要がある場面において、標準的なVB6の数値型では精度が不足する。本機能は.NET FrameworkのSystem.Numerics.BigIntegerと互換性のあるAPIを提供し、VB6アプリケーションにおいて任意精度の整数演算を可能にする。

**機能の利用シーン**：
- RSA暗号などの暗号アルゴリズムの実装
- 非常に大きな数値を扱う科学計算
- 高精度が要求される金融計算
- 数学的なアルゴリズムの実装（階乗、フィボナッチ数列など）

**主要な処理内容**：
1. 四則演算（加算、減算、乗算、除算、剰余）
2. ビット演算（AND、OR、XOR、NOT、シフト）
3. 比較演算（等価、大小比較）
4. 数学関数（絶対値、否定、べき乗、最大公約数、平方根、階乗）
5. 型変換（バイト配列、文字列、各種数値型）
6. ビット操作（TestBit、SetBit、ClearBit、FlipBit）

**関連システム・外部連携**：
- BigNumberMathモジュールによる数学演算
- NumberFormatterによる文字列変換
- IFormattable、IComparableインターフェース実装

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 該当なし | - | - | BigIntegerはライブラリ機能として他のアプリケーションから利用される |

## 機能種別

計算処理 / データ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Value | Variant | Yes | 初期化値（Long, Double, Currency, Decimal, Byte配列, Int64） | サポートされる型であること |
| Other/Value | BigInteger | Varies | 演算対象 | Nothingの場合は0として扱う |
| Bit | Long | Yes (bit ops) | ビット位置 | 0以上であること |
| Exponent | Long | Yes (Pow) | べき乗の指数 | 任意の整数 |
| Format | String | No | 出力形式（"d", "x", "X"） | 有効なフォーマット指定子 |

### 入力データソース

- アプリケーションからのメソッド呼び出し
- BInt関数によるBigInteger生成

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| BigInteger | BigInteger | 演算結果の整数オブジェクト |
| Sign | Sign | 符号（-1, 0, 1） |
| IsZero | Boolean | ゼロ判定 |
| IsOne | Boolean | 1判定 |
| IsEven | Boolean | 偶数判定 |
| Precision | Long | バイト精度 |
| ByteArray | Byte() | バイト配列表現 |
| String | String | 文字列表現 |

### 出力先

呼び出し元アプリケーションへの戻り値

## 処理フロー

### 処理シーケンス（加算の例）

```
1. 入力値のNothingチェック（NothingはZeroとして扱う）
2. 特殊ケースの処理
   └─ いずれかがゼロの場合は他方を返す
3. BigNumberMath.Addの呼び出し
   └─ 2の補数表現での加算処理
4. 結果のBigIntegerオブジェクト生成
5. 結果の返却
```

### フローチャート（四則演算）

```mermaid
flowchart TD
    A[開始] --> B{引数がNothing?}
    B -->|Yes| C[Zeroとして扱う]
    B -->|No| D{特殊ケース?}
    C --> D
    D -->|0との演算| E[最適化された結果を返却]
    D -->|1との演算| F[最適化された結果を返却]
    D -->|通常| G[BigNumberMath呼び出し]
    G --> H[新しいBigIntegerを生成]
    H --> I[結果返却]
    E --> I
    F --> I
    I --> J[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-84-01 | 負数表現 | 2の補数表現で負数を表す | 全ての負数処理 |
| BR-84-02 | Nothing処理 | Nothingは0として扱う | 全ての演算メソッド |
| BR-84-03 | 不変性 | BigIntegerはイミュータブル | 全ての演算は新しいオブジェクトを返す |
| BR-84-04 | ゼロ除算 | ゼロで除算するとDivideByZeroException | Divide, DivRem, Modulus |
| BR-84-05 | 精度制限 | メモリ容量のみに制限される | 全ての演算 |

### 計算ロジック

**2の補数変換**:
負の整数は2の補数として内部表現される。これはVBの標準的な整数型と同じ方式。

**ビット操作**:
```
TestBit: digit(index) And value
SetBit: digit(index) Or value
ClearBit: digit(index) And (Not value)
FlipBit: TestBit ? ClearBit : SetBit
```

## データベース操作仕様

本機能はデータベースを直接操作しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| DivideByZeroException | ゼロ除算 | 除数が0 | 0以外の除数を使用する |
| ArgumentOutOfRangeException | 範囲外エラー | Bit < 0 | 0以上のビット位置を指定する |
| ArgumentException | 引数エラー | サポートされない型での初期化 | サポートされる型を使用する |
| ArgumentNullException | Null引数 | ToByteArrayで配列がNull | 有効な配列を渡す |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 内部表現はInteger配列（Byteではなく）を使用し、処理速度を最適化
- 特殊ケース（0、1、-1との演算）は最適化された処理パスを使用

## セキュリティ考慮事項

- 暗号アルゴリズムで使用する場合は、サイドチャネル攻撃に注意

## 備考

- .NET FrameworkのSystem.Numerics.BigIntegerと互換性のあるAPI
- IObject、IComparable、IFormattableインターフェースを実装
- BInt関数を使用して簡単にBigIntegerを作成可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | BigNumber型（mNumber）の構造を理解 |

**主要データ構造（54-60行目）**:
```vb
Private Type Digit
    Index As Long
    Value As Long
End Type

Private mNumber As BigNumber  ' Sign, Digits(), Precision
```

#### Step 2: 初期化処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | Init, InitFromLong, InitFromFloat等の初期化メソッド |

**主要処理フロー（893-920行目）**:
- VarTypeに応じて適切な初期化メソッドを呼び出し
- Long, Integer, Byte, Double, Single, Currency, Decimal, Byte配列, Int64をサポート

#### Step 3: 四則演算を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | Add, Subtract, Multiply, Divide, Modulusメソッド |

**主要処理フロー**:
- **337-347行目**: Add - ゼロチェック後、AddTo呼び出し
- **370-380行目**: Subtract - ゼロチェック後、SubtractFrom呼び出し
- **403-419行目**: Multiply - ゼロ/1チェック後、MultiplyBy呼び出し
- **442-457行目**: Divide - ゼロ除算チェック後、DivideInto呼び出し
- **526-533行目**: Modulus - ゼロ除算チェック後、ModulusOf呼び出し

#### Step 4: ビット演算を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | BitwiseAnd, BitwiseOr, BitwiseXor, BitwiseNot, ShiftLeft, ShiftRight |

**主要処理フロー**:
- **699-707行目**: BitwiseAnd
- **719-729行目**: BitwiseOr
- **741-751行目**: BitwiseXor
- **763-772行目**: BitwiseNot
- **830-843行目**: ShiftLeft
- **799-822行目**: ShiftRight

#### Step 5: ビット操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | TestBit, SetBit, ClearBit, FlipBit |

**主要処理フロー**:
- **557-570行目**: TestBit - ビット位置のテスト
- **580-610行目**: SetBit - ビットを1に設定
- **618-649行目**: ClearBit - ビットを0に設定
- **686-692行目**: FlipBit - ビットの反転

### プログラム呼び出し階層図

```
BigInteger
    │
    ├─ 初期化
    │      ├─ Init (VarType分岐)
    │      ├─ InitFromLong
    │      ├─ InitFromFloat
    │      ├─ InitFromCurrency
    │      ├─ InitFromDecimal
    │      ├─ InitFromArray
    │      └─ InitFromInt64
    │
    ├─ 四則演算
    │      ├─ Add → AddTo → InitAdd → BigNumberMath.Add
    │      ├─ Subtract → SubtractFrom → InitSubtract → BigNumberMath.Subtract
    │      ├─ Multiply → MultiplyBy → InitMultiply → BigNumberMath.Multiply
    │      ├─ Divide → DivideInto → InitDivide → BigNumberMath.Divide
    │      └─ Modulus → ModulusOf → InitModulus → BigNumberMath.Remainder
    │
    ├─ ビット演算
    │      ├─ BitwiseAnd → BitwiseAndWith → InitBitwiseAnd → BigNumberMath.BitwiseAnd
    │      ├─ BitwiseOr → BitwiseOrWith → InitBitwiseOr → BigNumberMath.BitwiseOr
    │      ├─ BitwiseXor → BitwiseXorWith → InitBitwiseXor → BigNumberMath.BitwiseXor
    │      ├─ BitwiseNot → InitBitwiseNot → BigNumberMath.BitwiseNot
    │      ├─ ShiftLeft → InitShiftLeft → ShiftLeftNumber
    │      └─ ShiftRight → InitShiftRight → ShiftRightNumber
    │
    └─ 出力
           ├─ ToString → NumberFormatter.FormatBigNumber
           └─ ToByteArray (直接バイト配列生成)
```

### データフロー図

```
[入力]                      [内部表現]                    [出力]

Long/Integer/Byte ────┐
                      │
Double/Single ────────┼──▶ BigNumber ──▶ BigInteger
                      │    (Sign, Digits[], Precision)
Currency/Decimal ─────┤                      │
                      │                      ▼
Byte[] ───────────────┤              ┌──────┴──────┐
                      │              │             │
Int64 ────────────────┘          Byte[]      String
                                 (ToByteArray)  (ToString)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| BigInteger.cls | `Source/CorLib/System.Numerics/BigInteger.cls` | ソース | メインクラス |
| BigIntegerStatic.cls | `Source/CorLib/System.Numerics/BigIntegerStatic.cls` | ソース | 静的メソッド（Zero, One, MinusOne等） |
| BigNumberMath.bas | `Source/CorLib/System.Numerics/BigNumberMath.bas` | ソース | 数学演算ロジック |
| modBigIntegerHelpers.bas | `Source/CorLib/System.Numerics/modBigIntegerHelpers.bas` | ソース | ヘルパー関数 |
| NumberFormatter.cls | `Source/CorLib/System/NumberFormatter.cls` | ソース | 文字列フォーマット |
| BigIntegerTests.cls | `Source/Tests/System.Numerics/BigIntegerTests.cls` | テスト | 単体テスト |
