# 機能設計書 95-MemoryMappedFile

## 概要

本ドキュメントは、VBCorLibライブラリにおけるMemoryMappedFile機能の設計を記述するものである。MemoryMappedFileは、メモリマップトファイルを表し、ディスク上のファイルの内容をメモリにマップするクラスである。

### 本機能の処理概要

MemoryMappedFileクラスは、ファイルの内容をメモリ空間にマップすることで、ファイルI/Oよりも高速なアクセスを可能にする。また、プロセス間でメモリを共有するためにも使用できる。

**業務上の目的・背景**：大容量ファイルの効率的な処理、プロセス間通信、高速なランダムアクセスが必要なシナリオにおいて、従来のファイルI/Oよりも優れたパフォーマンスを提供する。

**機能の利用シーン**：大容量ファイルの部分読み込み、プロセス間データ共有、データベースエンジンの実装、高速なファイル比較、メモリ効率の良いデータ処理に使用される。

**主要な処理内容**：
1. ファイルからのマッピング作成（CreateFromFile）- 既存ファイルをメモリにマップ
2. 新規マッピング作成（CreateNew）- 名前付きメモリマップを作成
3. 既存マッピングのオープン（OpenExisting）- 既存の名前付きマップを開く
4. ビューアクセサ作成（CreateViewAccessor）- ランダムアクセス用のビューを作成
5. ビューストリーム作成（CreateViewStream）- シーケンシャルアクセス用のストリームを作成
6. リソース解放（Dispose）- マッピングハンドルを閉じる

**関連システム・外部連携**：Windows API（CreateFileMapping、OpenFileMapping、MapViewOfFile、UnmapViewOfFile）を使用。

**権限による制御**：MemoryMappedFileAccessで読み取り/書き込み/実行権限を制御。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Simply VB Unit Runner | 参照画面 | MemoryMappedFile関連テスト（MemoryMappedFileTests）の実行 |

## 機能種別

メモリ管理 / ファイルI/O / プロセス間通信

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Path | String | Yes（CreateFromFile） | マップするファイルのパス | 既存ファイル |
| Mode | FileMode | Yes（CreateFromFile） | ファイルオープンモード | CreateNew/Open等 |
| MapName | String | No | メモリマップの名前（プロセス間共有用） | - |
| Capacity | Currency | Yes（CreateNew） | マップのサイズ（バイト） | 0より大きい値 |
| Access | MemoryMappedFileAccess | No | アクセス権限 | ReadWrite/Read/Write等 |
| Offset | Currency | No（CreateViewAccessor/Stream） | ビューの開始オフセット | 0以上 |
| Size | Long | No（CreateViewAccessor/Stream） | ビューのサイズ | 0以上 |

### 入力データソース

- ファイルシステム上のファイル
- 名前付きメモリマップ（プロセス間共有）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| MemoryMappedFile | MemoryMappedFile | メモリマップトファイルオブジェクト |
| MemoryMappedViewAccessor | MemoryMappedViewAccessor | ランダムアクセスビュー |
| MemoryMappedViewStream | MemoryMappedViewStream | ストリームアクセスビュー |
| SafeMemoryMappedFileHandle | SafeMemoryMappedFileHandle | 安全なハンドルラッパー |

### 出力先

- 呼び出し元への戻り値
- メモリ空間へのマッピング

## 処理フロー

### 処理シーケンス

```
1. CreateFromFile呼び出し
   └─ CreateFileでファイルを開く
   └─ CreateFileMappingでマッピングを作成
   └─ SafeMemoryMappedFileHandleに保存

2. CreateNew呼び出し
   └─ CreateFileMappingで新規マッピング作成
   └─ SafeMemoryMappedFileHandleに保存

3. OpenExisting呼び出し
   └─ OpenFileMappingで既存マッピングを開く
   └─ SafeMemoryMappedFileHandleに保存

4. CreateViewAccessor呼び出し
   └─ MemoryMappedViewAccessorを生成
   └─ MapViewOfFileでビューをマップ

5. CreateViewStream呼び出し
   └─ MemoryMappedViewStreamを生成
   └─ MapViewOfFileでビューをマップ
   └─ MemoryStreamをラップ
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B{作成方法}
    B -->|CreateFromFile| C[CreateFile API]
    B -->|CreateNew| D[CreateFileMapping API]
    B -->|OpenExisting| E[OpenFileMapping API]
    C --> F[CreateFileMapping API]
    F --> G[SafeMemoryMappedFileHandle生成]
    D --> G
    E --> G
    G --> H[MemoryMappedFile返却]
    H --> I{ビュー作成}
    I -->|CreateViewAccessor| J[MemoryMappedViewAccessor生成]
    I -->|CreateViewStream| K[MemoryMappedViewStream生成]
    J --> L[MapViewOfFile API]
    K --> L
    L --> M[ビュー返却]
    M --> N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-95-01 | 容量制限 | Capacityは0より大きい必要がある | CreateNew呼び出し時 |
| BR-95-02 | ページ境界 | ビューオフセットはアロケーショングラニュラリティ境界に調整 | CreateViewAccessor/Stream呼び出し時 |
| BR-95-03 | アクセス権継承 | ビューのアクセス権はファイルのアクセス権を超えられない | CreateViewAccessor/Stream呼び出し時 |
| BR-95-04 | 名前の一意性 | 同名のマップは同一のメモリ領域を共有 | 名前付きマップ作成時 |

### 計算ロジック

```
ページ境界調整:
PageAlignedOffset = Int(Offset / AllocationGranularity) * AllocationGranularity
StartIndex = Offset Mod AllocationGranularity
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ArgumentNull | ArgumentNullException | 必須パラメータがNull | 有効な値を渡す |
| ArgumentOutOfRange_NeedNonNegNum | ArgumentOutOfRangeException | 負のCapacity/Offset/Size | 0以上の値を設定 |
| Argument_MapNameEmptyString | ArgumentException | MapNameが空文字 | 有効な名前を設定するかNothingを使用 |
| FileNotFound | FileNotFoundException | ファイルが存在しない | ファイルパスを確認 |
| Win32Error | Win32Exception | API呼び出し失敗 | エラーコードを確認 |

### リトライ仕様

特になし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ファイルI/Oよりも高速なランダムアクセスを提供
- 大容量ファイルでも必要な部分のみをメモリにマップ
- プロセス間でメモリコピーなしにデータを共有可能

## セキュリティ考慮事項

- 名前付きマップは同一セッション内の他プロセスからアクセス可能
- 機密データを含む場合は名前付きマップの使用に注意
- WriteOnlyアクセスでは読み取りを防止できる

## 備考

- .NET FrameworkのSystem.IO.MemoryMappedFiles.MemoryMappedFileクラスに対応
- Currency型を使用して64ビットサイズをサポート
- Disposeで確実にリソースを解放すること

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

メモリマップトファイルの主要な構造体と列挙体を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | MemoryMappedFile.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFile.cls` | プライベート変数（47-49行目）：mFileHandle、mMappingHandle |

**読解のコツ**: SafeMemoryMappedFileHandleとSafeFileHandleがリソース管理を担当している。

#### Step 2: 静的ファクトリメソッドを理解する

MemoryMappedFileStaticクラスが提供するファクトリメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MemoryMappedFileStatic.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFileStatic.cls` | CreateFromFile（67-110行目）でファイルからマップ作成 |
| 2-2 | MemoryMappedFileStatic.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFileStatic.cls` | CreateNew（131-157行目）で新規マップ作成 |
| 2-3 | MemoryMappedFileStatic.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFileStatic.cls` | OpenExisting（173-192行目）で既存マップをオープン |

**主要処理フロー**:
- **67-110行目**: CreateFromFile - CreateFileでファイルを開き、CreateFileMappingでマッピング作成
- **131-157行目**: CreateNew - CreateFileMappingで名前付きマップを新規作成
- **173-192行目**: OpenExisting - OpenFileMappingで既存の名前付きマップを開く

#### Step 3: ビュー作成を理解する

MemoryMappedFileクラスのビュー作成メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | MemoryMappedFile.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFile.cls` | CreateViewAccessor（63-75行目）でアクセサ作成 |
| 3-2 | MemoryMappedFile.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFile.cls` | CreateViewStream（87-99行目）でストリーム作成 |

**主要処理フロー**:
- **63-75行目**: CreateViewAccessor - Cor.NewMemoryMappedViewAccessorで新しいアクセサを生成
- **87-99行目**: CreateViewStream - Cor.NewMemoryMappedViewStreamで新しいストリームを生成

### プログラム呼び出し階層図

```
MemoryMappedFile.CreateFromFile
    │
    ├─ CreateFile（Windows API）
    │
    └─ CreateFileMapping（Windows API）
           │
           └─ SafeMemoryMappedFileHandle

MemoryMappedFile.CreateViewAccessor
    │
    └─ MemoryMappedViewAccessor.Init
           │
           └─ MapViewOfFile（Windows API）

MemoryMappedFile.CreateViewStream
    │
    └─ MemoryMappedViewStream.Init
           │
           ├─ MapViewOfFile（Windows API）
           │
           └─ MemoryStream（ラッパー）
```

### データフロー図

```
[入力]                    [処理]                         [出力]

ファイルパス ────────────▶ CreateFromFile ────────────────▶ MemoryMappedFile
Mode/Access ─────────────┘      │
                                ▼
                         CreateFile API
                                │
                                ▼
                         CreateFileMapping API

Offset/Size ─────────────▶ CreateViewAccessor ──────────▶ MemoryMappedViewAccessor
Access ──────────────────┘      │
                                ▼
                         MapViewOfFile API
                                │
                                ▼
                         バイト配列へのマッピング
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| MemoryMappedFile.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFile.cls` | ソース | メモリマップトファイルの実装 |
| MemoryMappedFileStatic.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedFileStatic.cls` | ソース | 静的ファクトリメソッド |
| MemoryMappedViewAccessor.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedViewAccessor.cls` | ソース | ランダムアクセスビュー |
| MemoryMappedViewStream.cls | `Source/CorLib/System.IO.MemoryMappedFiles/MemoryMappedViewStream.cls` | ソース | ストリームアクセスビュー |
| SafeMemoryMappedFileHandle.cls | `Source/CorLib/Microsoft.Win32.SafeHandles/SafeMemoryMappedFileHandle.cls` | ソース | 安全なハンドル管理 |
