# 機能設計書 99-ArgumentException

## 概要

本ドキュメントは、VBCorLibライブラリにおけるArgumentException機能の設計を記述するものである。ArgumentExceptionは、メソッドに渡された引数が無効な場合にスローされる例外クラスである。

### 本機能の処理概要

ArgumentExceptionクラスは、メソッドの引数検証で発生するエラーを表現する例外である。エラーメッセージに加えて、問題のあった引数の名前（ParamName）を保持し、エラーの特定を容易にする。

**業務上の目的・背景**：メソッドの入力検証において、無効な引数が渡された場合に詳細なエラー情報を提供する。ParamNameにより、どの引数に問題があったかを明確に伝達できる。

**機能の利用シーン**：引数の型が不正、引数の値が期待範囲外（ただしNullや範囲外はより具体的な派生例外）、引数の組み合わせが不正、引数の形式が不正な場合に使用される。

**主要な処理内容**：
1. エラー情報の保持（Message/ParamName）
2. Exceptionからの継承機能
3. SystemExceptionインターフェースの実装
4. メッセージへのParamName追加表示

**関連システム・外部連携**：特になし

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連なし（内部クラス） |

## 機能種別

エラー処理 / 例外クラス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Message | String | No | エラーメッセージ | - |
| ParamName | String | No | 問題のあった引数名 | - |
| InnerException | Exception | No | 原因となった内部例外 | - |

### 入力データソース

- アプリケーションコードからの直接生成
- VBCorLib内部での例外生成

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Message | String | エラーメッセージ + "Parameter name: " + ParamName |
| ParamName | String | 問題のあった引数名 |
| HResult | Long | COR_E_ARGUMENT |
| ErrorNumber | Long | vbInvalidProcedureCall |

### 出力先

- 呼び出し元への戻り値

## 処理フロー

### 処理シーケンス

```
1. ArgumentException生成
   └─ NewArgumentExceptionでインスタンス生成
   └─ ExceptionBaseに委譲して初期化
   └─ Message、ParamName、InnerExceptionを設定

2. Messageプロパティアクセス
   └─ GetExceptionMessageで基本メッセージ取得
   └─ ParamNameがあれば追加表示

3. 他のプロパティはExceptionと同様にExceptionBaseに委譲
```

### フローチャート

```mermaid
flowchart TD
    A[NewArgumentException] --> B[ExceptionBase生成]
    B --> C[Message設定]
    C --> D[ParamName設定]
    D --> E[InnerException設定]
    E --> F[ArgumentException返却]
    F --> G{Messageアクセス?}
    G -->|Yes| H[GetExceptionMessage]
    H --> I{ParamName存在?}
    I -->|Yes| J[Message + ParamName追加]
    I -->|No| K[Message返却]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-99-01 | HResult | HResultはCOR_E_ARGUMENT（&H5） | HResultアクセス時 |
| BR-99-02 | ErrorNumber | ErrorNumberはvbInvalidProcedureCall | ErrorNumberアクセス時 |
| BR-99-03 | Message形式 | ParamNameがある場合、Messageの末尾に"Parameter name: {ParamName}"を追加 | Messageアクセス時 |
| BR-99-04 | デフォルトMessage | Messageが空の場合、Arg_ArgumentExceptionリソースを使用 | Messageアクセス時 |

### 計算ロジック

特になし

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

ArgumentExceptionクラス自体は例外を発生させない（例外情報を保持するのみ）。

### リトライ仕様

特になし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 例外オブジェクトの生成は軽量

## セキュリティ考慮事項

- エラーメッセージに引数の値など機密情報を含めないこと

## 備考

- .NET FrameworkのSystem.ArgumentExceptionクラスに対応
- ArgumentNullException、ArgumentOutOfRangeExceptionの基底クラスとして使用可能
- Implements Exception, SystemExceptionでインターフェースを実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: クラス構造を理解する

ArgumentExceptionのクラス宣言とインターフェース実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | Implements IObject, Exception, SystemException（57-59行目） |

**読解のコツ**: ArgumentExceptionはExceptionインターフェースとSystemExceptionインターフェースの両方を実装し、mBaseでExceptionBaseに委譲する。

#### Step 2: 初期化処理を理解する

例外オブジェクトの生成と初期化を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | Class_InitProperties（241-243行目）でExceptionBase生成 |
| 2-2 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | Init（245-249行目）でMessage、ParamName、InnerException設定 |

**主要処理フロー**:
- **241-243行目**: Class_InitProperties - Cor.NewExceptionBaseでExceptionBaseを生成
- **245-249行目**: Init - Message、ParamNameをmBaseに設定、InnerExceptionを設定

#### Step 3: ParamNameの取得を理解する

ParamNameがExceptionBase.GetValueで取得される仕組みを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | ParamName（84-86行目）でmBase.GetValueを使用 |

**主要処理フロー**:
- **84-86行目**: ParamName Get - mBase.GetValue(PropParamName, vbNullString)で値を取得
- **61行目**: PropParamName定数 = "ParamName"

#### Step 4: Messageの拡張を理解する

ParamNameをMessageに追加する処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | Message（164-172行目）でParamNameを追加 |

**主要処理フロー**:
- **165行目**: GetExceptionMessageで基本メッセージを取得
- **167-171行目**: ParamNameがあれば、vbCrLf + "Parameter name: " + ParamNameを追加

#### Step 5: インターフェース実装を理解する

Exception/SystemExceptionインターフェースの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | Exception_*（282-333行目）でインターフェース実装 |
| 5-2 | ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | SystemException_*（336-389行目）でインターフェース実装 |

### プログラム呼び出し階層図

```
NewArgumentException
    │
    └─ ArgumentException.Init
           │
           ├─ mBase.Message = Message
           │
           ├─ mBase.SetValue PropParamName, ParamName
           │
           └─ mBase.InnerException = InnerException

ArgumentException.Message
    │
    ├─ GetExceptionMessage(mBase, Arg_ArgumentException)
    │
    └─ ParamName追加（存在時）
           └─ vbCrLf & "Parameter name: " & ParamName

ArgumentException.ParamName
    │
    └─ mBase.GetValue(PropParamName, vbNullString)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

Message ─────────────────▶ ArgumentException.Init ──────▶ mBase.Message設定
ParamName ───────────────┘      │
InnerException ──────────────────┘

（なし） ─────────────────▶ ArgumentException.Message ──▶ 拡張メッセージ
                                │
                                ▼
                         GetExceptionMessage
                                │
                                ▼
                         ParamName追加判定
                                │
                                ▼
                         最終メッセージ生成
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | ソース | 引数例外クラス |
| Exception.cls | `Source/CorLib/System/Exception.cls` | ソース | 基底クラス |
| ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | ソース | 実装ヘルパー |
| ArgumentNullException.cls | `Source/CorLib/System/ArgumentNullException.cls` | ソース | 派生クラス |
