# API設計意図書

## 概要

本ドキュメントは、VBCorLibライブラリの公開API設計における意図、設計判断の根拠、および代替案との比較について記載しています。VBCorLibは.NET Framework 2.0のクラスライブラリをVB6/VBAで再実装したCOMライブラリであり、コレクション、暗号化、エンコーディング、ファイル操作、グローバリゼーションなど幅広い機能を提供します。

> **注意**: VBCorLibはHTTP API（REST/GraphQL等）ではなく、COMライブラリとして提供されるプログラミングAPIです。本ドキュメントでは、ライブラリの公開クラス、インターフェース、メソッドを「API」として設計意図を文書化しています。

---

## 業務要件との関連（なぜこのAPIが必要か）

### 対応する業務要件

| 業務要件ID | 業務要件名 | APIに求められる要件 |
| --- | --- | --- |
| BR-001 | レガシーVB6/VBAシステムの近代化 | .NET互換のモダンなプログラミングパターンをVB6/VBAで利用可能にする |
| BR-002 | VB6/VBAと.NETアプリケーション間のデータ共有 | 同一のデータ構造・シリアライゼーション形式を使用した相互運用性の実現 |
| BR-003 | 国際化対応（多言語・多文化サポート） | Unicode対応、多様な文字エンコーディング、各国カレンダー・日時書式のサポート |
| BR-004 | セキュリティ要件への対応 | 産業標準の暗号化・ハッシュアルゴリズム（AES、RSA、SHA-256等）の提供 |
| BR-005 | 大容量ファイル処理 | 2GBを超えるファイルの操作能力（VB6標準の制限を克服） |

### 背景にあると推察される業務課題

#### 1. VB6/VBAの言語機能の限界

VB6/VBAは1990年代後半に設計された言語であり、以下の重大な制限がありました：

- **コレクション機能の貧弱さ**: 標準のCollection型は機能が限定的で、ソート・検索・型安全性が不足
- **文字エンコーディングの制限**: Unicode対応が不完全で、国際化対応が困難
- **暗号化機能の欠如**: 標準ライブラリに暗号化機能がなく、セキュリティ要件を満たせない
- **ファイル操作の制限**: 2GBを超えるファイルを扱えない
- **日付・時刻操作の不足**: タイムゾーン処理、精密な時間計算が困難

これらの制限は、企業の基幹業務システムをVB6/VBAで構築・維持する上で深刻な障害となっていました。

#### 2. .NETへの段階的移行ニーズ

多くの企業がVB6/VBAで構築されたレガシーシステムを保有しており、一括での.NET移行は以下の理由で困難でした：

- 大規模なコードベースの書き換えコスト
- 業務停止リスク
- 既存のVBA（Excel、Access）マクロとの連携継続の必要性

VBCorLibは、完全な移行を行わずとも.NET互換のプログラミングパターンを活用できるようにすることで、この課題に対処しています。

### 業務課題に対するアプローチ方針

VBCorLibは以下のアプローチで業務課題を解決しています：

1. **.NET Framework APIの忠実な再実装**: .NET Framework 2.0の主要クラスを「ほぼ同一」のインターフェースで提供。これにより：
   - .NETの豊富なドキュメント・ナレッジを活用可能
   - 将来の.NET移行時にコード変換が容易

2. **COM互換性の維持**: VB6およびVBA（Excel、Access、Word等）から直接参照・利用可能なCOMライブラリとして提供

3. **段階的な機能強化**: バージョン2.0から4.0にかけて、暗号化、BigInteger、MemoryMappedFile等の高度な機能を段階的に追加

### 想定されるAPIの利用者と利用シーン

| 利用者種別 | 利用目的 | 利用シーン |
| --- | --- | --- |
| VB6アプリケーション開発者 | レガシーアプリケーションの機能拡張・保守 | 既存VB6システムへの新機能追加（暗号化、国際化対応等） |
| Excel/Accessマクロ開発者 | VBAマクロの高度化 | 業務自動化マクロでの複雑なデータ処理、ファイル操作 |
| システム統合担当者 | VB6/.NET間のデータ連携 | 異種システム間でのデータ形式の統一、シリアライゼーション |
| セキュリティ担当者 | レガシーシステムのセキュリティ強化 | 暗号化処理の追加、ハッシュによるデータ整合性検証 |

### 業務上のAPI品質要件

| 項目 | 業務要求 | 根拠 |
| --- | --- | --- |
| API互換性 | .NET Framework 2.0との高い互換性 | 既存.NETドキュメント・ナレッジの活用、将来の移行容易性 |
| パフォーマンス | VB6標準機能と同等以上 | 既存業務処理のパフォーマンス劣化を防止 |
| 安定性 | 本番環境での長期運用に耐える | 基幹業務システムでの利用を想定 |
| 後方互換性 | メジャーバージョン間での互換性維持 | 既存コードの動作継続を保証 |

### 検討した実現方式（業務観点）

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | VB6標準機能のみで対応 | 追加ライブラリ不要 | 機能制限が大きい、生産性低下 | 低 |
| B | Windows API直接呼び出し | 高い柔軟性 | 開発難易度高、保守性低、危険性 | 低 |
| C | サードパーティCOMコンポーネント購入 | 即座に利用可能 | ライセンスコスト、ベンダー依存 | 中 |
| D（採用） | .NET互換のオープンソースライブラリ | .NET互換、無償、カスタマイズ可能 | 初期学習コスト | 高 |

### 選択理由（業務観点）

1. **業務要件との適合性**: .NET Framework互換のAPIを提供することで、企業が必要とするコレクション、暗号化、国際化、ファイル操作の全ての機能をカバー

2. **外部連携の業務要求**: .NETアプリケーションと同一のデータ構造・シリアライゼーション形式を使用することで、異種システム間のデータ連携が容易

3. **将来の業務拡大への対応**: .NET互換APIを学習することで、将来的な.NET移行時のスキル移転が容易。新機能追加時も.NETの豊富なリソースを活用可能

4. **開発・運用の効率性**: MITライセンスのオープンソースとして提供されており、ライセンスコスト不要。ソースコードが公開されているためカスタマイズ・デバッグが可能

---

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| API形式 | 独自設計 / .NET互換 / Java互換 | .NET Framework 2.0互換 | 既存ナレッジ活用、.NET移行容易性 |
| 名前空間構造 | フラット / 階層型 | System.* 階層型（.NET準拠） | .NETとの一貫性、論理的整理 |
| インターフェース設計 | 具象クラスのみ / インターフェース中心 | インターフェース中心（IObject、ICollection等） | 多態性、拡張性、テスト容易性 |
| エラー処理 | VB6標準エラー / 例外クラス階層 | .NET互換例外クラス階層 | 詳細なエラー情報、一貫したハンドリング |
| バージョニング | セマンティック / 日付ベース | セマンティックバージョニング（メジャー.マイナー） | 互換性管理の明確化 |

---

## 設計判断と根拠

### 1. API形式の選択

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 独自設計API | VB6に最適化した独自のAPI設計 | VB6の慣習に沿う | 学習コスト、移行困難 |
| .NET Framework互換API | .NETのクラス・メソッドを忠実に再実装 | 既存ドキュメント活用、移行容易 | .NETの設計がVB6に最適とは限らない |
| Java互換API | JavaのCollection Framework等を模倣 | 別のエコシステムの活用 | VB6/VBA利用者に馴染みがない |

#### 採用した方針

**.NET Framework 2.0互換API**

コードおよびドキュメントから、VBCorLibは.NET Framework 2.0のクラスライブラリを忠実に再実装していることが確認できます。クラス名、メソッド名、シグネチャは可能な限り.NETと一致しています。

#### 選択理由

1. **豊富な既存リソースの活用**: .NET Frameworkには膨大なドキュメント、チュートリアル、Q&Aが存在。VBCorLib利用者はこれらを直接参照可能

2. **学習コストの削減**: .NETを知っている開発者は即座にVBCorLibを使用可能。逆にVBCorLibで学んだ知識は.NETに移行しても活用可能

3. **データ互換性**: 同一のクラス構造を使用することで、VB6/.NET間でのデータシリアライゼーション・デシリアライゼーションが容易

4. **将来の移行パス**: VBCorLibを使用したコードは、.NETへの移行時に最小限の変更で済む

#### 不採用とした選択肢の理由

- **独自設計API**: 学習コストが高く、.NET移行時に全面的な書き換えが必要になる
- **Java互換API**: VB6/VBA開発者のコミュニティでJavaは一般的でなく、リソースの活用が困難

---

### 2. インターフェース設計パターン

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 具象クラスのみ | インターフェースを使用せず具象クラスのみ提供 | シンプル、学習容易 | 柔軟性低、テスト困難 |
| インターフェース中心 | 共通インターフェースを定義し、複数実装を可能に | 多態性、拡張性、テスト容易 | 複雑性増加 |
| 抽象クラス中心 | 抽象クラスで共通実装を提供 | コード再利用 | VB6は抽象クラス非対応（インターフェースで代替） |

#### 採用した方針

**インターフェース中心設計**

VBCorLibは以下の主要インターフェースを定義しています：

```
IObject        - 基底インターフェース（Equals、GetHashCode、ToString）
ICloneable     - 複製機能
IComparable    - 比較機能
IFormattable   - 書式指定文字列変換
ICollection    - コレクション基本機能
IEnumerable    - For Each対応
IEnumerator    - イテレータ
IList          - リスト機能
IDictionary    - 辞書機能
IComparer      - カスタム比較
IEqualityComparer - 等価性比較
```

#### 選択理由

1. **多態性の実現**: 異なるコレクション実装（ArrayList、Hashtable、SortedList等）を同一のインターフェース（ICollection、IEnumerable）で扱える

2. **拡張性**: 利用者が独自のコレクションクラスを作成する際、標準インターフェースを実装することでVBCorLibの他のクラスと連携可能

3. **一貫したプログラミングモデル**: 全てのオブジェクトがIObjectを実装することで、Equals、GetHashCode、ToStringが統一的に利用可能

4. **.NETとの互換性**: .NETと同じインターフェース階層を採用することで、.NETのパターン・プラクティスをそのまま適用可能

#### インターフェース設計の特徴

| パターン | 採用方針 | 選択理由 |
| --- | --- | --- |
| 基底インターフェース | IObjectを全クラスに実装 | 一貫したオブジェクト操作（比較、ハッシュ、文字列化） |
| コレクション階層 | IEnumerable -> ICollection -> IList/IDictionary | .NET互換、機能の段階的拡張 |
| スタティッククラス | 名前Static（例：CorDateTimeStatic）を用意 | VB6はスタティックメソッドを直接サポートしないため |

---

### 3. 名前空間構造の設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| フラット構造 | 全クラスを単一名前空間に配置 | シンプル、参照容易 | 名前衝突、整理困難 |
| 階層型（.NET準拠） | System.Collections、System.IO等の階層 | 論理的整理、.NET互換 | 参照時に長い名前 |
| 機能別グループ | 独自のカテゴリ分類 | 柔軟な整理 | .NETとの乖離 |

#### 採用した方針

**.NET Framework準拠の階層型名前空間**

```
System                          - 基本型（CorArray、CorString、CorDateTime等）
System.Collections              - コレクション（ArrayList、Hashtable等）
System.Text                     - 文字列処理（StringBuilder、Encoding等）
System.IO                       - ファイル操作（File、Directory、Stream等）
System.Globalization            - 国際化（CultureInfo、Calendar等）
System.Security.Cryptography    - 暗号化（SHA256、RSA、AES等）
System.Numerics                 - 数値型（BigInteger）
System.Resources                - リソース管理
Microsoft.Win32                 - レジストリアクセス
```

#### 選択理由

1. **.NETとの一貫性**: 開発者が.NETドキュメントを参照する際、同じ名前空間構造であるため混乱がない

2. **論理的な機能分類**: 関連するクラスが同一名前空間にまとめられ、発見可能性が向上

3. **将来の拡張性**: 新しい機能カテゴリを追加する際、.NETの名前空間規則に従って整理可能

---

### 4. エラー処理設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| VB6標準エラー（Err.Raise） | エラー番号とメッセージのみ | VB6慣習に沿う | エラー情報が限定的 |
| .NET互換例外クラス階層 | ArgumentException、IOException等の例外型 | 詳細情報、型による分岐 | 実装複雑 |
| エラーコード戻り値 | メソッドがエラーコードを返す | シンプル | 戻り値の意味が二重化 |

#### 採用した方針

**.NET互換例外クラス階層**

VBCorLibは以下の例外クラス階層を提供しています：

```
Exception
├── SystemException
│   ├── ArgumentException
│   │   ├── ArgumentNullException
│   │   └── ArgumentOutOfRangeException
│   ├── ArithmeticException
│   │   ├── DivideByZeroException
│   │   └── OverflowException
│   ├── FormatException
│   ├── IndexOutOfRangeException
│   ├── InvalidCastException
│   ├── InvalidOperationException
│   │   └── ObjectDisposedException
│   ├── NotSupportedException
│   ├── IOException
│   │   ├── FileNotFoundException
│   │   ├── DirectoryNotFoundException
│   │   └── PathTooLongException
│   └── CryptographicException
└── ApplicationException
```

#### 選択理由

1. **詳細なエラー情報**: 例外オブジェクトにはMessage、ParamName、ActualValue等の詳細情報を格納可能

2. **型による分岐**: `TypeOf ex Is ArgumentNullException`のような型チェックで、エラーの種類に応じた処理が可能

3. **一貫したエラーハンドリング**: VBCorLib全体で統一された例外パターンを使用

4. **.NETとの互換性**: .NETと同じ例外型を使用することで、エラーハンドリングのパターンが共通化

---

### 5. スタティックメソッドの実装パターン

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 標準モジュール（.bas） | Public Functionとしてモジュールに定義 | シンプル、VB6標準 | 名前空間がない、グローバル汚染 |
| Staticクラスパターン | 名前Staticクラス（例：CorDateTimeStatic）を使用 | 名前空間の整理、.NET互換パターン | やや冗長 |
| シングルトンパターン | グローバルインスタンスを提供 | オブジェクト指向的 | 状態管理が複雑 |

#### 採用した方針

**Staticクラスパターン + コンストラクタ関数**

VBCorLibでは以下のパターンを使用しています：

1. **Staticサフィックスクラス**: `CorDateTimeStatic`、`TimeSpanStatic`等のクラスにスタティックメソッド相当を実装

2. **Statics.basモジュール**: グローバルアクセス用のプロパティを提供
   ```vb
   Public Property Get CorDateTime() As CorDateTimeStatic
   Public Property Get TimeSpan() As TimeSpanStatic
   ```

3. **コンストラクタ関数**: `NewArrayList`、`NewHashtable`等のファクトリ関数を提供

#### 選択理由

1. **.NET風の呼び出し構文**: `CorDateTime.Now`、`TimeSpan.FromDays(7)`のような.NET風の構文で呼び出し可能

2. **名前の衝突回避**: クラス名でスコープを区切ることで、同名のメソッドが異なるクラスに存在可能

3. **コード補完の活用**: IDEでクラス名を入力すると、利用可能なメソッドが一覧表示される

---

### 6. COMコンポーネントとしての設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ActiveX DLL | COMコンポーネントとして登録 | VB6/VBA/他言語から利用可能 | レジストリ登録が必要 |
| 標準DLL | C呼び出し規約のDLL | レジストリ不要 | VB6からの呼び出しが複雑 |
| .NETアセンブリ（COM相互運用） | .NETで実装しCOM公開 | 最新技術活用 | .NETランタイム必要、複雑 |

#### 採用した方針

**ActiveX DLL（COMコンポーネント）**

VBCorLibはVB6で実装されたActiveX DLLとして提供されています。

#### 選択理由

1. **ネイティブVB6対応**: VB6で実装されているため、VB6/VBAから完全にネイティブとして動作

2. **ランタイム不要**: .NETランタイムやその他の外部依存がない（VB6ランタイムのみ必要）

3. **広い互換性**: VB6、VBA（Excel、Access、Word等）、VBScript、その他COM対応言語から利用可能

4. **Type Library提供**: IntelliSense（コード補完）が動作し、開発効率向上

---

### 7. 後方互換性の維持方針

#### バージョン履歴から見る互換性方針

README.TXTの履歴から、VBCorLibは以下の互換性方針を採用していることが推測されます：

| バージョン遷移 | 対応内容 | 互換性への影響 |
| --- | --- | --- |
| 2.x -> 3.0 | クラス名変更（cArray->CorArray等） | **破壊的変更** |
| 3.x内 | バグ修正、最適化 | 後方互換維持 |
| 3.x -> 4.0 | 細部の改善、新機能追加 | 一部破壊的変更 |

#### 採用した方針

**セマンティックバージョニング風の管理**

- **メジャーバージョン**: 破壊的変更を含む可能性あり
- **マイナーバージョン**: 後方互換性を維持した機能追加・修正

#### 選択理由

1. **既存コードの保護**: マイナーバージョン間では既存コードが動作継続

2. **計画的な移行**: 破壊的変更はメジャーバージョンに集約され、移行計画が立てやすい

3. **明確な変更履歴**: README.TXTに詳細な変更履歴を記載し、**BREAKING** タグで破壊的変更を明示

---

### 8. Unicode対応とエンコーディング設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ANSI専用 | Windows ANSIコードページのみ | シンプル | 国際化困難 |
| UTF-8専用 | UTF-8のみサポート | モダン、Web互換 | レガシーシステムとの連携困難 |
| マルチエンコーディング | 多様なエンコーディングをサポート | 高い柔軟性 | 実装複雑 |

#### 採用した方針

**マルチエンコーディングサポート**

VBCorLibは以下のエンコーディングをサポートしています：

```
ASCIIEncoding       - ASCII（7ビット）
UTF7Encoding        - UTF-7
UTF8Encoding        - UTF-8（BOM有無対応）
UnicodeEncoding     - UTF-16（LE/BE対応）
UTF32Encoding       - UTF-32
DBCSCodePageEncoding - Windows コードページ
```

#### 選択理由

1. **国際化対応**: 多言語コンテンツの処理が可能

2. **相互運用性**: .NETアプリケーション、Webシステム、レガシーシステムとのデータ交換に対応

3. **バイナリデータ処理**: 文字列とバイト配列間の正確な変換が可能

---

## 将来の拡張性考慮

### 想定される変更

1. **.NET Framework 4.x以降の機能追加**
   - Task並列ライブラリ相当の非同期処理
   - LINQ相当のクエリ機能
   - 追加の暗号化アルゴリズム

2. **新しいWindows APIへの対応**
   - Windows 10/11固有のAPI
   - 64ビット対応の強化

### 後方互換性

以下の方針で後方互換性を維持しています：

1. **インターフェースの安定性**: IObject、ICollection等の基盤インターフェースは変更しない

2. **非推奨化プロセス**: 機能削除時は事前に非推奨として警告、次メジャーバージョンで削除

3. **明確な変更履歴**: 全ての変更をREADME.TXTに記録、破壊的変更は**BREAKING**タグで明示

---

## 備考

- VBCorLibは.NET Framework 2.0相当のAPIを提供していますが、一部の機能は独自に拡張または省略されています
- 詳細なAPIドキュメントは http://www.kellyethridge.com/vbcorlib/doc/CorLib.html で参照可能です
- ライセンスはMITライセンスであり、商用利用を含む自由な利用が可能です
- 本ドキュメントの設計意図は、コードおよび関連ドキュメントの分析に基づく推測を含みます
