# セキュリティ設計意図書

## 概要

本ドキュメントは、VBCorLibプロジェクトにおけるセキュリティ設計の意図と判断根拠を記載する。VBCorLibは.NET Frameworkの暗号化機能をVisual Basic 6向けに移植したライブラリであり、Windows CryptoAPIを活用した包括的な暗号化サービスを提供する。本設計意図書では、各セキュリティ機能の選定理由、代替案の検討結果、および業務観点での必要性を明確化する。

---

## 業務要件との関連（なぜこのセキュリティ設計が必要か）

### 対応する業務要件
| 業務要件ID | 業務要件名 | セキュリティに求められる要件 |
| --- | --- | --- |
| BR-001 | レガシーシステムのデータ保護 | VB6アプリケーションで扱う機密データを現代の暗号化基準で保護する |
| BR-002 | .NET互換の暗号化実装 | .NET Framework互換のAPIを提供し、異種システム間でのデータ交換を安全に行う |
| BR-003 | 既存VB6資産の延命 | 既存のVB6アプリケーションを書き換えることなくセキュリティ対策を強化する |
| BR-004 | 規制対応 | 業界標準の暗号化アルゴリズムをサポートし、コンプライアンス要件に対応する |

### 業務観点でのセキュリティ要件

#### 取り扱うデータの機密性
VBCorLibを利用するアプリケーションが取り扱う可能性のあるデータとその保護要件を以下に示す。

| データ種別 | 機密性レベル | 業務上の保護理由 |
| --- | --- | --- |
| 暗号化キー・秘密鍵 | 最高 | 漏洩時に全ての暗号化データが危殆化。不正アクセス・情報漏洩の直接的原因となる |
| パスワード・認証情報 | 最高 | システムへの不正アクセスに直結。パスワードリスト攻撃の踏み台となり得る |
| 個人情報・顧客データ | 高 | 個人情報保護法等の法的義務。漏洩時の企業信用失墜・賠償責任リスク |
| 業務トランザクションデータ | 中 | 改ざん検知が必要。デジタル署名による完全性保証が求められる |
| 設定情報 | 低 | システム設定の不正変更防止。整合性検証が必要な場合あり |

#### 法令・規制への対応
VBCorLibが業務利用される際に準拠すべき可能性のある法令・規制を以下に示す。

| 法令/規制 | 概要 | 業務への影響 |
| --- | --- | --- |
| 個人情報保護法 | 個人情報の適切な取り扱い義務 | 暗号化による保護措置が安全管理措置の一部として求められる |
| 不正競争防止法 | 営業秘密の保護 | 暗号化は秘密管理性を担保する重要な手段 |
| 電子署名法 | 電子署名の法的効力 | RSA/DSA署名機能が法的に有効な電子署名の基盤となる |
| 金融庁ガイドライン | 金融機関向けセキュリティ基準 | 暗号化アルゴリズムの強度要件への適合 |
| NIST SP 800-131A | 暗号アルゴリズムの推奨基準 | 鍵長・アルゴリズム選定の根拠として参照 |

#### お客様からのセキュリティ要求
VBCorLibライブラリを利用する開発者・組織から想定されるセキュリティ要求を以下に示す。

| 要求事項 | 内容 | 対応方針 |
| --- | --- | --- |
| 業界標準アルゴリズムの採用 | AES、RSA、SHA-256等の標準アルゴリズムサポート | 対応必須：主要アルゴリズムを網羅的に実装 |
| .NET Framework互換性 | .NETの暗号化APIと同等の機能・互換性 | 対応必須：APIシグネチャ・動作の互換性を維持 |
| 安全なキー管理 | Windows CSPを活用した安全なキー保管 | 対応必須：CspParameters、キーコンテナ機能を実装 |
| 弱鍵対策 | 既知の弱鍵の検出・拒否機能 | 対応必須：DES/TripleDES弱鍵検出機能を実装 |

### 検討したセキュリティレベル（業務観点）

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | 基本的な暗号化機能のみ（DES、MD5中心） | 実装が単純、軽量 | 現代の基準では脆弱、法令準拠困難 | 低 |
| B | 推奨アルゴリズムのみ（AES、SHA-256以降） | 高セキュリティ、将来性あり | レガシーシステムとの互換性なし | 中 |
| C（採用） | 包括的アルゴリズムサポート＋推奨/非推奨の明確化 | 互換性確保＋セキュリティガイダンス提供 | 実装工数が大きい | 高 |

### 選択理由（業務観点）

1. **業務要件との適合性**: VB6で開発された既存システムを維持しながら、セキュリティを強化する必要がある。案Cは新旧両方のアルゴリズムをサポートしつつ、推奨/非推奨を明示することで、既存システムの段階的な移行を支援する。

2. **法令・規制への準拠**: 金融庁ガイドラインやNIST基準では、AES-256、RSA-2048以上、SHA-256以上が推奨される。案Cはこれらの推奨アルゴリズムを完全サポートしつつ、ドキュメントで非推奨アルゴリズムを明示する。

3. **顧客からの信頼確保**: .NET Frameworkと同等のAPIを提供することで、開発者は既存の知識・経験を活用できる。また、Windows CryptoAPIを活用することで、OS標準の検証済み暗号化実装を利用できる。

4. **インシデント発生時の業務影響**: 弱鍵検出機能、キークリア機能、安全な乱数生成など、多層防御を実装することで、単一障害点による致命的な情報漏洩リスクを軽減する。

---

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| 暗号化基盤 | 独自実装 / OS標準API利用 | Windows CryptoAPI | 検証済み実装の活用、OS標準の信頼性 |
| 対称暗号アルゴリズム | AES単独 / 複数アルゴリズム | Rijndael(AES)/TripleDES/DES/RC2 | .NET互換性、レガシーサポート、推奨の明示 |
| 非対称暗号アルゴリズム | RSA単独 / RSA+DSA | RSA + DSA | 暗号化と署名の両用途をカバー |
| ハッシュアルゴリズム | SHA-256のみ / 複数アルゴリズム | SHA-1/SHA-256/384/512/MD5/RIPEMD-160 | 互換性確保、用途に応じた選択肢提供 |
| 鍵導出関数 | 独自実装 / 標準準拠 | PBKDF2 (RFC 2898) | 業界標準、辞書攻撃耐性 |
| 乱数生成 | VB標準Rnd / 暗号学的乱数 | CryptoAPI乱数生成 | 予測不可能性の確保 |
| キー管理 | メモリのみ / CSPコンテナ | CSPキーコンテナ + メモリ管理 | 永続化オプション、安全な消去 |

---

## 設計判断と根拠

### 1. 暗号化基盤の選択

#### 背景にあると推察する業務課題

1. **VB6環境での暗号化機能の欠如**: Visual Basic 6には、現代のセキュリティ要件を満たす組み込み暗号化機能が存在しない。これは、VB6で開発された業務システムが、データ保護の面で脆弱になりうるリスクを意味する。

2. **独自実装のリスク**: 暗号化アルゴリズムを独自に実装した場合、実装の誤りによるセキュリティホール、タイミング攻撃への脆弱性、パフォーマンス問題など、多くのリスクが生じる。

3. **検証コストの問題**: 独自実装した暗号化ライブラリの安全性を検証するには、専門家によるセキュリティ監査が必要であり、これは中小規模のプロジェクトでは現実的でない。

#### 業務課題に対するアプローチ方針

Windows CryptoAPIを活用することで、以下の効果が期待される：

- **検証済み実装の活用**: MicrosoftがFIPS 140-2認証を取得した暗号化プロバイダを利用可能
- **自動的なセキュリティアップデート**: OS更新で暗号化実装も更新される
- **ハードウェアアクセラレーション**: 対応CPUでのAES-NI等の自動活用

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| 独自VB6実装 | VB6でアルゴリズムを直接実装 | 外部依存なし | セキュリティリスク高、低速 |
| 外部DLL（OpenSSL等） | 既存暗号化ライブラリをラップ | 高機能 | 配布複雑、ライセンス問題 |
| Windows CryptoAPI（採用） | OS標準暗号化APIを利用 | 検証済み、配布不要 | Windows依存 |
| .NET Framework呼び出し | .NETの暗号化機能をCOM経由で利用 | 高機能 | .NET依存、パフォーマンス |

#### 採用した方針

**Windows CryptoAPIベースの実装**を採用。VB6からCryptoAPI関数を呼び出し、OSの暗号化プロバイダを活用する設計とした。

**根拠となるコード**:
- `RSACryptoServiceProvider.cls`: `CryptGenKey`, `CryptEncrypt`, `CryptDecrypt`等のCryptoAPI関数を直接呼び出し
- `RNGCryptoServiceProvider.cls`: `CryptGenRandom`による暗号学的乱数生成
- `CspParameters.cls`: CSPプロバイダタイプ・フラグの管理

#### 選択理由（現状維持・別方式との比較必須）

1. **「独自実装を採用する」を不採用とした理由**:
   - 暗号化アルゴリズムの独自実装は、サイドチャネル攻撃（タイミング攻撃、電力解析攻撃）への脆弱性を生む可能性が高い
   - NIST等の認証取得に膨大な時間とコストが必要
   - セキュリティホール発見時の修正・検証サイクルを自社で維持する必要がある

2. **「OpenSSL等の外部DLLを利用する」を不採用とした理由**:
   - ライセンス条項（OpenSSL License）の遵守が複雑
   - DLLのバージョン管理・配布が煩雑
   - VB6アプリケーションとの統合でマーシャリングオーバーヘッドが発生

3. **Windows CryptoAPIを採用した理由**:
   - Windowsに標準搭載されており追加配布不要
   - FIPS 140-2 Level 1認証を取得したプロバイダが利用可能
   - Microsoft Security Response Centerによる継続的な脆弱性対応
   - VB6のDeclare文で直接呼び出し可能な設計

#### 不採用とした選択肢の理由

- **独自VB6実装**: セキュリティリスクが容認できないレベル。業務データ保護の責任を全うできない
- **外部DLL**: 配布・保守の複雑さが業務運用コストを増大させる
- **.NET経由**: .NET Framework依存が発生し、純粋なVB6環境での利用が制限される

---

### 2. 対称暗号アルゴリズムの選択

#### 背景にあると推察する業務課題

1. **アルゴリズムの陳腐化リスク**: DES（64ビット鍵）は現代では数時間で解読可能であり、機密データの保護には不適切。しかし、レガシーシステムとのデータ交換でDESが必要なケースが存在する。

2. **標準化への対応**: AES（Rijndael）は政府・金融機関で採用が義務付けられており、これをサポートしないライブラリは業務利用が困難。

3. **.NET互換性の確保**: .NET Frameworkと同じAPIシグネチャを提供することで、既存の.NETコードの知見を活用可能にする必要がある。

#### 業務課題に対するアプローチ方針

複数のアルゴリズムを提供しつつ、セキュリティ設計書で推奨・非推奨を明示することで、以下を実現：

- 新規開発ではAES-256（Rijndael）を推奨
- レガシー互換が必要な場合のみ3DES/DESを許可
- ドキュメントで移行パスを明示

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| AES（Rijndael）のみ | 推奨アルゴリズムのみ実装 | 高セキュリティ、シンプル | レガシー互換性なし |
| DES/TripleDES/RC2/Rijndael | 複数アルゴリズムを網羅 | 互換性確保 | 誤用リスク |
| 複数アルゴリズム＋推奨明示（採用） | 網羅的実装＋ガイダンス | 互換性＋セキュリティ指針 | 実装コスト |

#### 採用した方針

**Rijndael（AES）を推奨としつつ、TripleDES/DES/RC2もサポート**する設計を採用。

**デフォルト設定**（`RijndaelManaged.cls`より）:
- キーサイズ: 256ビット（`DEF_KEYSIZE`）
- ブロックサイズ: 128ビット（`DEF_BLOCKSIZE`）
- 暗号モード: CBC（`CipherMode.CBC`）
- パディング: PKCS7（`PaddingMode.PKCS7`）

#### 根拠となるコード

```vb
' RijndaelManaged.cls - Class_Initialize
Private Sub Class_Initialize()
    Set mBase = New SymmetricAlgorithmBase
    With mBase
        .AddLegalBlockSize DEF_MINBLOCKSIZE, DEF_MAXBLOCKSIZE, DEF_BLOCKSKIPSIZE
        .AddLegalKeySize DEF_MINKEYSIZE, DEF_MAXKEYSIZE, DEF_KEYSKIPSIZE
        .BlockSize = DEF_BLOCKSIZE      ' 128ビット
        .KeySize = DEF_KEYSIZE          ' 256ビット
        .FeedbackSize = DEF_FEEDBACKSIZE
        .Mode = CipherMode.CBC          ' 安全なデフォルト
        .Padding = PaddingMode.PKCS7    ' 標準パディング
    End With
End Sub
```

#### 選択理由（現状維持・別方式との比較必須）

1. **「AESのみを提供する」を不採用とした理由**:
   - 既存のVB6システムでTripleDES/DESを使用しているケースへの対応が不可能になる
   - .NET Frameworkとの完全互換を実現できない
   - 段階的な移行パスを提供できない

2. **「アルゴリズム選択のガイダンスなしで提供する」を不採用とした理由**:
   - 開発者がDESを新規システムで採用するリスクがある
   - セキュリティ監査で指摘を受ける可能性がある

3. **複数アルゴリズム＋推奨明示を採用した理由**:
   - 業界標準（AES-256）をデフォルトとして安全な選択を促進
   - レガシー互換が明示的に必要な場合のみ旧アルゴリズムを使用可能
   - セキュリティ設計書で非推奨アルゴリズムを明記し、監査対応を容易化

#### 暗号モードの設計意図

| モード | デフォルト採用 | 理由 |
| --- | --- | --- |
| CBC | はい | 同一平文の暗号文パターン漏洩を防止、業界標準 |
| ECB | いいえ（サポート） | パターン漏洩リスクあり、レガシー互換用 |
| CFB | いいえ（サポート） | ストリーム暗号的用途向け |

---

### 3. 弱鍵検出機能

#### 背景にあると推察する業務課題

DES/TripleDESには、使用すると暗号化の安全性が著しく低下する「弱鍵」「準弱鍵」が存在する。これらの鍵は：
- 弱鍵（4種）: 二重暗号化しても元に戻る
- 準弱鍵（12種）: ペアで使用すると逆暗号化効果が発生

乱数生成で偶然これらの鍵が生成される確率は極めて低いが、攻撃者が意図的に弱鍵を指定する可能性がある。

#### 業務課題に対するアプローチ方針

弱鍵・準弱鍵を検出し、使用を拒否する機能を提供することで、意図的・偶発的な脆弱性の混入を防止する。

#### 採用した方針

**`DESStatic.cls`に`IsWeakKey`/`IsSemiWeakKey`メソッドを実装**

**根拠となるコード**:
```vb
' DESStatic.cls
Public Function IsWeakKey(ByRef RgbKey() As Byte) As Boolean
    IsWeakKey = MatchesKnownKey(RgbKey, mWeakKeys)
End Function

Public Function IsSemiWeakKey(ByRef RgbKey() As Byte) As Boolean
    IsSemiWeakKey = MatchesKnownKey(RgbKey, mSemiWeakKeys)
End Function

' 既知の弱鍵リスト
Private Sub InitWeakKeys()
    ReDim mWeakKeys(3)
    mWeakKeys(0) = Parse64BitKeyString("0101010101010101")
    mWeakKeys(1) = Parse64BitKeyString("FEFEFEFEFEFEFEFE")
    mWeakKeys(2) = Parse64BitKeyString("E0E0E0E0F1F1F1F1")
    mWeakKeys(3) = Parse64BitKeyString("1F1F1F1F0E0E0E0E")
End Sub
```

#### 選択理由

1. **業界標準への準拠**: .NET Frameworkも同様の弱鍵検出機能を提供
2. **多層防御**: 乱数生成時にも弱鍵チェックを実施し、生成ループで弱鍵を排除
3. **明示的なエラー通知**: 弱鍵検出時にCryptographicExceptionをスローし、開発者に明確に通知

---

### 4. 鍵導出関数（PBKDF2）の選択

#### 背景にあると推察する業務課題

1. **パスワードから直接鍵を生成するリスク**: ユーザーパスワードをそのまま暗号化キーとして使用すると、辞書攻撃・レインボーテーブル攻撃に脆弱。

2. **業界標準への準拠必要性**: パスワードベース暗号化（PBE）には、RFC 2898（PKCS#5 v2.0）のPBKDF2が業界標準として広く採用されている。

#### 業務課題に対するアプローチ方針

PBKDF2をRfc2898DeriveBytesクラスとして実装し、以下の防御策を提供：
- **ソルト**: レインボーテーブル攻撃の無効化
- **反復処理**: ブルートフォース攻撃の計算コスト増大
- **HMAC-SHA1ベース**: 検証済みのPRF（擬似乱数関数）使用

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| パスワード直接使用 | パスワードをそのまま鍵として使用 | シンプル | 極めて脆弱 |
| 単純ハッシュ | SHA256(password)等 | 高速 | レインボーテーブル攻撃に脆弱 |
| PBKDF2（採用） | ソルト＋反復＋HMAC | 業界標準、高セキュリティ | 計算コスト（意図的） |
| Argon2/scrypt | メモリハード関数 | 最高セキュリティ | 実装複雑、CryptoAPI非対応 |

#### 採用した方針

**RFC 2898準拠のPBKDF2実装**を`Rfc2898DeriveBytes.cls`として提供。

**根拠となるコード**:
```vb
' Rfc2898DeriveBytes.cls
Private Const BlockSize As Long = 20   ' HMACSHA1のハッシュ長

Friend Sub Init(ByRef Password As Variant, ByRef Salt As Variant, ByVal Iterations As Long)
    InitHasher Password          ' HMAC-SHA1をキーとしてパスワードで初期化
    InitSalt Salt                ' 8バイト以上のソルト必須
    IterationCount = Iterations  ' 反復回数設定
End Sub

' ソルトの検証
Public Property Let Salt(ByRef Value() As Byte)
    ValidateArray Value, NameOfValue

    If Len1D(Value) < 8 Then _
        Error.Argument Cryptography_PasswordDerivedBytes_FewBytesSalt, "Value"

    mSalt = Value
    Reset
End Property
```

#### 選択理由（現状維持・別方式との比較必須）

1. **「パスワード直接使用」を不採用とした理由**:
   - 辞書攻撃で瞬時に解読される危険性
   - セキュリティ監査で必ず指摘される設計

2. **「単純ハッシュ」を不採用とした理由**:
   - ソルトなしではレインボーテーブル攻撃に脆弱
   - 反復なしではGPUによる高速解読が可能

3. **「Argon2/scrypt」を不採用とした理由**:
   - Windows CryptoAPIでネイティブサポートされていない
   - VB6での実装が複雑で検証困難
   - 現時点ではPBKDF2でも十分なセキュリティを提供可能

4. **PBKDF2を採用した理由**:
   - RFC 2898として標準化されており、広く検証済み
   - .NET FrameworkのRfc2898DeriveBytesとAPI互換
   - HMAC-SHA1ベースでCryptoAPI機能を活用可能
   - セキュリティ設計書で反復回数10,000回以上を推奨し、安全な利用を促進

---

### 5. 暗号学的乱数生成

#### 背景にあると推察する業務課題

1. **VB標準Rnd関数の脆弱性**: VB6のRnd関数は線形合同法に基づき、予測可能な擬似乱数を生成する。暗号化キー・IV・ソルトの生成には使用不可。

2. **予測可能な乱数による暗号化の無効化**: 攻撃者が乱数シードを推測できると、生成されるキー・IVを予測でき、暗号化が実質的に無効になる。

#### 業務課題に対するアプローチ方針

CryptoAPIの`CryptGenRandom`関数を使用し、OS提供の暗号学的乱数生成器を活用する。

#### 採用した方針

**`RNGCryptoServiceProvider.cls`でCryptoAPI乱数生成を提供**

**根拠となるコード**:
```vb
' RNGCryptoServiceProvider.cls
Public Sub GetBytes(ByRef Data() As Byte)
    VerifyProvider

    If CorArray.IsNull(Data) Then _
        Error.ArgumentNull "Data", ArgumentNull_Array

    If CryptGenRandom(mProvider, CorArray.Length(Data), Data(LBound(Data))) = BOOL_FALSE Then _
        Throw Cor.NewCryptographicException(GetMessage(Err.LastDllError))
End Sub
```

#### 選択理由

1. **OS標準の検証済み実装**: Windows CryptoAPIの乱数生成はFIPS 140-2認証済み
2. **エントロピー源の多様性**: ハードウェアイベント、システム状態等を活用
3. **GetNonZeroBytesの提供**: ゼロバイトを含まない乱数生成も可能（パディング攻撃対策）

---

### 6. 非対称暗号（RSA/DSA）の設計

#### 背景にあると推察する業務課題

1. **鍵交換と署名の両方が必要**: 業務システムでは、データの暗号化（RSA暗号化）とデータの完全性検証（デジタル署名）の両方が求められる。

2. **鍵サイズの将来性**: 計算能力の向上に伴い、512ビット以下のRSA鍵は既に危殆化。将来を見据えた鍵サイズのサポートが必要。

#### 業務課題に対するアプローチ方針

RSAとDSA両方を実装し、暗号化・署名・鍵交換の全ユースケースをカバーする。

#### 採用した方針

**RSACryptoServiceProvider.cls / DSACryptoServiceProvider.cls**で以下を提供：

| 機能 | RSA | DSA |
| --- | --- | --- |
| データ暗号化 | 対応（PKCS#1 v1.5 / OAEP） | 非対応 |
| デジタル署名 | 対応（MD5/SHA-1/SHA-256/384/512） | 対応 |
| 鍵サイズ | 384-16384ビット（デフォルト1024） | 可変 |

**根拠となるコード**:
```vb
' RSACryptoServiceProvider.cls - Class_Initialize
Private Sub Class_Initialize()
    ReDim mLegalKeySizes(0)
    If CryptoAPI.HasHighEncryption Then
        Set mLegalKeySizes(0) = Cor.NewKeySizes(384, 16384, 8)
        mKeySize = 1024
    Else
        Set mLegalKeySizes(0) = Cor.NewKeySizes(384, 512, 8)
        mKeySize = 512
    End If
    ' ...
End Sub
```

#### 選択理由

1. **用途の分離**: 暗号化にはRSA、署名専用システムにはDSAを使い分け可能
2. **OAEP対応**: より安全なパディング方式をサポート（PKCS#1 v1.5の脆弱性対策）
3. **柔軟な鍵サイズ**: 高暗号化システムでは最大16384ビットをサポートし、長期的な安全性を確保

---

### 7. キー管理とセキュアな消去

#### 背景にあると推察する業務課題

1. **メモリ上のキー残存リスク**: 暗号化キーがメモリ上に残存すると、メモリダンプ、スワップファイル、クラッシュダンプ等から漏洩する可能性がある。

2. **キーの永続化要件**: 一部のシステムでは、アプリケーション再起動後もキーを利用可能にする必要がある。

#### 業務課題に対するアプローチ方針

- メモリ上のキーは使用後に明示的に消去するAPIを提供
- 永続化が必要な場合はCSPキーコンテナを活用

#### 採用した方針

**Clearメソッドによる明示的消去 + PersistKeyInCspによる永続化オプション**

**根拠となるコード**:
```vb
' SymmetricAlgorithmBase.cls
Public Sub Clear()
    Erase mIV
    Erase mKey
End Sub

' RSACryptoServiceProvider.cls
Public Sub Clear()
    If Not mIsDisposed Then
        CryptoAPI.DestroyKey mKey                              ' CryptoAPIキー破棄
        CryptoAPI.ReleaseContext mProvider, Not mPersistKeyInCsp ' コンテキスト解放
        mIsDisposed = True
    End If
End Sub
```

#### 選択理由

1. **明示的なライフサイクル管理**: 開発者がキーの有効期間を制御可能
2. **CSP連携**: Windows標準のキーコンテナで安全に永続化
3. **.NET互換性**: .NETのSymmetricAlgorithm.Clear()と同等の動作

---

### 8. 入力検証とXMLセキュリティ

#### 背景にあると推察する業務課題

1. **XMLインジェクション**: RSA/DSAのキーパラメータはXML形式でインポート/エクスポート可能であり、不正なXMLによる攻撃リスクがある。

2. **バッファオーバーフロー**: 不正な入力長による配列境界外アクセスの防止が必要。

#### 業務課題に対するアプローチ方針

SecurityElementクラスでXML入力を検証し、不正なタグ・属性・テキストを拒否する。

#### 採用した方針

**SecurityElement.cls / SecurityElementStatic.cls**でXML検証機能を提供

**検証メソッド**:
- `IsValidTag`: タグ名の検証
- `IsValidText`: テキスト内容の検証
- `IsValidAttributeName`: 属性名の検証
- `IsValidAttributeValue`: 属性値の検証

**根拠となるコード**:
```vb
' SecurityElement.cls
Public Property Let Tag(ByRef Value As String)
    If Not SecurityElement.IsValidTag(Value) Then _
        Throw Cor.NewArgumentException(Environment.GetResourceString(Argument_InvalidElementTag, Value))

    mTag = Value
End Property

Public Property Let Text(ByRef Value As String)
    If Not SecurityElement.IsValidText(Value) Then _
        Throw Cor.NewArgumentException(Environment.GetResourceString(Argument_InvalidElementText, Value))

    mText = Value
End Property
```

#### 選択理由

1. **入力検証の徹底**: 全ての入力をホワイトリスト方式で検証
2. **XMLエスケープ処理**: 特殊文字（&lt; &gt; &amp; &quot; &apos;）の適切なエスケープ
3. **RSA/DSAパラメータ保護**: キーのXMLインポート時の安全性確保

---

## コンプライアンス考慮

### 準拠規格
| 規格 | 対応状況 | 関連設計 | 選択理由 |
| --- | --- | --- | --- |
| FIPS 140-2 | 間接準拠 | CryptoAPI経由でFIPS認証済みプロバイダ利用可能 | Windows CSPが認証取得済み |
| RFC 2898 (PKCS#5 v2.0) | 準拠 | Rfc2898DeriveBytesクラス | パスワードベース鍵導出の業界標準 |
| PKCS#1 | 準拠 | RSA暗号化・署名 | RSAの標準パディング・署名方式 |
| X.509 | 部分対応 | RSAParameters/DSAParametersのXMLシリアライズ | 鍵パラメータの標準形式 |

---

## 将来の拡張性考慮

### 想定される変更

1. **SHA-1の完全廃止対応**: SHA-1は衝突攻撃に対して脆弱であることが確認されており、将来的に完全廃止が見込まれる。現時点で署名用途ではSHA-256以上を推奨しているが、将来的にはSHA-1署名機能の非推奨化・削除が必要になる可能性がある。

2. **新規アルゴリズムの追加**: ChaCha20-Poly1305、Ed25519等の現代的アルゴリズムへの対応要求が発生する可能性がある。ただし、Windows CryptoAPIでのネイティブサポート状況に依存する。

3. **鍵サイズ要件の引き上げ**: 量子コンピュータの進歩に伴い、RSA-2048以上、AES-256が必須となる可能性がある。現時点でこれらは既にサポートしている。

### セキュリティ強化計画

1. **ドキュメントの継続的更新**: セキュリティ設計書の「推奨/非推奨」を、業界動向に合わせて定期的に見直す

2. **新規CSPプロバイダへの対応**: Windows新バージョンで提供される新しいCSPプロバイダを活用可能な設計を維持

3. **後方互換性の管理**: 非推奨アルゴリズムは「レガシーサポート」として明示的に残しつつ、新規利用を抑制する方針

---

## 備考

- VBCorLibはMIT Licenseの下でオープンソースとして提供されている
- Windows CryptoAPIに依存するため、Windowsプラットフォームでのみ動作する
- 本ライブラリはセキュリティ機能を「提供」するものであり、利用側アプリケーションでの適切な使用が前提となる
- セキュリティに関する重大な問題が発見された場合、速やかにアップデートを適用することを推奨する
