# データベース設計意図書

## 概要

本ドキュメントは、VBCorLibプロジェクトにおけるデータベース設計の意図・目的・設計判断の根拠を記載するものです。

**重要な結論: VBCorLibプロジェクトはデータベース機能を意図的に含まないライブラリです。**

VBCorLibは.NET Frameworkの多くのクラスをVisual Basic 6（VB6）で再実装したユーティリティライブラリであり、データベースアクセス機能を提供しないという設計判断がなされています。本ドキュメントでは、この「データベース機能を含まない」という設計判断の意図と根拠を詳細に説明します。

## 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| --- | --- | --- | --- |
| データベース機能の包含 | 含む / 含まない | **含まない** | スコープの明確化、単一責任の原則 |
| データ永続化方式 | DB / ファイル / メモリのみ | **メモリのみ（ファイルI/Oは別機能）** | ライブラリの汎用性確保 |
| 外部依存関係 | ADODB/DAO依存 / 依存なし | **依存なし** | 移植性・互換性の確保 |
| データ構造管理 | DB正規化 / インメモリ構造体 | **インメモリ構造体（Type）** | VB6ネイティブ実装による高速性 |

---

## 設計判断と根拠

### 1. データベース機能を含まないという設計判断

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| データベース機能を含む | ADODB/DAOラッパー等を実装 | ワンストップソリューション | スコープ肥大化、外部依存増加 |
| データベース機能を含まない | DB機能は利用者側で実装 | 軽量、単一責任、移植性 | 利用者が別途DB層を用意する必要 |

#### 採用した方針
**データベース機能を含まない**

#### 選択理由

本プロジェクトの分析から、以下の理由でデータベース機能を含まない設計判断がなされたと推測されます。

1. **単一責任の原則（SRP）の遵守**
   - VBCorLibは「.NET Frameworkのクラスライブラリ相当の機能をVB6で提供する」という明確な責務を持っています
   - .NET FrameworkのSystem.Data名前空間に相当するデータベースアクセス機能は、この責務の範囲外と判断されたと推測されます

2. **外部依存関係の最小化**
   - ADODB/DAOはCOMコンポーネントであり、環境依存性があります
   - ライブラリとしての移植性・再利用性を高めるため、外部依存を避けたと推測されます

3. **VB6のエコシステムとの整合性**
   - VB6環境では既にADODB/DAOが広く使用されており、新たなデータベースアクセス層を提供する必要性が低いと判断されたと推測されます

4. **ライブラリのスコープ明確化**
   - 暗号化、コレクション、テキスト処理等のユーティリティに特化することで、メンテナンス性を確保しています

#### 不採用とした選択肢の理由
- **データベース機能を含む**: VB6環境では既にADODB/DAOという成熟したデータベースアクセス手段が存在するため、車輪の再発明を避けたと推測されます。また、データベース機能の追加はライブラリの複雑性を大幅に増加させ、テスト・保守コストが増大する懸念があったと考えられます。

---

### 2. インメモリデータ構造の採用

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| データベースによる永続化 | RDBMS/ファイルDBを使用 | データ永続性、トランザクション | 外部依存、セットアップ必要 |
| ファイルによる永続化 | INI/XML/バイナリファイル | シンプル、外部依存なし | 並行アクセス、整合性管理が困難 |
| インメモリ構造体のみ | VB6のType/Classを使用 | 高速、外部依存ゼロ | アプリ終了でデータ消失 |

#### 採用した方針
**インメモリ構造体（VB6のType）を採用**

#### 選択理由

データベース設計書で確認された内部データ構造（Type定義）は、以下の理由でインメモリ構造体として設計されたと推測されます。

1. **パフォーマンス最適化**
   - VB6のType構造体はメモリ上で連続配置されるため、高速なアクセスが可能です
   - 暗号化処理やコレクション操作など、頻繁なデータアクセスが必要な処理に適しています

2. **VB6言語機能の活用**
   - VB6のTypeはC言語のstructに相当し、低レベルなメモリ操作が可能です
   - Win32 API呼び出しとの親和性が高く、パフォーマンスクリティカルな処理に適しています

3. **ライブラリとしての設計**
   - ライブラリは呼び出し元アプリケーションに永続化の責務を委ねるのが一般的です
   - データの永続化方法は利用者が選択できるようになっています

#### 主要な内部データ構造と設計意図

| 構造体名 | 所在 | 設計意図 |
| --- | --- | --- |
| EnumeratorWrapper | System.Collections/Enumeration.bas | For Each構文をサポートするための列挙子ラッパー。VB6でも.NET的なコレクション列挙を実現 |
| DelegateVTable/LocalDelegate | System/Delegation.bas | デリゲート（関数ポインタ）パターンの実装。イベント駆動・コールバック機能の基盤 |
| HashBucket | System.Collections/HashtableHelper.bas | ハッシュテーブルの衝突解決用バケット構造。O(1)検索性能の実現 |
| BigNumber | System.Numerics/BigNumberMath.bas | 任意精度整数演算のための内部表現。VB6の数値型制限を超える計算を可能に |
| BufferBucket | System/CharAllocation.bas | 効率的な文字列メモリ管理。頻繁な文字列操作のパフォーマンス最適化 |

#### 不採用とした選択肢の理由
- **データベースによる永続化**: ライブラリの目的は一時的なデータ処理であり、永続化は利用者の責務とするのが適切と判断されたと推測されます
- **ファイルによる永続化**: System.IOモジュールでファイル操作機能は別途提供されており、内部データ構造の永続化は利用者に委ねる設計と推測されます

---

### 3. 外部依存関係の設計方針

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| --- | --- | --- | --- |
| ADODB/DAO依存 | Microsoft Data Access Componentsを使用 | 機能豊富、実績あり | 環境依存、バージョン互換性問題 |
| サードパーティDB依存 | SQLite等の軽量DBを使用 | 軽量、セットアップ簡単 | DLL配布、ライセンス管理 |
| 依存なし | 自己完結型ライブラリ | 移植性最大化、配布簡単 | 機能が限定される |

#### 採用した方針
**外部データベース依存なし**

#### 選択理由

1. **配布の簡素化**
   - 外部依存がないことで、VBCorLib.dllのみの配布で完結します
   - 利用者の環境構築コストが最小化されます

2. **バージョン互換性の確保**
   - MDAC/ADO等のバージョン違いによるトラブルを回避できます
   - Windows XP〜Windows 11まで幅広い環境で動作可能と推測されます

3. **ライブラリ設計のベストプラクティス**
   - ライブラリは最小の外部依存で最大の価値を提供すべきという原則に従っています

---

### 4. 正規化レベルの選択（内部データ構造）

#### 採用した方針
**正規化の概念は適用外（インメモリ構造体のため）**

#### 設計意図

VBCorLibの内部データ構造は、関係データベースの正規化理論ではなく、以下の観点で設計されています。

1. **メモリ効率**
   - 構造体のサイズを最小化し、キャッシュヒット率を向上
   - 例: HashBucketはキーと値のペアを直接保持

2. **アクセス速度**
   - 間接参照を最小化し、直接アクセスを優先
   - 例: DelegateVTableは関数ポインタテーブルを直接保持

3. **実装の単純化**
   - VB6の言語制約の中で最もシンプルな構造を採用
   - 例: EnumeratorWrapperはIEnumeratorインターフェースの最小実装

---

## テーブル/データモデルの存在理由（業務観点）

### 内部データ構造（Type定義）の存在理由

VBCorLibにはデータベーステーブルは存在しませんが、内部データ構造の設計意図を業務観点から説明します。

#### 対応する業務要件
| 業務要件ID | 業務要件名 | 要件概要 |
| --- | --- | --- |
| BR-001 | .NET互換コレクション提供 | VB6環境で.NET Frameworkと同等のコレクション操作を可能にする |
| BR-002 | 高性能暗号化機能提供 | VB6アプリケーションで現代的な暗号化アルゴリズムを使用可能にする |
| BR-003 | 大規模数値演算提供 | VB6の数値型制限を超える演算を可能にする |

#### 想定される利用シーン
| 項目 | 内容 |
| --- | --- |
| データ発生元 | VB6アプリケーションの実行時メモリ |
| データ参照先 | 同一アプリケーション内の各モジュール |
| データの業務上の重要性 | ライブラリ機能の基盤となる一時的なデータ構造 |

#### 背景にあると推察する業務課題

| No | 推察される業務課題 | 課題の影響・深刻度 |
| --- | --- | --- |
| 1 | VB6標準ライブラリの機能不足 | 開発者が個別に同様の機能を実装する必要があり、開発効率が低下 |
| 2 | VB6アプリケーションの保守コスト増大 | レガシーシステムのメンテナンスにおいて、現代的な機能が使えず生産性が低い |
| 3 | .NET移行が困難なシステムの存在 | 移行コストやリスクにより、VB6システムを継続使用せざるを得ない状況 |

#### 業務課題に対するアプローチ方針

| 業務課題 | アプローチ方針 | 期待される効果 |
| --- | --- | --- |
| VB6標準ライブラリの機能不足 | .NET Framework相当のクラスをVB6で再実装 | 開発者が馴染みのあるAPIで高機能な処理を実装可能 |
| VB6アプリケーションの保守コスト増大 | 共通ライブラリとして提供 | 個別実装の削減、コード品質の向上 |
| .NET移行が困難なシステムの存在 | VB6環境のままで機能強化を可能に | 移行リスクを回避しつつ、機能向上を実現 |

#### 検討したデータモデル案

| 案 | 概要 | メリット | デメリット | 業務適合性 |
| --- | --- | --- | --- | --- |
| A | 手動管理継続（Excel/紙台帳） | 適用外 | 適用外 | 適用外 |
| B | データベースでの永続化 | データ永続性 | 外部依存、オーバーエンジニアリング | 低（ライブラリには不適） |
| C（採用） | インメモリ構造体 | 高速、軽量、依存なし | 永続化は利用者責務 | 高 |

#### 選択理由（業務観点）

1. **業務要件との適合性**: VBCorLibはユーティリティライブラリであり、データの永続化は利用者アプリケーションの責務です。ライブラリとしては一時的なデータ処理に特化することで、最大の汎用性を確保しています。

2. **データ整合性の観点**: インメモリ構造体は単一プロセス内で完結するため、整合性の管理が単純です。複数プロセス間のデータ共有は利用者が適切な手段（ファイル、データベース等）を選択します。

3. **将来の業務変化への対応**: 永続化方式を固定しないことで、利用者は要件に応じて最適な永続化手段を選択できます。

#### 不採用とした案の理由
- **案A（手動管理継続）**: プログラムライブラリには適用外
- **案B（データベースでの永続化）**: ライブラリの責務を超えており、外部依存を追加することで移植性が低下。利用者が既に使用しているデータベースとの統合が困難になる可能性があるため不採用。

---

## 将来の拡張性考慮

### 想定される変更

1. **System.Data相当のモジュール追加**
   - 将来的に.NET FrameworkのSystem.Data名前空間に相当する機能が追加される可能性があります
   - その場合、既存のコレクションクラス（ArrayList, Hashtable等）と統合されることが想定されます
   - 対応方針: 現在のインメモリ構造体設計を維持しつつ、DataTable/DataSet相当のクラスを追加する形での拡張が可能

2. **シリアライゼーション機能の強化**
   - 現在のSystem.Runtime.Serializationモジュールを拡張し、データベース連携用のシリアライゼーション機能が追加される可能性があります
   - 対応方針: 既存のシリアライゼーション基盤を活用した拡張

### スケーラビリティ

VBCorLibはライブラリであり、データベースのシャーディングやパーティショニングは適用外です。ただし、以下の点で拡張性が考慮されています。

1. **コレクションクラスのスケーラビリティ**
   - ArrayList, Hashtable等は動的にサイズが拡張可能
   - メモリ許容量の範囲内で大量データの処理が可能

2. **BigInteger演算のスケーラビリティ**
   - 任意精度の整数演算をサポート
   - メモリ許容量の範囲内で非常に大きな数値の計算が可能

---

## 備考

### VBCorLibをデータベースアプリケーションで活用する方法

VBCorLibはデータベース機能を含みませんが、以下の方法でデータベースアプリケーションと組み合わせて活用できます。

1. **ADODBとの組み合わせ**
   ```vb
   ' VBCorLibのArrayListでRecordsetデータを管理
   Dim list As New ArrayList
   Do Until rs.EOF
       list.Add rs.Fields("CustomerName").Value
       rs.MoveNext
   Loop
   ```

2. **暗号化機能との組み合わせ**
   - データベースに保存する前にVBCorLibの暗号化機能でデータを暗号化
   - 取得時に復号化してアプリケーションで使用

3. **BigIntegerとの組み合わせ**
   - データベースでは格納できない大きな数値の計算をVBCorLib側で実行
   - 結果を文字列等に変換してデータベースに保存

### 本ドキュメントの適用範囲

本ドキュメントは、VBCorLibプロジェクトのソースコード分析に基づいて作成されています。「データベース機能を含まない」という設計判断の意図を説明することに主眼を置いています。将来的にデータベース関連機能が追加された場合は、本ドキュメントを更新する必要があります。
