# 機能設計意図書

## 概要

本書は、VBCorLibライブラリの各機能が**なぜ実装されるに至ったか**を業務課題起点で記述する。
**各機能が存在する業務上の理由**を明らかにすることを目的とし、技術的な設計判断は別途アーキテクチャ設計意図書で記載している。

## 1. ドメイン概要

### 対象ドメイン

#### 業界・業種

ソフトウェア開発業界 - VB6/VBAアプリケーション開発

VBCorLibは、Visual Basic 6およびVBA環境で.NET Frameworkと互換性のあるクラスライブラリを提供する開発基盤である。業界横断的に利用可能であり、特にレガシーVB6システムを運用している企業（金融、製造、小売など）において価値を発揮する。

#### 主なステークホルダー

- **VB6/VBAアプリケーション開発者**: 日常の開発業務でライブラリを使用
- **システム移行担当者**: VB6から.NETへの段階的移行を計画・実行
- **運用保守担当者**: 既存VB6システムの機能拡張・保守
- **技術リーダー/アーキテクト**: 技術選定・設計方針決定

#### システムの位置づけ

VB6/VBA環境における.NET互換基盤ライブラリ

VBCorLibは、レガシーVB6/VBA環境と現代的な.NET Framework環境の橋渡しを行う中間層として機能する。既存VB6資産を活用しながら、.NET互換のAPIを提供することで、将来の.NET移行を見据えた開発を可能にする。

### このドメインにおける一般的な課題

#### 1. VB6/VBA標準機能の制約

**発生背景**

VB6/VBAの標準ランタイムは1998年に設計されたものであり、現代のアプリケーション開発で必要とされる多くの機能が欠落している。日付時刻処理の精度不足（ミリ秒未対応）、コレクションの種類制限、暗号化機能の欠如、文字エンコーディングの不十分なサポートなどが代表例である。

**業務への影響**

- 独自実装の増加によるコードの重複・品質低下
- 外部ライブラリ依存による保守性の低下
- .NETシステムとのデータ連携における互換性問題
- セキュリティ要件を満たせないシステムの発生

#### 2. .NETとのデータ互換性の欠如

**発生背景**

VB6システムと.NETシステムが混在する環境では、日付時刻、数値、文字列などの基本データ型の表現方法が異なるため、システム間連携時にデータ変換が必要となる。

**業務への影響**

- データ変換処理の手動実装が必要
- 変換エラーによるデータ不整合リスク
- 開発・保守工数の増加
- システム統合プロジェクトの遅延

#### 3. 国際化・多言語対応の困難さ

**発生背景**

VB6の標準機能では、複数の文字エンコーディング、カルチャ固有の書式設定、多言語リソース管理などの国際化対応が困難である。

**業務への影響**

- グローバル展開時のシステム改修コスト増大
- 文字化けによるデータ破損リスク
- カルチャ依存の書式設定（日付、通貨）の手動実装

#### 4. セキュリティ要件への対応困難

**発生背景**

VB6標準には暗号化・ハッシュ計算・デジタル署名などのセキュリティ機能がなく、Windows APIの直接呼び出しや外部ライブラリに依存する必要がある。

**業務への影響**

- コンプライアンス要件（PCI-DSS、GDPR等）への対応困難
- 機密データの安全な取り扱いが困難
- セキュリティ実装の属人化

## 2. 業務プロセス別の機能と想定価値

### 業務プロセス1: データ構造・コレクション操作

#### プロセス概要

**業務フロー**

データ取得 → データ構造化 → 検索・フィルタ → ソート → 処理・出力

**関係者**

アプリケーション開発者

**発生頻度**

非常に高い（ほぼすべてのアプリケーションで使用）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| VB6標準機能の制約 | VB6のCollectionは機能が限定的で、高速検索やソートができない |
| .NETとのデータ互換性 | .NETのArrayList/Hashtableと互換性のあるデータ交換ができない |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 独自コレクション実装に1機能あたり8〜16時間、年間50機能で400〜800時間
- **品質リスク**: 独自実装によるバグ混入、パフォーマンス低下
- **保守コスト**: 複数の独自実装の維持管理に年間100時間以上

#### 本業務をシステム化することの想定効果

- 標準化されたAPI使用による開発効率向上（独自実装工数ゼロ化）
- .NET互換データ構造による連携容易化
- テスト済みライブラリ使用による品質向上

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| ArrayList | 動的配列の管理（追加・削除・ソート・検索） | 独自実装に16時間/機能 | 標準API使用で実装工数90%削減 |
| Hashtable | キー・値ペアの高速検索・管理 | 独自実装に16時間/機能 | O(1)検索による処理高速化 |
| Stack | LIFO構造のデータ管理 | 独自実装に8時間/機能 | 標準API使用で実装工数90%削減 |
| Queue | FIFO構造のデータ管理 | 独自実装に8時間/機能 | 標準API使用で実装工数90%削減 |
| SortedList | ソート済みキー・値ペア管理 | 独自実装に24時間/機能 | 自動ソート機能による処理簡素化 |
| BitArray | ビット単位のフラグ管理 | 独自実装に8時間/機能 | メモリ効率的なフラグ管理 |
| CorArray | 配列の検索・ソート・コピー | 手動ループ実装に毎回30分 | 標準APIで1行実装可能 |

### 業務プロセス2: データ暗号化・セキュリティ

#### プロセス概要

**業務フロー**

データ入力 → 暗号化/ハッシュ化 → 保存/送信 → 復号化/検証 → 利用

**関係者**

セキュリティ担当者、アプリケーション開発者、運用保守担当者

**発生頻度**

高い（機密データを扱うすべてのシステムで必要）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| セキュリティ要件への対応困難 | VB6標準に暗号化機能がなく、機密データ保護が困難 |
| VB6標準機能の制約 | Windows API直接呼び出しは複雑で実装ミスのリスクが高い |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 暗号化API呼び出しの独自実装に40〜80時間/アルゴリズム
- **セキュリティリスク**: 実装ミスによる脆弱性、監査指摘リスク
- **コンプライアンスリスク**: PCI-DSS、GDPR等の要件不適合
- **法的リスク**: 機密データ漏洩時の損害賠償・信用失墜

#### 本業務をシステム化することの想定効果

- 標準化された暗号化APIによる実装ミス防止
- コンプライアンス要件への迅速な対応
- セキュリティ監査での適合証明が容易

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| RijndaelManaged | AES暗号化・復号化 | 独自実装に80時間 | 標準API使用で2時間で実装完了 |
| TripleDESCryptoServiceProvider | 3DES暗号化・復号化 | 独自実装に60時間 | レガシーシステム連携に即対応 |
| DESCryptoServiceProvider | DES暗号化・復号化 | 独自実装に40時間 | レガシーデータの復号対応 |
| RC2CryptoServiceProvider | RC2暗号化・復号化 | 独自実装に40時間 | 特定システム互換性確保 |
| RSACryptoServiceProvider | RSA公開鍵暗号化・署名 | 独自実装に100時間 | PKIベースのセキュリティ実装 |
| DSACryptoServiceProvider | DSAデジタル署名 | 独自実装に80時間 | デジタル署名による改ざん検知 |
| SHA1Managed | SHA-1ハッシュ計算 | 独自実装に40時間 | データ整合性検証 |
| SHA256Managed | SHA-256ハッシュ計算 | 独自実装に40時間 | 強力なハッシュによるセキュリティ確保 |
| SHA384Managed | SHA-384ハッシュ計算 | 独自実装に40時間 | 高セキュリティ要件対応 |
| SHA512Managed | SHA-512ハッシュ計算 | 独自実装に40時間 | 最高強度のハッシュ計算 |
| MD5CryptoServiceProvider | MD5ハッシュ計算 | 独自実装に40時間 | レガシーシステム互換性 |
| RIPEMD160Managed | RIPEMD-160ハッシュ計算 | 独自実装に40時間 | 特定規格対応 |
| HMACSHA1 | SHA-1ベースHMAC認証 | 独自実装に50時間 | メッセージ認証 |
| HMACSHA256 | SHA-256ベースHMAC認証 | 独自実装に50時間 | 強力なメッセージ認証 |
| HMACSHA384 | SHA-384ベースHMAC認証 | 独自実装に50時間 | 高セキュリティメッセージ認証 |
| HMACSHA512 | SHA-512ベースHMAC認証 | 独自実装に50時間 | 最高強度メッセージ認証 |
| HMACMD5 | MD5ベースHMAC認証 | 独自実装に50時間 | レガシー互換メッセージ認証 |
| HMACRIPEMD160 | RIPEMD-160ベースHMAC認証 | 独自実装に50時間 | 特定規格対応 |
| Rfc2898DeriveBytes | パスワードからの鍵導出 | 独自実装に60時間 | 安全なパスワードベース暗号化 |
| CryptoStream | ストリーム暗号化 | 独自実装に30時間 | 大容量データの効率的暗号化 |
| ToBase64Transform | Base64エンコード変換 | 独自実装に8時間 | データ変換処理効率化 |
| FromBase64Transform | Base64デコード変換 | 独自実装に8時間 | データ変換処理効率化 |

### 業務プロセス3: テキスト・エンコーディング処理

#### プロセス概要

**業務フロー**

データ入力/取得 → 文字コード判定 → エンコード/デコード → 処理 → 保存/送信

**関係者**

アプリケーション開発者、データ連携担当者

**発生頻度**

非常に高い（ファイル処理・通信処理を行う全システム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| 国際化・多言語対応の困難さ | VB6標準では多様なエンコーディングに対応困難 |
| .NETとのデータ互換性 | .NETシステムとのテキストデータ交換で文字化け発生リスク |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: エンコーディング処理の独自実装に20〜40時間/エンコーディング
- **データ品質リスク**: 文字化けによるデータ破損
- **顧客対応コスト**: 文字化け問題の調査・修正に1件あたり2〜8時間

#### 本業務をシステム化することの想定効果

- 多様なエンコーディングの統一的なAPI提供
- 文字化け問題の大幅削減
- 国際化対応の容易化

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| Encoding | エンコーディング基盤 | 独自実装に40時間 | 統一的なエンコーディングAPI |
| UTF8Encoding | UTF-8エンコード・デコード | 独自実装に20時間 | Webシステム連携の標準化 |
| UTF7Encoding | UTF-7エンコード・デコード | 独自実装に20時間 | メール処理対応 |
| UnicodeEncoding | UTF-16エンコード・デコード | 独自実装に20時間 | Windows内部文字列との互換 |
| UTF32Encoding | UTF-32エンコード・デコード | 独自実装に20時間 | 特殊文字完全対応 |
| ASCIIEncoding | ASCIIエンコード・デコード | 独自実装に10時間 | 基本テキスト処理 |
| CodePageEncoding | Windowsコードページ対応 | 独自実装に40時間 | レガシーシステム連携 |
| StringBuilder | 効率的な文字列連結 | 手動実装に10時間 | 大量文字列処理の高速化 |
| CorString | 文字列ユーティリティ | 手動実装に20時間 | 文字列操作の効率化 |

### 業務プロセス4: ファイル・ストリーム操作

#### プロセス概要

**業務フロー**

ファイル指定 → 読み込み/書き込み → データ処理 → クローズ → 結果確認

**関係者**

アプリケーション開発者、バッチ処理担当者

**発生頻度**

非常に高い（ファイル処理を行う全システム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| VB6標準機能の制約 | VB6標準のファイル操作は2GB制限あり、機能も限定的 |
| .NETとのデータ互換性 | ファイル属性・タイムスタンプ形式の互換性問題 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: ファイル操作の拡張実装に30〜60時間
- **機能制約リスク**: 2GB超ファイルが処理できない
- **保守コスト**: 複数の独自実装の維持管理

#### 本業務をシステム化することの想定効果

- 2GB超ファイル対応による機能制約解消
- .NET互換のファイル操作APIによる移行容易化
- 統一的なストリームAPIによる処理の標準化

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| File | ファイルの作成・コピー・削除・読み書き | 独自実装に20時間 | 統一的なファイル操作API |
| FileInfo | ファイル情報の取得・設定 | 独自実装に10時間 | ファイル属性管理の簡素化 |
| FileStream | ファイルストリーム操作 | 独自実装に30時間 | 2GB超ファイル対応 |
| Directory | ディレクトリ操作 | 独自実装に15時間 | ディレクトリ操作の標準化 |
| DirectoryInfo | ディレクトリ情報管理 | 独自実装に10時間 | ディレクトリ属性管理 |
| Path | ファイルパス操作 | 独自実装に10時間 | パス操作の安全性向上 |
| DriveInfo | ドライブ情報取得 | 独自実装に8時間 | ドライブ容量監視 |
| MemoryStream | メモリ上のストリーム操作 | 独自実装に15時間 | メモリ内データ処理 |
| BinaryReader | バイナリデータ読み込み | 独自実装に15時間 | 構造化データ読み込み |
| BinaryWriter | バイナリデータ書き込み | 独自実装に15時間 | 構造化データ書き込み |
| StreamReader | テキスト読み込み | 独自実装に15時間 | エンコーディング対応読み込み |
| StreamWriter | テキスト書き込み | 独自実装に15時間 | エンコーディング対応書き込み |
| StringReader | 文字列からの読み込み | 独自実装に8時間 | 文字列のストリーム処理 |
| StringWriter | 文字列への書き込み | 独自実装に8時間 | 文字列のストリーム生成 |
| MemoryMappedFile | メモリマップトファイル | 独自実装に40時間 | 大容量ファイルの効率的処理 |
| MemoryMappedViewAccessor | メモリマップアクセス | 独自実装に20時間 | 高速ランダムアクセス |
| IniResourceWriter | INIファイル操作 | 独自実装に15時間 | 設定ファイル管理 |

### 業務プロセス5: 日付時刻・グローバリゼーション処理

#### プロセス概要

**業務フロー**

日付時刻入力 → 形式変換 → 計算・比較 → 書式設定 → 表示・保存

**関係者**

アプリケーション開発者、国際化担当者

**発生頻度**

非常に高い（日付時刻を扱う全システム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| VB6標準機能の制約 | VB6のDate型はミリ秒未対応、範囲制限あり |
| 国際化・多言語対応の困難さ | タイムゾーン・カレンダー・書式の国際化対応が困難 |
| .NETとのデータ互換性 | 日付時刻形式の違いによる連携問題 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 高精度日付時刻処理の独自実装に60〜100時間
- **データ品質リスク**: 精度不足によるタイムスタンプ不整合
- **国際化コスト**: 各国カレンダー対応に1カレンダーあたり40時間

#### 本業務をシステム化することの想定効果

- ミリ秒精度の日付時刻処理
- タイムゾーン変換の標準化
- 多様なカレンダーシステム対応

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| CorDateTime | 高精度日付時刻管理 | 独自実装に60時間 | ミリ秒精度・.NET互換 |
| TimeSpan | 時間間隔管理 | 独自実装に30時間 | 時間計算の標準化 |
| TimeZone | タイムゾーン変換 | 独自実装に40時間 | 国際時刻対応 |
| CultureInfo | カルチャ情報管理 | 独自実装に50時間 | ロケール対応の一元管理 |
| DateTimeFormatInfo | 日時書式情報 | 独自実装に30時間 | カルチャ依存書式設定 |
| NumberFormatInfo | 数値書式情報 | 独自実装に30時間 | カルチャ依存数値書式 |
| GregorianCalendar | グレゴリオ暦 | 独自実装に20時間 | 標準暦計算 |
| JapaneseCalendar | 和暦 | 独自実装に40時間 | 日本向けシステム対応 |
| KoreanCalendar | 韓国暦 | 独自実装に30時間 | 韓国向けシステム対応 |
| TaiwanCalendar | 台湾暦（民国暦） | 独自実装に30時間 | 台湾向けシステム対応 |
| ThaiBuddhistCalendar | タイ仏暦 | 独自実装に30時間 | タイ向けシステム対応 |
| HijriCalendar | イスラム暦 | 独自実装に40時間 | 中東向けシステム対応 |
| HebrewCalendar | ヘブライ暦 | 独自実装に40時間 | イスラエル向けシステム対応 |
| JulianCalendar | ユリウス暦 | 独自実装に30時間 | 歴史的日付処理 |

### 業務プロセス6: 数値計算・乱数生成

#### プロセス概要

**業務フロー**

数値入力 → 計算処理 → 結果検証 → 出力

**関係者**

アプリケーション開発者、業務担当者

**発生頻度**

高い（数値計算を行う全システム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| VB6標準機能の制約 | 大整数演算・64ビット整数のサポートがない |
| セキュリティ要件への対応困難 | 暗号論的乱数生成機能がない |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 大整数演算の独自実装に100時間以上
- **計算精度リスク**: オーバーフローによる計算誤り
- **セキュリティリスク**: 弱い乱数使用による脆弱性

#### 本業務をシステム化することの想定効果

- 任意精度の整数演算
- 64ビット整数のサポート
- 暗号論的に安全な乱数生成

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| BigInteger | 任意精度整数演算 | 独自実装に100時間 | 大規模計算対応 |
| Int64 | 64ビット整数演算 | 独自実装に40時間 | .NET互換数値処理 |
| Random | 疑似乱数生成 | 独自実装に20時間 | 再現可能な乱数生成 |
| RNGCryptoServiceProvider | 暗号論的乱数生成 | 独自実装に40時間 | セキュアな乱数生成 |
| CorMath | 数学関数 | 独自実装に15時間 | 数学計算の標準化 |
| BitConverter | バイト配列変換 | 独自実装に15時間 | データ型変換効率化 |
| Convert | データ型変換 | 独自実装に20時間 | 汎用型変換 |
| Buffer | バッファ操作 | 独自実装に10時間 | メモリ操作効率化 |
| Char | 文字操作 | 独自実装に10時間 | 文字単位処理 |

### 業務プロセス7: リソース・設定管理

#### プロセス概要

**業務フロー**

リソース定義 → リソース読み込み → カルチャ選択 → リソース取得 → 表示

**関係者**

アプリケーション開発者、ローカライズ担当者

**発生頻度**

高い（多言語・設定管理を行う全システム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| 国際化・多言語対応の困難さ | カルチャ別リソース管理の仕組みがない |
| VB6標準機能の制約 | レジストリ操作・リソース管理が煩雑 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 多言語リソース管理の独自実装に80時間
- **保守コスト**: 言語追加時の改修に1言語あたり20時間
- **品質リスク**: リソース管理の不整合

#### 本業務をシステム化することの想定効果

- カルチャフォールバックによる多言語対応
- .NET互換リソース形式のサポート
- 統一的なレジストリ・設定管理API

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| ResourceManager | リソースの一元管理 | 独自実装に50時間 | 多言語リソース管理 |
| ResourceReader | リソースファイル読み込み | 独自実装に30時間 | .RESファイル対応 |
| ResourceWriter | リソースファイル書き込み | 独自実装に30時間 | リソースファイル生成 |
| ResourceSet | カルチャ別リソースセット | 独自実装に20時間 | カルチャ別管理 |
| RegistryKey | レジストリキー操作 | 独自実装に30時間 | 設定の永続化 |
| Guid | GUID生成・操作 | 独自実装に20時間 | 一意識別子管理 |
| Version | バージョン管理 | 独自実装に10時間 | バージョン比較 |
| Environment | 環境情報取得 | 独自実装に15時間 | システム情報アクセス |

### 業務プロセス8: スレッド・同期処理

#### プロセス概要

**業務フロー**

タスク定義 → スレッド制御 → 同期待機 → 結果取得

**関係者**

アプリケーション開発者

**発生頻度**

中程度（並行処理を行うシステム）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| VB6標準機能の制約 | VB6のスレッド制御機能が限定的 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: スレッド制御の独自実装に40時間
- **品質リスク**: デッドロック・レースコンディションのリスク

#### 本業務をシステム化することの想定効果

- 安全なスレッドスリープ・待機機能
- 高精度時間計測
- .NET互換の同期プリミティブ

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| Thread | スレッドスリープ・待機 | 独自実装に20時間 | 安全なスレッド制御 |
| WaitHandle | 待機ハンドル操作 | 独自実装に30時間 | 同期プリミティブ |
| Ticker | 高精度時間計測 | 独自実装に15時間 | パフォーマンス計測 |

### 業務プロセス9: 例外・エラー処理

#### プロセス概要

**業務フロー**

処理実行 → エラー検出 → 例外生成 → エラーハンドリング → ログ・通知

**関係者**

アプリケーション開発者、運用保守担当者

**発生頻度**

非常に高い（全システムで必要）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| .NETとのデータ互換性 | VB6のエラー処理と.NETの例外処理の互換性がない |
| VB6標準機能の制約 | 構造化された例外階層がない |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 例外クラス階層の独自実装に50時間
- **保守コスト**: エラー情報の不足による調査工数増加
- **品質リスク**: エラー処理の不統一

#### 本業務をシステム化することの想定効果

- .NET互換の例外階層
- スタックトレース・内部例外のサポート
- 統一的なエラーハンドリング

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| Exception | 例外基底クラス | 独自実装に30時間 | .NET互換例外処理 |
| ArgumentException | 引数例外 | 独自実装に5時間 | 引数エラーの標準化 |
| ArgumentNullException | Null引数例外 | 独自実装に5時間 | Nullチェックの標準化 |
| ArgumentOutOfRangeException | 範囲外引数例外 | 独自実装に5時間 | 範囲エラーの標準化 |
| IOException | IO例外 | 独自実装に10時間 | ファイルエラーの標準化 |
| FileNotFoundException | ファイル未検出例外 | 独自実装に5時間 | ファイル不在エラーの標準化 |
| DirectoryNotFoundException | ディレクトリ未検出例外 | 独自実装に5時間 | ディレクトリ不在エラーの標準化 |
| FormatException | 書式例外 | 独自実装に5時間 | 書式エラーの標準化 |
| InvalidOperationException | 無効操作例外 | 独自実装に5時間 | 状態エラーの標準化 |
| InvalidCastException | キャスト例外 | 独自実装に5時間 | 型変換エラーの標準化 |
| ObjectDisposedException | 破棄済みオブジェクト例外 | 独自実装に5時間 | リソース管理エラーの標準化 |

### 業務プロセス10: オブジェクト基盤・インターフェース

#### プロセス概要

**業務フロー**

オブジェクト生成 → インターフェース実装 → 比較・複製 → 書式設定

**関係者**

アプリケーション開発者

**発生頻度**

非常に高い（オブジェクト指向開発の基盤）

#### 関連するドメイン課題

| ドメイン課題 | この業務プロセスでの具体的な影響 |
| ------------ | -------------------------------- |
| .NETとのデータ互換性 | オブジェクト比較・複製の互換性がない |
| VB6標準機能の制約 | 標準的なオブジェクトインターフェースがない |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **工数コスト**: 共通インターフェースの独自設計に40時間
- **保守コスト**: 不統一なインターフェースの維持管理
- **品質リスク**: オブジェクト操作の不整合

#### 本業務をシステム化することの想定効果

- .NET互換のオブジェクトインターフェース
- 統一的な比較・複製・書式設定
- コレクションとの相互運用性

#### 主な関連機能と業務インパクト

| 機能名 | 何をする機能か | 機能がない場合の運用工数 | 機能があることによるインパクト |
| ------ | ---------------------- | ------------------------ | ------------------------------ |
| IObject | オブジェクト基底インターフェース | 独自設計に20時間 | 共通操作の標準化 |
| IComparable | 比較可能インターフェース | 独自設計に10時間 | ソート・比較の標準化 |
| IComparer | 比較ロジックインターフェース | 独自設計に10時間 | カスタム比較の標準化 |
| IEqualityComparer | 等価比較インターフェース | 独自設計に10時間 | ハッシュ・等価の標準化 |
| ICloneable | 複製可能インターフェース | 独自設計に10時間 | オブジェクト複製の標準化 |
| IFormattable | 書式設定可能インターフェース | 独自設計に10時間 | カスタム書式の標準化 |
| IFormatProvider | 書式プロバイダインターフェース | 独自設計に10時間 | カルチャ依存書式の標準化 |
| Comparer | デフォルト比較実装 | 独自実装に15時間 | 汎用比較ロジック |
| EqualityComparer | デフォルト等価比較実装 | 独自実装に15時間 | 汎用等価比較ロジック |
| Console | コンソール入出力 | 独自実装に20時間 | コンソールアプリ開発 |

## 3. 各機能想定意図サマリー

| 機能名 | 背景にある業務課題 | 業務負荷（推察） | 主な解決アプローチ | 手動運用を選ばなかった理由 |
| --- | --- | --- | --- | --- |
| ArrayList | VB6のCollectionは機能不足 | 高（頻繁に使用） | 動的配列・ソート・検索API | 独自実装は品質・保守性に問題 |
| Hashtable | 高速なキー検索機構がない | 高（頻繁に使用） | ハッシュベース検索 | O(1)検索は独自実装困難 |
| Stack | LIFO構造の標準実装がない | 中（特定処理で使用） | Push/Pop API | 独自実装は冗長 |
| Queue | FIFO構造の標準実装がない | 中（特定処理で使用） | Enqueue/Dequeue API | 独自実装は冗長 |
| SortedList | 自動ソートコレクションがない | 中（特定処理で使用） | キーソート済みコレクション | 手動ソートは非効率 |
| BitArray | ビット操作が煩雑 | 低（特定処理で使用） | ビット配列API | 独自実装はエラー発生リスク |
| CorArray | 配列操作の標準APIがない | 高（頻繁に使用） | 検索・ソート・コピーAPI | 毎回のループ実装は非効率 |
| RijndaelManaged | AES暗号化機能がない | 高（セキュリティ必須） | .NET互換AES実装 | セキュリティ実装ミスは致命的 |
| TripleDESCryptoServiceProvider | 3DES暗号化機能がない | 中（レガシー連携） | .NET互換3DES実装 | 独自実装は脆弱性リスク |
| DESCryptoServiceProvider | DES暗号化機能がない | 中（レガシー連携） | .NET互換DES実装 | 独自実装は脆弱性リスク |
| RC2CryptoServiceProvider | RC2暗号化機能がない | 低（特定連携） | .NET互換RC2実装 | 独自実装は脆弱性リスク |
| RSACryptoServiceProvider | 公開鍵暗号機能がない | 高（PKI必須） | .NET互換RSA実装 | 独自実装は極めて困難 |
| DSACryptoServiceProvider | デジタル署名機能がない | 中（署名必須） | .NET互換DSA実装 | 独自実装は極めて困難 |
| SHA1Managed | SHA-1ハッシュ機能がない | 高（整合性検証） | .NET互換SHA-1実装 | 独自実装はエラーリスク |
| SHA256Managed | SHA-256ハッシュ機能がない | 高（整合性検証） | .NET互換SHA-256実装 | 独自実装はエラーリスク |
| SHA384Managed | SHA-384ハッシュ機能がない | 中（高セキュリティ） | .NET互換SHA-384実装 | 独自実装はエラーリスク |
| SHA512Managed | SHA-512ハッシュ機能がない | 中（高セキュリティ） | .NET互換SHA-512実装 | 独自実装はエラーリスク |
| MD5CryptoServiceProvider | MD5ハッシュ機能がない | 中（レガシー連携） | .NET互換MD5実装 | 独自実装はエラーリスク |
| RIPEMD160Managed | RIPEMD-160機能がない | 低（特定規格） | .NET互換RIPEMD-160実装 | 独自実装はエラーリスク |
| HMACSHA1 | HMAC-SHA1機能がない | 高（認証必須） | .NET互換HMAC-SHA1実装 | 独自実装は脆弱性リスク |
| HMACSHA256 | HMAC-SHA256機能がない | 高（認証必須） | .NET互換HMAC-SHA256実装 | 独自実装は脆弱性リスク |
| HMACSHA384 | HMAC-SHA384機能がない | 中（高セキュリティ） | .NET互換HMAC-SHA384実装 | 独自実装は脆弱性リスク |
| HMACSHA512 | HMAC-SHA512機能がない | 中（高セキュリティ） | .NET互換HMAC-SHA512実装 | 独自実装は脆弱性リスク |
| HMACMD5 | HMAC-MD5機能がない | 中（レガシー連携） | .NET互換HMAC-MD5実装 | 独自実装は脆弱性リスク |
| HMACRIPEMD160 | HMAC-RIPEMD160機能がない | 低（特定規格） | .NET互換HMAC-RIPEMD160実装 | 独自実装は脆弱性リスク |
| Rfc2898DeriveBytes | PBKDF2機能がない | 高（パスワード暗号化） | .NET互換PBKDF2実装 | 独自実装は脆弱性リスク |
| CryptoStream | ストリーム暗号化機能がない | 高（大容量暗号化） | ストリーム変換API | 独自実装は複雑 |
| ToBase64Transform | Base64変換機能が限定的 | 高（データ交換） | ストリームベース変換 | 独自実装は非効率 |
| FromBase64Transform | Base64変換機能が限定的 | 高（データ交換） | ストリームベース変換 | 独自実装は非効率 |
| Encoding | エンコーディング基盤がない | 高（全システム） | 統一エンコーディングAPI | 独自実装は品質問題 |
| UTF8Encoding | UTF-8対応が不十分 | 高（Web連携） | .NET互換UTF-8実装 | 文字化けリスク回避 |
| UTF7Encoding | UTF-7対応がない | 低（メール処理） | .NET互換UTF-7実装 | 特定処理対応 |
| UnicodeEncoding | UTF-16対応が不十分 | 高（Windows連携） | .NET互換UTF-16実装 | 内部文字列互換 |
| UTF32Encoding | UTF-32対応がない | 低（特殊文字） | .NET互換UTF-32実装 | 完全Unicode対応 |
| ASCIIEncoding | ASCII処理の標準化 | 中（基本処理） | .NET互換ASCII実装 | 処理の標準化 |
| CodePageEncoding | コードページ対応が煩雑 | 中（レガシー連携） | Windowsコードページ対応 | レガシーシステム連携 |
| StringBuilder | 文字列連結が非効率 | 高（大量文字列処理） | 可変長バッファ | パフォーマンス向上 |
| CorString | 文字列操作が煩雑 | 高（頻繁に使用） | 文字列ユーティリティ | コード簡素化 |
| File | ファイル操作APIが不十分 | 高（頻繁に使用） | .NET互換ファイルAPI | 2GB超対応・標準化 |
| FileInfo | ファイル情報取得が煩雑 | 中（ファイル管理） | オブジェクト指向ファイル情報 | コード簡素化 |
| FileStream | ファイルストリームが2GB制限 | 高（大容量処理） | 2GB超対応ストリーム | 大容量ファイル対応 |
| Directory | ディレクトリ操作が煩雑 | 高（頻繁に使用） | .NET互換ディレクトリAPI | コード簡素化 |
| DirectoryInfo | ディレクトリ情報取得が煩雑 | 中（ディレクトリ管理） | オブジェクト指向ディレクトリ情報 | コード簡素化 |
| Path | パス操作が煩雑・エラー多発 | 高（頻繁に使用） | .NET互換パスAPI | パス処理の安全性 |
| DriveInfo | ドライブ情報取得が煩雑 | 中（容量監視） | .NET互換ドライブAPI | 監視処理の標準化 |
| MemoryStream | メモリストリームがない | 高（メモリ内処理） | .NET互換メモリストリーム | ストリームAPI統一 |
| BinaryReader | バイナリ読み込みが煩雑 | 高（構造化データ） | .NET互換バイナリリーダー | データ読み込み効率化 |
| BinaryWriter | バイナリ書き込みが煩雑 | 高（構造化データ） | .NET互換バイナリライター | データ書き込み効率化 |
| StreamReader | テキスト読み込みが煩雑 | 高（テキスト処理） | エンコーディング対応リーダー | 文字コード対応 |
| StreamWriter | テキスト書き込みが煩雑 | 高（テキスト処理） | エンコーディング対応ライター | 文字コード対応 |
| StringReader | 文字列ストリーム処理がない | 中（文字列処理） | 文字列ストリームリーダー | ストリームAPI統一 |
| StringWriter | 文字列ストリーム生成がない | 中（文字列生成） | 文字列ストリームライター | ストリームAPI統一 |
| MemoryMappedFile | メモリマップ機能がない | 中（大容量処理） | メモリマップトファイルAPI | 大容量高速処理 |
| MemoryMappedViewAccessor | メモリマップアクセスがない | 中（ランダムアクセス） | ビューアクセサAPI | ランダムアクセス効率化 |
| IniResourceWriter | INI操作が煩雑 | 低（設定管理） | INIファイルAPI | 設定ファイル管理 |
| CorDateTime | Date型が精度・範囲不足 | 高（日付時刻処理） | .NET互換DateTime | ミリ秒精度・広範囲対応 |
| TimeSpan | 時間間隔型がない | 高（時間計算） | .NET互換TimeSpan | 時間計算の標準化 |
| TimeZone | タイムゾーン対応がない | 中（国際対応） | タイムゾーン変換API | 国際時刻対応 |
| CultureInfo | カルチャ情報管理がない | 高（国際化） | .NET互換カルチャ情報 | ロケール一元管理 |
| DateTimeFormatInfo | 日時書式情報がない | 高（書式設定） | カルチャ依存日時書式 | ロケール対応書式 |
| NumberFormatInfo | 数値書式情報がない | 高（書式設定） | カルチャ依存数値書式 | ロケール対応書式 |
| GregorianCalendar | グレゴリオ暦基盤がない | 高（暦計算） | .NET互換グレゴリオ暦 | 暦計算の標準化 |
| JapaneseCalendar | 和暦対応がない | 中（日本向け） | 和暦変換API | 日本市場対応 |
| KoreanCalendar | 韓国暦対応がない | 低（韓国向け） | 韓国暦変換API | 韓国市場対応 |
| TaiwanCalendar | 台湾暦対応がない | 低（台湾向け） | 民国暦変換API | 台湾市場対応 |
| ThaiBuddhistCalendar | タイ仏暦対応がない | 低（タイ向け） | タイ仏暦変換API | タイ市場対応 |
| HijriCalendar | イスラム暦対応がない | 低（中東向け） | ヒジュラ暦変換API | 中東市場対応 |
| HebrewCalendar | ヘブライ暦対応がない | 低（イスラエル向け） | ヘブライ暦変換API | イスラエル市場対応 |
| JulianCalendar | ユリウス暦対応がない | 低（歴史処理） | ユリウス暦変換API | 歴史的日付処理 |
| BigInteger | 大整数演算ができない | 高（金融・科学計算） | 任意精度整数API | オーバーフロー回避 |
| Int64 | 64ビット整数がない | 高（大数値処理） | .NET互換Int64 | 数値範囲拡大 |
| Random | 乱数生成が基本的 | 高（シミュレーション） | .NET互換Random | 再現可能乱数 |
| RNGCryptoServiceProvider | 暗号論的乱数がない | 高（セキュリティ） | 暗号論的乱数API | セキュアな乱数 |
| CorMath | 数学関数が不足 | 中（数値計算） | 拡張数学関数 | 計算の標準化 |
| BitConverter | 型変換が煩雑 | 高（データ変換） | .NET互換BitConverter | 型変換の効率化 |
| Convert | 型変換が煩雑 | 高（データ変換） | .NET互換Convert | 汎用型変換 |
| Buffer | バッファ操作が煩雑 | 中（メモリ操作） | .NET互換Buffer | メモリ操作効率化 |
| Char | 文字操作が煩雑 | 中（文字処理） | .NET互換Char | 文字操作の標準化 |
| ResourceManager | リソース管理機構がない | 高（多言語対応） | .NET互換リソース管理 | 多言語一元管理 |
| ResourceReader | リソース読み込みが煩雑 | 中（リソース処理） | .RESファイルリーダー | リソース読み込み効率化 |
| ResourceWriter | リソース書き込みが煩雑 | 中（リソース生成） | .RESファイルライター | リソース生成効率化 |
| ResourceSet | カルチャ別リソースがない | 中（多言語） | カルチャ別リソースセット | カルチャ別管理 |
| RegistryKey | レジストリ操作が煩雑 | 中（設定管理） | .NET互換レジストリAPI | 設定管理の標準化 |
| Guid | GUID生成が煩雑 | 高（識別子生成） | .NET互換GUID | 一意識別子管理 |
| Version | バージョン比較が煩雑 | 中（バージョン管理） | .NET互換Version | バージョン比較 |
| Environment | 環境情報取得が煩雑 | 中（システム情報） | .NET互換Environment | 環境情報アクセス |
| Thread | スレッド制御が煩雑 | 中（並行処理） | .NET互換Thread | スレッド制御 |
| WaitHandle | 待機制御がない | 中（同期処理） | .NET互換WaitHandle | 同期プリミティブ |
| Ticker | 高精度計測がない | 中（パフォーマンス） | 高精度タイマー | 時間計測 |
| Exception | 例外階層がない | 高（エラー処理） | .NET互換例外階層 | エラー処理の標準化 |
| ArgumentException | 引数例外がない | 高（入力検証） | .NET互換引数例外 | 引数エラー標準化 |
| ArgumentNullException | Null例外がない | 高（入力検証） | .NET互換Null例外 | Nullエラー標準化 |
| ArgumentOutOfRangeException | 範囲例外がない | 高（入力検証） | .NET互換範囲例外 | 範囲エラー標準化 |
| IOException | IO例外がない | 高（ファイル処理） | .NET互換IO例外 | IOエラー標準化 |
| FileNotFoundException | ファイル未検出例外がない | 高（ファイル処理） | .NET互換ファイル未検出例外 | ファイルエラー標準化 |
| DirectoryNotFoundException | ディレクトリ未検出例外がない | 中（ディレクトリ処理） | .NET互換ディレクトリ未検出例外 | ディレクトリエラー標準化 |
| FormatException | 書式例外がない | 中（パース処理） | .NET互換書式例外 | 書式エラー標準化 |
| InvalidOperationException | 無効操作例外がない | 中（状態管理） | .NET互換無効操作例外 | 状態エラー標準化 |
| InvalidCastException | キャスト例外がない | 中（型変換） | .NET互換キャスト例外 | 型変換エラー標準化 |
| ObjectDisposedException | 破棄済み例外がない | 中（リソース管理） | .NET互換破棄済み例外 | リソースエラー標準化 |
| IObject | オブジェクト基底がない | 高（全オブジェクト） | 共通オブジェクトインターフェース | 操作の統一 |
| IComparable | 比較インターフェースがない | 高（ソート・比較） | 比較可能インターフェース | ソートの統一 |
| IComparer | 比較ロジックインターフェースがない | 中（カスタム比較） | 比較ロジックインターフェース | カスタム比較 |
| IEqualityComparer | 等価比較インターフェースがない | 中（ハッシュ・等価） | 等価比較インターフェース | ハッシュ統一 |
| ICloneable | 複製インターフェースがない | 中（オブジェクト複製） | 複製可能インターフェース | 複製の統一 |
| IFormattable | 書式インターフェースがない | 中（書式設定） | 書式設定可能インターフェース | 書式の統一 |
| IFormatProvider | 書式プロバイダがない | 中（カルチャ書式） | 書式プロバイダインターフェース | カルチャ書式 |
| Comparer | デフォルト比較がない | 中（汎用比較） | デフォルト比較実装 | 汎用比較ロジック |
| EqualityComparer | デフォルト等価比較がない | 中（汎用等価） | デフォルト等価比較実装 | 汎用等価ロジック |
| Console | コンソール操作が煩雑 | 中（コンソールアプリ） | .NET互換コンソールAPI | コンソール操作効率化 |

## 4. 技術設計判断サマリー

> 技術的な設計判断の詳細（アーキテクチャ、DB設計、キャッシュ戦略等）は **アーキテクチャ設計意図書** を参照してください。

本機能群に関連する主要な技術設計判断のサマリーを以下に示す。

| 設計論点 | 採用した方針 | 選択理由（概要） |
| --- | --- | --- |
| .NET互換性 | System名前空間と同等の構造 | 既存.NETコードの移植を容易にするため |
| Windows API連携 | CryptoAPI/Registry API等の直接呼び出し | VB6環境での実装の確実性と性能確保のため |
| インターフェース設計 | IObject/IComparable等の標準インターフェース | コレクションとの相互運用性確保のため |
| エラー処理 | .NET互換の例外クラス階層 | 移行時のエラー処理互換性のため |
| 内部データ表現 | Currency型による64ビット精度 | VB6での高精度数値処理のため |

## 備考

- 本ライブラリはMITライセンスで提供されている
- VB6/VBA環境でのみ動作し、.NET環境では不要
- 将来の.NET移行を見据えた設計となっている
- 個々の機能は独立して使用可能であり、必要な機能のみを選択的に利用できる
