# 通知設計意図書

## 概要

本書は、VBCorLibライブラリの各通知機構が**なぜ実装されるに至ったか**を業務課題起点で記述する。
**各通知が存在するプログラム上の理由**を明らかにすることを目的とし、併せて技術設計判断の根拠も記載する。

**重要な前提**: VBCorLibは「VB6/VBA環境で.NET Frameworkの機能を利用可能にする」ためのライブラリであり、本書で扱う「通知」は一般的な業務システムにおける「外部への通知」（メール、SMS等）ではなく、**ライブラリ内部のプログラム通知機構**（イベント、コールバック、例外）を指す。

## 1. ドメイン概要

> ドメイン概要（業界・業種、主なステークホルダー、システムの位置づけ、ドメイン共通課題等）は **[機能設計意図書](./機能設計意図書.md)** を参照してください。

**注記**: 機能設計意図書が存在しないため、以下にドメイン概要を記載する。

### 1.1 プロジェクトの位置づけ

| 項目 | 内容 |
|------|------|
| プロジェクト名 | VBCorLib |
| 分類 | 開発ライブラリ / フレームワーク |
| 対象環境 | VB6 / VBA（Visual Basic 6.0 / Visual Basic for Applications） |
| 目的 | .NET Frameworkの機能をVB6/VBA環境で利用可能にする |
| ライセンス | GNU Library General Public License |

### 1.2 ステークホルダー

| ステークホルダー | 役割 | 関心事 |
|----------------|------|--------|
| VB6/VBA開発者 | ライブラリ利用者 | .NET的な機能（コレクション、タイマー、暗号化、例外処理等）をVB6で利用したい |
| レガシーシステム保守者 | 既存システム維持 | VB6アプリケーションを.NET移行せずに機能強化したい |
| ライブラリ開発者 | VBCorLib開発・保守 | .NETとの互換性維持、VB6の制約内での実装 |

### 1.3 本ライブラリにおける「通知」の特殊性

VBCorLibで扱う「通知」は、一般的な業務システムの通知（メール送信、SMS、プッシュ通知等）とは異なり、**プログラム内部の通信機構**である。

| 通知種別 | 説明 | 例 |
|---------|------|-----|
| イベント通知 | VB6のWithEventsによるイベント駆動 | Ticker.Elapsed、SafeHandle.ReleaseHandle |
| コールバック通知 | AddressOfによる関数ポインタコールバック | Ticker Callback、CallbackSorter |
| 例外通知 | 構造化例外処理によるエラー伝播 | ArgumentException、IOException等 |
| コンソール通知 | Windows APIコールバックによる通知 | Console.Break、Console.Beep |

---

## 2. 業務プロセス別の通知と想定価値

VBCorLibは汎用ライブラリであり、特定の業務プロセスに紐づくものではない。代わりに、**機能カテゴリ**別に通知を分類し、各通知が解決する技術課題と提供価値を整理する。

### 業務プロセス1: タイマー処理（定期実行・バックグラウンド処理）

#### プロセス概要

**業務フロー**

VB6アプリケーションで定期的な処理（ポーリング、自動保存、データ更新等）を実行する際のフロー。

1. タイマーの初期化と間隔設定
2. タイマー開始
3. 指定間隔経過時にコールバック/イベント発火
4. アプリケーションがタイマーイベントを処理
5. 必要に応じてタイマー停止

**関係者**

VB6/VBA開発者、エンドユーザー（間接的）

**発生頻度**

開発時の実装頻度に依存（タイマー処理を必要とするアプリケーションすべて）

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| VB6標準タイマーの制約 | Timerコントロールはフォーム依存であり、クラスモジュールからは使用できない |
| マルチタイマー管理の複雑さ | 複数の独立したタイマーを管理する仕組みがVB6標準にはない |
| コールバック実装の困難さ | 標準モジュールからタイマーイベントを受信する方法がない |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **開発工数の増大**: 各プロジェクトでWindows API（SetTimer）を直接実装する必要があり、開発者1人あたり4〜8時間の追加工数
- **品質リスク**: タイマーコールバックの誤実装によるメモリリーク、アプリケーションクラッシュのリスク
- **保守性低下**: 各プロジェクトで独自実装されたタイマー管理コードの保守負担

#### 本業務をシステム化することの想定効果

- **開発工数削減**: Tickerクラスにより、タイマー実装が数行のコードで完了
- **品質向上**: 検証済みのライブラリ使用により、メモリリーク等のリスクを低減
- **保守性向上**: 統一されたAPIにより、コードの可読性・保守性が向上

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| Ticker.Elapsed | 指定間隔でイベント発火 | 独自実装: 4〜8時間/プロジェクト | 標準化された実装で0.5時間/プロジェクト |
| Ticker Callback | 標準モジュールへのコールバック | 独自実装: 追加2〜4時間 | WithEvents不可環境でも利用可能 |

---

### 業務プロセス2: リソース管理（ハンドル・メモリ解放）

#### プロセス概要

**業務フロー**

OSリソース（ファイルハンドル、GDIオブジェクト等）を使用する際のライフサイクル管理。

1. リソース取得（ファイルオープン、ハンドル作成等）
2. リソース使用
3. 使用完了後のリソース解放通知
4. 適切な解放処理の実行
5. リソースの無効化

**関係者**

VB6/VBA開発者

**発生頻度**

リソースを使用するすべてのアプリケーション

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| リソースリークの発生 | 解放忘れによるシステムリソースの枯渇 |
| 二重解放エラー | 既に解放されたハンドルの再解放によるクラッシュ |
| 解放処理の標準化不足 | 各リソース種別に応じた解放処理の実装が必要 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **システム不安定化リスク**: リソースリークによる長期稼働アプリケーションの不安定化
- **デバッグ工数**: リソースリークの原因調査に1件あたり2〜8時間
- **障害対応コスト**: 本番環境でのリソース枯渇による緊急対応

#### 本業務をシステム化することの想定効果

- **リソース管理の自動化**: SafeHandleパターンによる確実なリソース解放
- **二重解放防止**: IsClosed/OwnsHandleフラグによる状態管理
- **派生クラスでの拡張性**: 各リソース種別に応じた解放処理のカスタマイズ

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| SafeHandle.ReleaseHandle | ハンドル解放時のイベント通知 | 各箇所で明示的な解放処理が必要 | 派生クラスで統一的な解放実装が可能 |

---

### 業務プロセス3: デリゲート・コールバック基盤

#### プロセス概要

**業務フロー**

関数ポインタを使用したコールバック機構の利用フロー。

1. コールバック関数の定義（標準モジュール内）
2. NewDelegate/InitDelegateでデリゲート作成
3. コールバックを必要とするAPIにデリゲートを渡す
4. イベント発生時にデリゲート経由でコールバック実行
5. 適切なクリーンアップ

**関係者**

VB6/VBA開発者

**発生頻度**

コールバックパターンを必要とするすべての実装

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| AddressOfの直接呼び出し不可 | VB6ではAddressOfで取得した関数ポインタを直接呼び出せない |
| 型安全なコールバックの困難さ | 引数の型チェックがない関数ポインタ呼び出しはエラーを起こしやすい |
| メモリ管理の複雑さ | デリゲートのライフサイクル管理が困難 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **実装不能**: VB6標準機能ではコールバックパターンの実装が困難
- **安全性リスク**: 低レベルAPI直接呼び出しによるクラッシュリスク
- **開発工数**: Matt Curlandのデリゲーションコード技術の独自実装が必要

#### 本業務をシステム化することの想定効果

- **コールバックパターンの実現**: VB6でも.NET的なデリゲートパターンが使用可能
- **型安全性の向上**: Action_T_T、Func_T_T_Long等の型付きデリゲート
- **軽量実装**: NewDelegate（動的メモリ）とInitDelegate（構造体ベース）の選択肢

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| Delegation.NewDelegate | 軽量COMオブジェクトによるコールバック | 独自実装: 40時間以上 | 標準化されたデリゲート機能を提供 |
| Delegation.InitDelegate | 構造体ベースの軽量デリゲート | メモリ管理の複雑さ増大 | 長寿命オブジェクト向けの安全な実装 |

---

### 業務プロセス4: 配列ソート（カスタム比較）

#### プロセス概要

**業務フロー**

ユーザー定義の比較ロジックを使用した配列ソートのフロー。

1. ソート対象配列の準備
2. 比較関数の定義（標準モジュール内）
3. CorArray.Sortに配列と比較関数を渡す
4. ソートアルゴリズム実行中、各比較時にコールバック発火
5. ソート完了

**関係者**

VB6/VBA開発者

**発生頻度**

カスタムソートを必要とするすべてのアプリケーション

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| VB6標準ソート機能の不在 | 配列をソートする標準機能がない |
| カスタム比較の困難さ | オブジェクトの複合条件ソートが実装困難 |
| アルゴリズム実装の負担 | QuickSort等の効率的なアルゴリズム実装が必要 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **開発工数**: ソートアルゴリズムの独自実装に8〜16時間
- **品質リスク**: 非効率または不正なソート実装
- **保守性低下**: 各プロジェクトで異なるソート実装

#### 本業務をシステム化することの想定効果

- **開発工数削減**: CorArray.Sortで数行のコードでソート実行
- **品質向上**: 検証済みQuickSortアルゴリズムの利用
- **柔軟性**: カスタム比較関数による任意の比較ロジック

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| CallbackSorter | ソート時のカスタム比較コールバック | ソート実装: 8〜16時間 | .NET的なArray.Sort(IComparer)相当の機能 |

---

### 業務プロセス5: 非同期処理完了通知

#### プロセス概要

**業務フロー**

非同期操作（ファイルI/O、ネットワーク通信等）の完了通知フロー。

1. 非同期操作の開始（BeginXxx）
2. IAsyncResultの取得
3. 他の処理を実行（または待機）
4. 非同期操作完了時にAsyncCallback.Execute発火
5. 結果の取得（EndXxx）

**関係者**

VB6/VBA開発者

**発生頻度**

非同期処理を必要とするアプリケーション

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| VB6シングルスレッド制約 | VB6は本質的にシングルスレッドであり、真の非同期は限定的 |
| 非同期パターンの標準化不足 | 非同期操作の完了通知方法が標準化されていない |
| コールバック管理の複雑さ | 非同期操作の状態管理が困難 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **設計の非標準化**: 各プロジェクトで異なる非同期パターン
- **保守性低下**: 非同期処理の追跡・デバッグが困難
- **移行障壁**: .NETへの移行時のパターン不一致

#### 本業務をシステム化することの想定効果

- **パターン標準化**: .NETのAsyncCallbackパターンとの互換性
- **状態管理**: IAsyncResultによる操作状態の統一的な管理
- **.NET移行容易性**: 同様のパターンでの実装により移行が容易

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| AsyncCallback.Execute | 非同期操作完了時のコールバック | 独自パターン実装が必要 | .NET互換の非同期パターン提供 |

---

### 業務プロセス6: コンソールアプリケーション制御

#### プロセス概要

**業務フロー**

コンソールアプリケーションにおける割り込み処理とユーザー通知のフロー。

1. コンソールアプリケーション起動
2. 処理実行中にユーザー操作（Ctrl+C等）発生
3. BreakイベントでOSからコールバック
4. アプリケーションがBreakType確認
5. 適切なクリーンアップ処理実行
6. 必要に応じてBeepによる音声通知

**関係者**

VB6/VBA開発者、コンソールアプリケーションユーザー

**発生頻度**

コンソールアプリケーション開発時

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| 割り込み処理の困難さ | Ctrl+C等のブレークイベント検出が複雑 |
| クリーンアップの確実性 | 中断時のリソース解放が保証されない |
| ユーザー通知の制限 | コンソール環境でのユーザーへの通知方法が限定的 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **データ損失リスク**: 中断時のデータ保存漏れ
- **実装工数**: SetConsoleCtrlHandler APIの直接実装が必要
- **ユーザビリティ低下**: 処理状況の通知手段がない

#### 本業務をシステム化することの想定効果

- **割り込み検出**: Console.Break/BreakTypeによる標準的な検出方法
- **安全な中断**: クリーンアップ処理の実装が容易
- **ユーザー通知**: Console.Beepによる音声フィードバック

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| Console.Break | コンソールブレーク発生通知 | SetConsoleCtrlHandler直接実装が必要 | 標準的なブレーク検出を提供 |
| Console.BreakType | ブレーク種別通知 | イベント種別判定の独自実装が必要 | Ctrl+C、シャットダウン等の種別識別 |
| Console.Beep | 音声通知 | Windows Beep APIの直接呼び出しが必要 | 簡潔なAPIで音声フィードバック |

---

### 業務プロセス7: 構造化例外処理

#### プロセス概要

**業務フロー**

エラー発生時の構造化例外処理フロー。

1. 処理実行中にエラー検出
2. 適切な例外クラス（ArgumentException等）生成
3. Throw関数で例外をスロー
4. Err.Raise経由でVBランタイムに通知
5. エラーハンドラでCatch関数により例外捕捉
6. 例外の型に応じた処理分岐

**関係者**

VB6/VBA開発者

**発生頻度**

エラー処理を必要とするすべてのアプリケーション

#### 関連する業務課題

| 業務課題 | このプロセスでの具体的な影響 |
| -------- | -------------------------------- |
| VB6 On Errorの制約 | エラー番号のみでの識別、型による分岐が困難 |
| エラー情報の不足 | スタックトレース、内部例外等の詳細情報がない |
| .NET移行障壁 | エラー処理パターンの大幅な変更が必要 |

#### 本業務をシステム化しない場合の想定リスク・コスト

- **デバッグ工数増大**: エラー原因特定に1件あたり30分〜2時間追加
- **品質リスク**: 不十分なエラーハンドリングによる予期せぬ動作
- **保守性低下**: エラー処理パターンの非標準化

#### 本業務をシステム化することの想定効果

- **構造化例外処理**: .NET的なThrow/Catchパターンの実現
- **詳細なエラー情報**: Message、ParamName、InnerException等
- **型による例外分岐**: CatchArgument、CatchFileNotFound等の型別捕捉
- **.NET移行容易性**: 同様のパターンでの実装

#### 主な関連通知と業務インパクト

| 通知名 | 何をする通知か | 通知がない場合の運用工数 | 通知があることによるインパクト |
| -------- | ---------------- | -------------------------- | -------------------------------- |
| ExceptionManagement.Throw | 例外のスロー | On Error + エラー番号のみ | 構造化例外のスロー |
| ExceptionManagement.Catch | 例外の捕捉 | Err.Numberでの分岐のみ | 型別例外の捕捉 |
| ArgumentException | 引数エラー通知 | 汎用エラー番号のみ | ParamName付きの詳細エラー |
| ArgumentNullException | Null引数エラー通知 | Err.Number 91のみ | Null引数の明示的識別 |
| ArgumentOutOfRangeException | 範囲外エラー通知 | Err.Number 5または9のみ | ActualValue付きの詳細エラー |
| FileNotFoundException | ファイル未検出通知 | Err.Number 53のみ | FileName付きの詳細エラー |
| DirectoryNotFoundException | ディレクトリ未検出通知 | Err.Number 76のみ | 明確なディレクトリエラー識別 |
| IOException | I/Oエラー通知 | 各種エラー番号の判別が必要 | I/Oエラーの統一的な捕捉 |
| InvalidOperationException | 無効操作通知 | エラー番号での識別のみ | 状態エラーの明示的識別 |
| OutOfMemoryException | メモリ不足通知 | Err.Number 7のみ | メモリエラーの明示的識別 |
| CryptographicException | 暗号エラー通知 | Windows APIエラーの解釈が必要 | 暗号処理エラーの統一的な通知 |

---

## 3. 各通知想定意図サマリー

以下に、通知設計書に記載されている**全21件**の通知について意図を記載する。

| 通知名 | 背景にある技術課題 | 技術負荷（推察） | 主な解決アプローチ | 手動実装を選ばなかった理由 | 画面表示で代用を選ばなかった理由 |
| -------- | ------------------ | ---------------- | ------------------ | -------------------------- | ---------------------------------- |
| 1. Ticker.Elapsed | VB6標準タイマーがフォーム依存 | 高（API実装4〜8時間） | WithEventsによるイベント通知 | Windows Timer API直接呼び出しの複雑さ | タイマーはUIではなく内部処理 |
| 2. SafeHandle.ReleaseHandle | リソース解放の標準化不足 | 中（派生クラス設計2〜4時間） | 派生クラスでの解放処理カスタマイズ | 各リソースで異なる解放APIの管理が複雑 | リソース解放は内部処理 |
| 3. Ticker Callback | 標準モジュールでタイマー受信不可 | 高（デリゲート実装必要） | AddressOfによるコールバック | WithEventsが使用できない環境への対応 | タイマーはUIではなく内部処理 |
| 4. CallbackSorter | VB6に配列ソート標準機能がない | 高（ソート実装8〜16時間） | カスタム比較コールバック | 汎用的なソート機能の提供 | ソートは内部処理 |
| 5. Delegation.NewDelegate | AddressOfの直接呼び出し不可 | 非常に高（40時間以上） | 軽量COMオブジェクトによるラップ | Matt Curlandコード技術の再発明が必要 | デリゲートは内部機構 |
| 6. Delegation.InitDelegate | 長寿命オブジェクトでのメモリ管理 | 高（メモリ管理設計必要） | 構造体ベースの軽量デリゲート | 動的メモリ確保のリスク回避 | デリゲートは内部機構 |
| 7. AsyncCallback.Execute | 非同期パターンの標準化不足 | 中（パターン設計2〜4時間） | インターフェースベースのコールバック | .NET互換パターンの提供 | 非同期処理は内部機構 |
| 8. Console.Break | コンソール割り込み検出の複雑さ | 中（API実装2〜4時間） | ポーリングベースのフラグ通知 | SetConsoleCtrlHandler直接実装の回避 | 割り込みは即時検出が必要 |
| 9. Console.BreakType | ブレーク種別判定の必要性 | 低（Break実装に含まれる） | 列挙型による種別識別 | イベント種別に応じた処理分岐の必要 | 種別判定は内部処理 |
| 10. Console.Beep | ユーザーへの音声通知の必要性 | 低（API呼び出し30分） | Windows Beep APIラッパー | 簡潔なAPIでの提供 | 画面を見ていないユーザーへの通知 |
| 11. ExceptionManagement.Throw | VB6 On Errorの型識別制約 | 高（例外フレームワーク構築） | 型付き例外のスロー | 構造化例外処理の実現 | 例外は即時通知が必要 |
| 12. ExceptionManagement.Catch | 例外の型別捕捉不可 | 高（Throwと対） | 型別Catch関数群 | 型による処理分岐の実現 | 例外捕捉は内部処理 |
| 13. ArgumentException | 引数エラーの詳細情報不足 | 中（例外クラス実装） | ParamName付き例外 | 引数名の明示による迅速なデバッグ | 例外は即時通知が必要 |
| 14. ArgumentNullException | Null引数の識別困難 | 中（ArgumentException派生） | 専用例外クラス | Null引数の明示的識別 | 例外は即時通知が必要 |
| 15. ArgumentOutOfRangeException | 範囲外エラーの詳細情報不足 | 中（ArgumentException派生） | ActualValue付き例外 | 実際の値の提供によるデバッグ支援 | 例外は即時通知が必要 |
| 16. FileNotFoundException | ファイルエラーの詳細情報不足 | 中（IOException派生） | FileName付き例外 | ファイル名の明示による問題特定 | 例外は即時通知が必要 |
| 17. DirectoryNotFoundException | ディレクトリエラーの識別困難 | 低（IOException派生） | 専用例外クラス | FileNotFoundとの区別 | 例外は即時通知が必要 |
| 18. IOException | I/Oエラーの統一的捕捉不可 | 中（例外基底クラス設計） | I/O例外の基底クラス | I/Oエラーの統一的な処理 | 例外は即時通知が必要 |
| 19. InvalidOperationException | 状態エラーの識別困難 | 低（SystemException派生） | 専用例外クラス | 引数エラーとの明確な区別 | 例外は即時通知が必要 |
| 20. OutOfMemoryException | メモリ不足の重大性認識 | 低（SystemException派生） | 専用例外クラス | 重大エラーの明示的識別 | 例外は即時通知が必要 |
| 21. CryptographicException | 暗号エラーの詳細情報不足 | 中（SecurityException設計） | 暗号専用例外クラス | Windows CryptoAPIエラーの統一的処理 | 例外は即時通知が必要 |

---

## 4. 技術設計判断と根拠

### 設計意図サマリー

| 設計論点 | 想定される選択肢 | 採用した方針 | 主な選択理由 |
| -------- | ---------------- | ------------ | ------------ |
| 通知チャネル選択 | イベント / コールバック / 例外 | すべて採用（用途別） | VB6の制約に応じた最適な方式選択 |
| 送信タイミング | 同期 / 非同期 | 同期（VB6制約） | VB6シングルスレッド制約による |
| リトライ戦略 | あり / なし | 基本なし | プログラム内部通知のためリトライ不要 |
| 配信制御 | あり / なし | なし | プログラム内部通知のため制御不要 |

---

### 4-1. 通知チャネル選択

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| WithEventsイベント | VB6標準のイベント機構 | クラスモジュールから使用可能 | 標準モジュール不可 |
| AddressOfコールバック | 関数ポインタによるコールバック | 標準モジュールから使用可能 | 実装複雑 |
| Err.Raise例外 | VBランタイムのエラー機構 | 広く認知されている | 型による分岐困難 |
| 戻り値 | 関数の戻り値で通知 | シンプル | 複雑な情報伝達困難 |

#### 採用した方針

用途に応じて複数のチャネルを採用：
- **イベント**: Ticker.Elapsed、SafeHandle.ReleaseHandle等
- **コールバック**: Ticker Callback、CallbackSorter、Delegation等
- **例外**: ArgumentException系、IOException系等

#### チャネル用途

| チャネル | 用途 | 優先度 | 選択理由 |
| -------- | ---- | ------ | -------- |
| イベント | タイマー、リソース解放 | 高 | VB6開発者に馴染みのある方式 |
| コールバック | 標準モジュール対応、ソート | 高 | WithEvents不可環境への対応 |
| 例外 | エラー通知 | 高 | .NET互換の構造化例外処理 |

#### 選択理由

VB6の言語制約（WithEventsはクラスモジュール限定、AddressOfの直接呼び出し不可等）に対応しつつ、.NET Frameworkとの互換性を維持するため、用途に応じた最適なチャネルを選択している。

#### 不採用とした選択肢の理由

- **戻り値のみ**: 複雑な情報（例外の詳細情報、複数の状態値等）の伝達が困難なため、補助的な用途に限定

---

### 4-2. テンプレート管理

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| コード内埋め込み | ソースコードにメッセージ | シンプル | 変更にコンパイル必要 |
| リソースファイル | 外部リソースで管理 | 多言語対応可能 | 実装複雑 |

#### 採用した方針

リソースファイル管理（Environment.GetResourceString）

#### テンプレート構成

| 種別 | 管理場所 | 変更頻度 | 選択理由 |
| ---- | -------- | -------- | -------- |
| 例外メッセージ | リソースファイル | 低 | 多言語対応の可能性、一元管理 |
| 定数値 | ソースコード | 非常に低 | HResult、ErrorNumber等の固定値 |

#### 選択理由

.NET Frameworkと同様のリソース管理アプローチを採用し、将来的な多言語対応や一元的なメッセージ管理を可能にしている。

#### 不採用とした選択肢の理由

- **完全な埋め込み**: メッセージの一元管理が困難になるため不採用

---

### 4-3. 送信タイミング設計

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| 同期送信 | 即時実行 | シンプル | 処理がブロック |
| 非同期送信 | バックグラウンド実行 | 処理継続可能 | VB6では限定的 |

#### 採用した方針

同期送信（VB6シングルスレッド制約による）

#### 送信パターン

| 通知種別 | タイミング | 他の選択肢 | 選択理由 |
| -------- | ---------- | ---------- | -------- |
| イベント | 同期即時 | なし | VB6のイベント機構が同期 |
| コールバック | 同期即時 | なし | VB6シングルスレッド制約 |
| 例外 | 同期即時 | なし | Err.Raiseが同期機構 |
| Beep | 同期ブロック | なし | Duration時間ブロック（仕様） |

#### 選択理由

VB6/VBAはシングルスレッドアパートメント（STA）で動作し、真のマルチスレッド処理はサポートされない。そのため、すべての通知は同期方式を採用している。

#### 不採用とした選択肢の理由

- **非同期送信**: VB6のシングルスレッド制約により、真の非同期実行は不可能

---

### 4-4. リトライ戦略

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| リトライあり | 失敗時に再試行 | 成功率向上 | 複雑性増大 |
| リトライなし | 1回のみ | シンプル | 一時的障害に弱い |

#### 採用した方針

基本的にリトライなし

#### リトライ設定

| チャネル | 最大回数 | 間隔 | バックオフ | 選択理由 |
| -------- | -------- | ---- | ---------- | -------- |
| イベント | 0 | - | - | AutoResetによる自動繰り返しのみ |
| コールバック | 0 | - | - | 呼び出し側で制御 |
| 例外 | 0 | - | - | 例外は1回のみスロー |

#### 選択理由

本ライブラリの「通知」はプログラム内部の通信機構であり、外部サービスへの通知とは異なる。リトライが必要な場合は呼び出し側で実装する設計としている。

#### 不採用とした選択肢の理由

- **自動リトライ**: プログラム内部通知ではリトライの責任は呼び出し側にあるため

---

### 4-5. 配信制御・スロットリング

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| スロットリングあり | 配信量制限 | リソース保護 | 遅延発生 |
| スロットリングなし | 制限なし | 即時配信 | 過負荷リスク |

#### 採用した方針

スロットリングなし

#### 制御設定

| 対象 | 制限 | 他の選択肢 | 選択理由 |
| ---- | ---- | ---------- | -------- |
| イベント | なし | - | 内部通知のため制限不要 |
| コールバック | なし | - | 呼び出し側で制御 |
| 例外 | なし | - | エラーは即座に通知が必要 |

#### 選択理由

プログラム内部の通知機構であり、外部サービスへの通知とは異なるため、配信量制限は不要と判断。タイマーの最小間隔（1ミリ秒）のみが実質的な制限となる。

#### 不採用とした選択肢の理由

- **スロットリングあり**: プログラム内部通知では即時性が重要なため

---

### 4-6. 通知設定（ユーザー制御）

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| 全通知必須 | 設定なし | シンプル | 柔軟性低い |
| オン/オフ設定 | ユーザー設定可 | 柔軟 | 実装複雑 |

#### 採用した方針

全通知必須（開発者によるコード制御）

#### 設定項目

| 項目 | 選択肢 | デフォルト | 選択理由 |
| ---- | ------ | ---------- | -------- |
| イベント | WithEvents宣言有無 | 使用時に宣言 | VB6標準の方式 |
| コールバック | AddressOf指定有無 | 任意 | 指定時のみ有効 |
| 例外 | なし | 常に発生 | エラー通知は必須 |

#### 選択理由

ライブラリレベルでの通知設定ではなく、開発者がコードで通知の受信を制御する設計。これはVB6の標準的なプログラミングパターンに沿っている。

#### 不採用とした選択肢の理由

- **ユーザー設定可能**: プログラム内部通知はアプリケーションロジックの一部であり、エンドユーザーが制御する対象ではない

---

### 4-7. 多言語・地域化対応

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| 単一言語 | 英語のみ | シンプル | グローバル対応不可 |
| 多言語対応 | リソース切り替え | グローバル対応 | 実装複雑 |

#### 採用した方針

単一言語（英語）をベースとし、リソースファイル構造で多言語対応可能

#### 対応言語

| 言語 | 対応状況 | テンプレート管理 | 選択理由 |
| ---- | -------- | ---------------- | -------- |
| 英語 | 完全対応 | リソースファイル | .NET Frameworkとの互換性 |
| その他 | 拡張可能 | リソースファイル追加 | 将来対応可能な設計 |

#### 選択理由

.NET Frameworkの英語メッセージとの互換性を重視し、英語ベースで実装。リソースファイル構造を採用しているため、将来的な多言語対応が可能。

#### 不採用とした選択肢の理由

- **複数言語同時対応**: 開発リソースの制約と、主要ユーザー層が英語圏であることから、初期バージョンでは英語のみ

---

### 4-8. 監視・配信状況追跡

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| ログ出力 | 通知ログ記録 | 追跡可能 | オーバーヘッド |
| 追跡なし | ログなし | 軽量 | デバッグ困難 |

#### 採用した方針

ライブラリレベルでの追跡なし（呼び出し側で必要に応じて実装）

#### 追跡項目

| 項目 | 追跡方法 | 保持期間 | 選択理由 |
| ---- | -------- | -------- | -------- |
| イベント発火 | なし | - | 呼び出し側で実装 |
| コールバック実行 | なし | - | 呼び出し側で実装 |
| 例外発生 | Err情報 | VBランタイム依存 | 標準的なエラー情報 |

#### 選択理由

ライブラリとしての軽量性を重視し、監視・追跡機能は呼び出し側の責任とした。例外については、VBランタイムのErr情報およびException.ToStringメソッドで詳細情報を取得可能。

#### 不採用とした選択肢の理由

- **詳細な追跡ログ**: パフォーマンスへの影響と、ライブラリの責務範囲を超えるため

---

### 4-9. エラーハンドリング

#### 想定される選択肢

| 選択肢 | 概要 | メリット | デメリット |
| ------ | ---- | -------- | ---------- |
| 黙殺 | エラー無視 | シンプル | 問題発見困難 |
| 例外スロー | 例外を発生 | 明確な通知 | 処理中断 |
| ログのみ | エラーログ | 追跡可能 | 能動的対応なし |

#### 採用した方針

例外スロー（構造化例外処理）

#### エラー処理

| エラー種別 | 処理 | 通知 | 選択理由 |
| ---------- | ---- | ---- | -------- |
| 引数エラー | ArgumentException | Throw | 呼び出し側で即座に対処必要 |
| I/Oエラー | IOException | Throw | 外部リソースエラーの明示 |
| 暗号エラー | CryptographicException | Throw | セキュリティ関連エラーの明示 |
| API失敗 | 各種例外 | Throw | Windows APIエラーの変換 |

#### 選択理由

.NET Frameworkと同様の構造化例外処理パターンを採用。エラー発生時は即座に例外をスローし、呼び出し側で適切なエラーハンドリングを行う設計。

#### 不採用とした選択肢の理由

- **黙殺**: 問題の発見が遅れ、デバッグが困難になるため
- **ログのみ**: VB6環境では標準的なログ機構がなく、例外スローが一般的

---

## 5. 将来の拡張性考慮

### 想定される変更

1. **新しい例外クラスの追加**: .NET Frameworkで追加される例外クラスへの対応（ただし、VB6/VBAの使用が減少しているため、大規模な追加は想定されない）

2. **非同期パターンの強化**: Task/asyncパターンの模倣実装（VB6の制約内での対応）

3. **追加の暗号アルゴリズム対応**: AES-GCM等の新しいアルゴリズムに対応するCryptographicException拡張

### 通知追加時の指針

1. **既存パターンの踏襲**: 新しい通知を追加する際は、既存の例外クラス階層、イベント通知パターン、コールバック機構に沿った設計とする

2. **.NET互換性の維持**: 可能な限り.NET Frameworkの同等機能と互換性のあるAPI設計とする

3. **VB6制約の考慮**: WithEventsのクラス限定、AddressOfの標準モジュール限定等のVB6制約を考慮した設計とする

4. **ドキュメント整備**: 新しい通知を追加する際は、通知設計書を作成し、設計意図を明確化する

---

## 備考

- 本書で扱う「通知」は、VBCorLibライブラリ内部のプログラム通信機構であり、一般的な業務システムの外部通知（メール、SMS等）とは性質が異なる
- VBCorLibは.NET Frameworkの機能をVB6で利用可能にすることを目的としており、通知設計もこの目的に沿っている
- 21件の通知設計書すべてについて、業務課題（技術課題）と設計意図を記載した
