# 通知設計書 12-ExceptionManagement.Catch

## 概要

本ドキュメントは、VBCorLibライブラリにおけるExceptionManagement.Catch通知ハンドリング機構の設計仕様を定義する。この機構は、Throw関数で発火された例外を型安全に捕捉し、適切なエラーハンドリングを可能にする。

### 本通知の処理概要

ExceptionManagement.Catchは、VB6/VBAにおける構造化例外処理を実現するための例外捕捉（キャッチ）機構である。Throwによって発火された例外オブジェクト、またはVBランタイムのErrオブジェクトを例外オブジェクトとして取得し、呼び出し元に通知する。

**業務上の目的・背景**：VB6/VBAのOn Error GoTo文によるエラーハンドリングでは、エラーの型を識別することが困難である。Catch機構を導入することで、例外の型に基づいた分岐処理が可能となり、より精密なエラーハンドリングを実現する。CatchArgument、CatchArgumentNull等の型別Catch関数により、特定の例外型のみを捕捉できる。

**通知の送信タイミング**：エラーハンドラ内でCatch関数が呼び出された時点で、保持されている例外オブジェクトまたはErrオブジェクトから例外情報を取得し、呼び出し元に通知する。

**通知の受信者**：Catch関数の呼び出し元。ByRef引数として渡されたException変数に例外オブジェクトが格納され、関数の戻り値としてBoolean（捕捉成功/失敗）が返される。

**通知内容の概要**：例外オブジェクト全体（Message、ErrorNumber、Source、InnerException、Data等のプロパティを含む）が通知される。型別Catch関数を使用した場合は、指定した型の例外のみが通知される。

**期待されるアクション**：呼び出し元は、Catch関数の戻り値がTrueの場合に例外オブジェクトを参照し、エラーメッセージの表示、ログ記録、エラーからの回復処理、または例外の再スローを行う。

## 通知種別

例外捕捉通知（プログラマティック通知 - 関数戻り値とByRef引数による）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（関数呼び出し） |
| 優先度 | 高（即時処理） |
| リトライ | 無（一度の呼び出しで完了） |

### 送信先決定ロジック

Catch関数の呼び出し元が受信者となる。ByRef引数として渡されたException変数に例外オブジェクトが格納される。

## 通知テンプレート

### 例外捕捉通知の場合

| 項目 | 内容 |
|-----|------|
| 戻り値 | Boolean（True=捕捉成功、False=捕捉なし） |
| ByRef引数 | Exception型オブジェクト |

### 本文テンプレート

```
' 汎用Catch
If Catch(Ex, Err) Then
    ' Ex に例外オブジェクトが格納される
End If

' 型別Catch
If CatchArgument(Ex) Then
    ' ArgumentException型の場合のみEx に格納される
End If
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| Ex | 例外オブジェクト | mException または Err から生成 | Yes |
| 戻り値 | 捕捉成功フラグ | Catch処理結果 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | Catch(Ex, Err) | mExceptionが存在、またはErr.Numberが0以外 | 汎用例外捕捉 |
| API呼び出し | CatchArgument(Ex) | mExceptionがArgumentException型 | 引数例外捕捉 |
| API呼び出し | CatchArgumentNull(Ex) | mExceptionがArgumentNullException型 | Null引数例外捕捉 |
| API呼び出し | CatchArgumentOutOfRange(Ex) | mExceptionがArgumentOutOfRangeException型 | 範囲外引数例外捕捉 |
| API呼び出し | CatchFileNotFound(Ex) | mExceptionがFileNotFoundException型 | ファイル未検出例外捕捉 |
| API呼び出し | CatchDirectoryNotFound(Ex) | mExceptionがDirectoryNotFoundException型 | ディレクトリ未検出例外捕捉 |
| API呼び出し | CatchPathTooLong(Ex) | mExceptionがPathTooLongException型 | パス長超過例外捕捉 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| mExceptionがNothing かつ Err.Numberが0 | 捕捉する例外が存在しない（戻り値False） |
| 型別Catchで型が一致しない | 対象外の例外型（戻り値False、Exは未設定） |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Catch呼び出し] --> B{mExceptionは存在するか}
    B -->|Yes| C[Ex = TakeException]
    B -->|No| D{Errが渡され、Err.Numberが0以外か}
    C --> E[戻り値 = True]
    D -->|Yes| F[CreateExceptionでException生成]
    D -->|No| G[戻り値 = False]
    F --> H[Ex = 生成したException]
    H --> I[Err.Clear]
    I --> E
    E --> J[VBA.Err.Clear]
    J --> K[終了]
    G --> K
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（メモリ内例外オブジェクトのみ使用）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 捕捉失敗 | mException未設定かつErr.Number=0 | 戻り値Falseで通知、Exは未設定 |
| 型不一致 | 型別Catchで例外型が不一致 | 戻り値False、mExceptionは保持継続 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0 |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- 例外オブジェクトに含まれる情報は呼び出し元に完全に公開される
- 機密情報を例外メッセージに含めないこと
- ログ出力時は例外情報のマスキングを検討

## 備考

- Catch関数呼び出し後、mExceptionはクリアされる（TakeExceptionによる）
- 型別Catch関数は、対象型の場合のみmExceptionをクリアする
- PeekException関数を使用すると、mExceptionをクリアせずに参照可能
- VBA.Err.Clearが最後に呼び出され、VBランタイムのエラー状態もクリアされる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

例外オブジェクトの保持と受け渡し方法を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ExceptionManagement.bas | `Source/CorLib/System/ExceptionManagement.bas` | mException変数とPeekException/TakeException関数 |

**読解のコツ**: mExceptionはモジュールレベル変数で、Throwで格納されCatchで取り出される。TakeExceptionは取得後にNothingをセットする。

#### Step 2: エントリーポイントを理解する

Catch関数群の実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExceptionManagement.bas | `Source/CorLib/System/ExceptionManagement.bas` | Catch関数と型別Catch関数の実装 |

**主要処理フロー**:
1. **行73-85**: Catch関数 - 汎用例外捕捉、mExceptionまたはErrから例外取得
2. **行87-92**: CatchArgument関数 - ArgumentException型のみ捕捉
3. **行94-99**: CatchArgumentNull関数 - ArgumentNullException型のみ捕捉
4. **行101-106**: CatchArgumentOutOfRange関数 - ArgumentOutOfRangeException型のみ捕捉
5. **行38-40**: PeekException関数 - mExceptionを参照（クリアしない）
6. **行42-45**: TakeException関数 - mExceptionを取得してクリア

#### Step 3: ヘルパー関数を理解する

例外オブジェクトの生成と管理関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExceptionManagement.bas | `Source/CorLib/System/ExceptionManagement.bas` | CreateException関数、ClearException関数 |

**主要処理フロー**:
- **行150-154**: CreateException - ErrオブジェクトからExceptionを生成
- **行34-36**: ClearException - mExceptionをNothingにクリア

### プログラム呼び出し階層図

```
Catch(Ex, Err)
    │
    ├─ PeekException()
    │      └─ mException参照
    │
    ├─ TakeException()
    │      └─ mException取得・クリア
    │
    └─ CreateException(Err.Description, Err.Number, Err.Source, Err.HelpFile)
           └─ Cor.NewException(...)

CatchArgument(Ex) / CatchArgumentNull(Ex) / etc.
    │
    ├─ TypeOf PeekException Is [SpecificType]
    │
    └─ TakeException()
```

### データフロー図

```
[入力]                      [処理]                        [出力]

mException ─────────────┐
                        ├──▶ Catch関数 ─────────────▶ Ex (ByRef)
Err (Optional) ─────────┘         │                      │
                                  │                      ▼
                                  ├─ TakeException()    Boolean戻り値
                                  │
                                  └─ VBA.Err.Clear
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ExceptionManagement.bas | `Source/CorLib/System/ExceptionManagement.bas` | ソース | Catch関数群の実装 |
| Exception.cls | `Source/CorLib/System/Exception.cls` | ソース | 基底例外インターフェース |
| ArgumentException.cls | `Source/CorLib/System/ArgumentException.cls` | ソース | 引数例外クラス |
| ArgumentNullException.cls | `Source/CorLib/System/ArgumentNullException.cls` | ソース | Null引数例外クラス |
| ArgumentOutOfRangeException.cls | `Source/CorLib/System/ArgumentOutOfRangeException.cls` | ソース | 範囲外引数例外クラス |
| FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | ソース | ファイル未検出例外クラス |
| DirectoryNotFoundException.cls | `Source/CorLib/System.IO/DirectoryNotFoundException.cls` | ソース | ディレクトリ未検出例外クラス |
