# 通知設計書 16-FileNotFoundException

## 概要

本ドキュメントは、VBCorLibライブラリにおけるFileNotFoundException例外通知の設計仕様を定義する。この例外は、指定されたファイルが存在しない場合にスローされ、呼び出し元にエラー状態を通知する。

### 本通知の処理概要

FileNotFoundExceptionは、ファイル操作において指定されたファイルが見つからない場合にスローされる例外クラスである。IOExceptionの派生クラスとして、ファイル未検出に特化した詳細情報（ファイル名）を提供する。

**業務上の目的・背景**：ファイル入出力処理において、対象ファイルが存在しないことを明確に通知する。単なるVBエラー番号53（ファイルが見つかりません）ではなく、ファイル名を含む詳細情報を提供することで、問題の特定と解決を容易にする。

**通知の送信タイミング**：ファイルオープン、読み込み、その他ファイル操作において、対象ファイルが存在しないことが検出された時点でスローされる。

**通知の受信者**：呼び出し元のメソッド。On Error GoToで設定されたエラーハンドラ、またはCatchFileNotFound関数を使用した例外ハンドリングコードが受信者となる。

**通知内容の概要**：エラーメッセージ（Message）、見つからなかったファイル名（FileName）、VBエラー番号（ErrorNumber = vbFileNotFound = 53）、HResult（COR_E_FILENOTFOUND）、発生元（Source）、原因となった内部例外（InnerException）が通知される。

**期待されるアクション**：呼び出し元は、FileNameプロパティで問題のファイルパスを確認し、ファイルの存在確認、パスの修正、またはユーザーへのファイル選択要求を行う。

## 通知種別

プログラム例外通知（VBランタイムのErr.Raiseを介したエラー通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（Throw経由でErr.Raise） |
| 優先度 | 高（即時処理） |
| リトライ | 無 |

### 送信先決定ロジック

VBランタイムのエラーハンドリング機構による自動決定。

## 通知テンプレート

### 例外通知の場合

| 項目 | 内容 |
|-----|------|
| エラー番号 | vbFileNotFound (53) |
| HResult | COR_E_FILENOTFOUND |
| メッセージ形式 | {Message} または Could not find file '{FileName}'. |

### 本文テンプレート

```
Could not find file '{ファイル名}'.
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| Message | エラーメッセージ | mBase.Message または デフォルトメッセージ | Yes |
| FileName | 見つからなかったファイル名 | mBase.GetValue("FileName") | No |
| ErrorNumber | VBエラー番号 | vbFileNotFound (固定値) | Yes |
| HResult | HRESULTコード | COR_E_FILENOTFOUND (固定値) | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| ファイル操作 | ファイルオープン | ファイルが存在しない | Open文、FileStream等 |
| ファイル操作 | ファイル読み込み | ファイルが存在しない | StreamReader等 |
| ファイル操作 | ファイル情報取得 | ファイルが存在しない | FileInfo等 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| ファイルが存在 | ファイルが見つかった場合は例外なし |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ファイル操作要求] --> B[ファイル存在確認]
    B --> C{ファイルは存在するか}
    C -->|Yes| D[正常処理継続]
    C -->|No| E[Cor.NewFileNotFoundException生成]
    E --> F[Init呼び出し - FileName, Message, InnerException設定]
    F --> G[Throw FileNotFoundException]
    G --> H[ExceptionManagement.Throw]
    H --> I[Err.Raise実行]
    I --> J[エラーハンドラへ伝播]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ファイル未検出 | 指定パスにファイルが存在しない | FileNameを参照しパス確認/修正 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0 |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- FileNameに含まれるパス情報が機密情報を露出しないよう注意
- サーバーサイドでは詳細なファイルパスをユーザーに公開しない
- ログ出力時はファイルパスのマスキングを検討

## 備考

- FileNotFoundExceptionはIOExceptionの派生クラス
- CatchFileNotFound関数で型別捕捉が可能
- FileNameが指定されMessageが未指定の場合、自動的にファイル名を含むメッセージが生成される
- IObject、Exception、SystemException、IOExceptionインターフェースを実装
- PropertyBagによるシリアライズ対応（Persistable = True）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FileNotFoundExceptionクラスの構造とプロパティを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | クラス定義、mBase変数、FileNameプロパティ |

**読解のコツ**: IOExceptionインターフェースを実装しており、ファイル名を追加プロパティとして持つ点に注目。

#### Step 2: エントリーポイントを理解する

コンストラクタとInit処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Constructors.cls | `Source/CorLib/System/Constructors.cls` | NewFileNotFoundException関数 |
| 2-2 | FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | Initメソッド、Class_InitProperties |

**主要処理フロー**:
1. **Constructors.cls 行408-411**: NewFileNotFoundException - 新規インスタンス生成とInit呼び出し
2. **FileNotFoundException.cls 行226-228**: Class_InitProperties - mBase初期化
3. **FileNotFoundException.cls 行230-240**: Init - FileName, Message, InnerException設定

#### Step 3: プロパティ実装を理解する

例外情報のプロパティ取得ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | Message, FileName, ErrorNumber, HResult |

**主要処理フロー**:
- **行74-76**: FileNameプロパティ - mBase.GetValueで取得
- **行108-110**: HResultプロパティ - COR_E_FILENOTFOUND固定値
- **行117-119**: ErrorNumberプロパティ - vbFileNotFound固定値
- **行155-157**: Messageプロパティ - GetExceptionMessageでメッセージ取得

### プログラム呼び出し階層図

```
Cor.NewFileNotFoundException(FileName, Message, InnerException)
    │
    └─ New FileNotFoundException
           │
           ├─ Class_InitProperties()
           │      └─ Cor.NewExceptionBase(Me, App)
           │
           └─ Init(FileName, Message, InnerException)
                  ├─ mBase.SetValue("FileName", FileName)
                  ├─ If Len(FileName) > 0 And CorString.IsNull(Message) Then
                  │      └─ mBase.Message = Environment.GetResourceString(IO_FileNotFound_Name, FileName)
                  ├─ Else
                  │      └─ mBase.Message = Message
                  └─ Set mBase.InnerException = InnerException
```

### データフロー図

```
[入力]                      [処理]                        [出力]

FileName ────────────┐
                     │
Message ─────────────┼──▶ FileNotFoundException.Init ──▶ mBase設定完了
                     │           │
InnerException ──────┘           │
                                 ▼
                          Throw FileNotFoundException
                                 │
                                 ▼
                          Err.Raise(53, Source, FormattedMessage)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FileNotFoundException.cls | `Source/CorLib/System.IO/FileNotFoundException.cls` | ソース | ファイル未検出例外クラス本体 |
| IOException.cls | `Source/CorLib/System.IO/IOException.cls` | ソース | 基底インターフェース |
| ExceptionBase.cls | `Source/CorLib/System/ExceptionBase.cls` | ソース | 例外基底ヘルパー |
| Constructors.cls | `Source/CorLib/System/Constructors.cls` | ソース | NewFileNotFoundExceptionコンストラクタ |
| ExceptionManagement.bas | `Source/CorLib/System/ExceptionManagement.bas` | ソース | CatchFileNotFound関数 |
