# 通知設計書 7-AsyncCallback.Execute

## 概要

本ドキュメントは、VBCorLibライブラリにおけるAsyncCallback.Execute非同期完了通知の設計を定義する。AsyncCallbackインターフェースは非同期操作の完了時にコールバックメソッドを呼び出す仕組みを提供する。

### 本通知の処理概要

AsyncCallback.Executeは、非同期操作が完了した際に呼び出されるコールバックインターフェースメソッドである。.NET FrameworkのAsyncCallbackデリゲートに相当し、IAsyncResultを引数として受け取り、非同期操作の結果を処理する。

**業務上の目的・背景**：ファイルI/O、ネットワーク通信、暗号化処理など、時間のかかる操作を非同期で実行し、完了時に通知を受け取る必要がある。AsyncCallbackは、非同期パターン（Begin/End）において、操作完了時のコールバック処理を標準化された方法で実装できるようにする。

**通知の送信タイミング**：非同期操作（BeginXxx）が完了した時点で、登録されたAsyncCallback実装のExecuteメソッドが呼び出される。

**通知の受信者**：AsyncCallbackインターフェースを実装したクラス。Implementsキーワードを使用して実装する。

**通知内容の概要**：Executeメソッドは、非同期操作の状態と結果を保持するIAsyncResultオブジェクトを引数として受け取る。受信者はこのオブジェクトを使用して操作結果を取得する。

**期待されるアクション**：受信者は、IAsyncResultを使用して非同期操作の結果を取得し（EndXxxメソッド呼び出し）、後続の処理を実行する。

## 通知種別

非同期完了通知（インターフェースベースのコールバック）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（完了時に即時コールバック） |
| 優先度 | 非同期操作の完了優先度に依存 |
| リトライ | 無 |

### 送信先決定ロジック

1. 非同期操作開始時（BeginXxx）にAsyncCallback実装オブジェクトを渡す
2. 操作完了時に内部でcallback.Execute(asyncResult)を呼び出す

## 通知テンプレート

### AsyncCallbackインターフェース

| 項目 | 内容 |
|-----|------|
| インターフェース名 | AsyncCallback |
| メソッド名 | Execute |
| 引数 | ByVal ar As IAsyncResult |
| 戻り値 | なし（Sub） |

### IAsyncResultインターフェース

| プロパティ | 型 | 説明 |
|-----------|---|------|
| AsyncState | Variant | ユーザー定義の状態オブジェクト |
| AsyncWaitHandle | WaitHandle | 待機用ハンドル |
| CompletedSyncronously | Boolean | 同期的に完了したか |
| IsCompleted | Boolean | 操作が完了したか |

### 実装例

```vb
' AsyncCallback実装クラス
Implements AsyncCallback

Private Sub AsyncCallback_Execute(ByVal ar As IAsyncResult)
    ' 非同期操作の結果を取得
    Dim result As Long
    result = SomeAsyncClass.EndOperation(ar)

    ' 結果を処理
    Debug.Print "Operation completed with result: " & result

    ' ユーザー状態を取得（必要な場合）
    Dim state As Variant
    state = ar.AsyncState
End Sub
```

### 使用例（非同期パターン）

```vb
' 非同期操作の開始
Dim callback As New MyAsyncHandler ' AsyncCallback実装
Dim asyncResult As IAsyncResult

asyncResult = SomeAsyncClass.BeginOperation(data, callback, state)

' 非同期操作が完了するとcallback.Executeが呼び出される
```

### 添付ファイル

該当なし（プログラム内部機能）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| ar | 非同期操作結果 | IAsyncResult実装 | Yes |
| AsyncState | ユーザー定義状態 | ar.AsyncState | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 非同期完了 | 非同期操作の終了 | AsyncCallbackが指定されている | 操作完了時に自動呼び出し |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| AsyncCallback未指定 | BeginXxxでcallbackにNothingを渡した場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[BeginXxx callback, state 呼び出し] --> B[非同期操作開始]
    B --> C[IAsyncResult作成]
    C --> D[バックグラウンド処理実行]
    D --> E{処理完了?}
    E -->|No| D
    E -->|Yes| F[IAsyncResult.IsCompleted = True]
    F --> G{AsyncCallback指定あり?}
    G -->|No| H[終了]
    G -->|Yes| I[callback.Execute ar]
    I --> J[AsyncCallback_Execute実行]
    J --> K[EndXxx ar で結果取得]
    K --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（メモリ内処理のみ）

### 更新テーブル一覧

該当なし（メモリ内処理のみ）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 操作エラー | 非同期操作自体が失敗 | EndXxxで例外をキャッチして処理 |
| タイムアウト | 操作が時間内に完了しない | AsyncWaitHandleでタイムアウト待機 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 操作依存 |
| リトライ間隔 | - |
| リトライ対象エラー | 一時的なエラー |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- 非同期操作の結果には機密情報が含まれる可能性がある
- AsyncStateに機密情報を保存しない
- 適切な例外処理でエラー情報の漏洩を防止

## 備考

- VB6はシングルスレッドのため、真の非同期実行は限定的
- AsyncCallbackはインターフェースとして定義されており、Implementsで実装が必要
- .NET FrameworkのAsyncCallbackデリゲートとは異なり、クラスベースの実装

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェース定義を理解する

AsyncCallbackとIAsyncResultのインターフェース定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AsyncCallback.cls | `Source/CorLib/System/AsyncCallback.cls` | Executeメソッド定義（50行目） |
| 1-2 | IAsyncResult.cls | `Source/CorLib/System/IAsyncResult.cls` | プロパティ定義（39-42行目） |

**読解のコツ**: AsyncCallbackはクラスとして定義されているが、Public Sub Execute定義のみで実装はない（インターフェース）。

#### Step 2: Executeメソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AsyncCallback.cls | `Source/CorLib/System/AsyncCallback.cls` | Execute Sub（50行目） |

**主要処理フロー**:
- **50行目**: Public Sub Execute(ByVal ar As IAsyncResult): End Sub

#### Step 3: IAsyncResultインターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IAsyncResult.cls | `Source/CorLib/System/IAsyncResult.cls` | 全プロパティ定義（39-42行目） |

**主要処理フロー**:
- **39行目**: AsyncState - ユーザー定義状態
- **40行目**: AsyncWaitHandle - 待機ハンドル
- **41行目**: CompletedSyncronously - 同期完了フラグ
- **42行目**: IsCompleted - 完了フラグ

### プログラム呼び出し階層図

```
[呼び出し元]
    │
    └─ BeginXxx(data, callback, state)
           │
           ├─ IAsyncResult作成
           │      ├─ AsyncState = state
           │      └─ IsCompleted = False
           │
           └─ 非同期処理開始
                  │
                  ▼
           [処理完了時]
                  │
                  ├─ IAsyncResult.IsCompleted = True
                  │
                  └─ callback.Execute(asyncResult)
                         │
                         └─ [AsyncCallback実装クラス]
                                │
                                └─ AsyncCallback_Execute(ar)
                                       │
                                       ├─ EndXxx(ar) で結果取得
                                       │
                                       └─ 後続処理
```

### データフロー図

```
[入力]                    [処理]                         [出力]

BeginXxxパラメータ  ───▶ 非同期操作         ───▶ IAsyncResult
                          │                         │
state               ───▶ │                   ───▶ ar.AsyncState
(ユーザー状態)            │
                          ▼
                    [操作完了]
                          │
                          ▼
callback.Execute    ◀─── 完了通知           ◀─── ar
(AsyncCallback)           │
                          ▼
                    EndXxx(ar)
                          │
                          ▼
                    操作結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AsyncCallback.cls | `Source/CorLib/System/AsyncCallback.cls` | ソース | コールバックインターフェース定義 |
| IAsyncResult.cls | `Source/CorLib/System/IAsyncResult.cls` | ソース | 非同期操作結果インターフェース |
| WaitHandle.cls | `Source/CorLib/System.Threading/WaitHandle.cls` | ソース | 待機ハンドルクラス |
