# API設計書

## 概要

本ドキュメントは、Visual Studio Code (VSCode) の Extension API 設計書です。VSCode はデスクトップアプリケーションであり、従来の REST API ではなく、拡張機能（Extension）向けの JavaScript/TypeScript API を提供しています。

Extension API は `vscode` モジュールを通じて拡張機能に公開され、エディタの機能を拡張・カスタマイズするためのインターフェースを提供します。

## 共通仕様

### API アクセス方法

```typescript
import * as vscode from 'vscode';
```

### 通信方式

- **IPC/RPC プロトコル**: Extension Host と Main Thread 間の通信
- **プロキシパターン**: `ExtHostContext` と `MainContext` を通じたプロセス間通信

### 共通パターン

| パターン | 説明 |
| --- | --- |
| Disposable | リソースの解放パターン。`dispose()` メソッドで登録解除 |
| Event | イベント購読パターン。`onDid*` 形式のイベント名 |
| Provider | 機能提供パターン。`register*Provider` で登録 |
| Thenable | Promise互換の非同期パターン |

### 共通エラーレスポンス

| エラー種別 | 説明 |
| --- | --- |
| ExtensionError | 拡張機能由来のエラー |
| SerializedError | IPC経由のシリアライズされたエラー |
| CancellationError | キャンセルされた操作のエラー |

## API一覧

| カテゴリ | 名前空間 | 説明 |
| --- | --- | --- |
| 環境 | vscode.env | エディタ実行環境情報の取得 |
| コマンド | vscode.commands | コマンドの登録・実行 |
| ウィンドウ | vscode.window | UI要素の操作・表示 |
| ワークスペース | vscode.workspace | ファイル・フォルダ操作 |
| 言語 | vscode.languages | 言語機能の登録 |
| デバッグ | vscode.debug | デバッガ連携 |
| 拡張機能 | vscode.extensions | 拡張機能の管理 |
| 認証 | vscode.authentication | 認証プロバイダ |
| タスク | vscode.tasks | タスク実行 |
| ノートブック | vscode.notebooks | Jupyter Notebook連携 |
| SCM | vscode.scm | ソース管理連携 |
| テスト | vscode.tests | テストフレームワーク連携 |
| ローカライゼーション | vscode.l10n | 国際化 |
| チャット | vscode.chat | AI チャット機能 |
| 言語モデル | vscode.lm | 言語モデル連携 |

---

## 各API名前空間定義

### env (環境)

エディタの実行環境に関する情報を提供する名前空間です。

#### 1-1 env.appName

エディタのアプリケーション名を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly string` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| string | アプリケーション名（例: 'VS Code'） |

---

#### 1-2 env.appRoot

エディタが実行されているルートフォルダパスを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly string` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| string | アプリケーションルートパス。環境によっては空文字 |

---

#### 1-3 env.machineId

コンピュータの一意識別子を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly string` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| string | マシンID |

---

#### 1-4 env.sessionId

現在のセッションの一意識別子を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly string` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| string | セッションID（エディタ起動ごとに変更） |

---

#### 1-5 env.language

ユーザーの優先言語を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly string` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| string | 言語コード（例: 'de-CH', 'fr', 'en-US'） |

---

#### 1-6 env.clipboard

システムクリップボードへのアクセスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly Clipboard` |
| 認証 | 不要 |
| 権限 | なし |

**Clipboardインターフェース**

```typescript
interface Clipboard {
  readText(): Thenable<string>;
  writeText(value: string): Thenable<void>;
}
```

---

#### 1-7 env.openExternal(uri)

外部アプリケーションでURIを開きます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `openExternal(target: Uri): Thenable<boolean>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| target | Uri | ○ | 開くURI |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Thenable\<boolean\> | 成功した場合 true |

---

#### 1-8 env.asExternalUri(uri)

外部アクセス可能な形式にURIを変換します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `asExternalUri(target: Uri): Thenable<Uri>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| target | Uri | ○ | 変換するURI |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Thenable\<Uri\> | クライアントマシンで使用可能なURI |

---

### commands (コマンド)

コマンドの登録と実行を行う名前空間です。

#### 2-1 commands.registerCommand(command, callback, thisArg?)

コマンドを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerCommand(command: string, callback: (...args: any[]) => any, thisArg?: any): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| command | string | ○ | コマンドの一意識別子 |
| callback | (...args: any[]) => any | ○ | コマンドハンドラ関数 |
| thisArg | any | - | ハンドラ実行時のthisコンテキスト |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Disposable | 破棄時にコマンドを登録解除 |

**使用例**

```typescript
const disposable = vscode.commands.registerCommand('extension.sayHello', () => {
  vscode.window.showInformationMessage('Hello World!');
});
```

---

#### 2-2 commands.registerTextEditorCommand(command, callback, thisArg?)

テキストエディタコマンドを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerTextEditorCommand(command: string, callback: (textEditor: TextEditor, edit: TextEditorEdit, ...args: any[]) => void, thisArg?: any): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| command | string | ○ | コマンドの一意識別子 |
| callback | Function | ○ | アクティブエディタとedit-builderにアクセス可能なハンドラ |
| thisArg | any | - | ハンドラ実行時のthisコンテキスト |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Disposable | 破棄時にコマンドを登録解除 |

---

#### 2-3 commands.executeCommand\<T\>(command, ...rest)

コマンドを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `executeCommand<T = unknown>(command: string, ...rest: any[]): Thenable<T>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| command | string | ○ | 実行するコマンドの識別子 |
| rest | any[] | - | コマンドに渡すパラメータ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Thenable\<T\> | コマンドの戻り値。戻り値がない場合は undefined |

**使用例**

```typescript
const result = await vscode.commands.executeCommand('vscode.openFolder', uri);
```

---

#### 2-4 commands.getCommands(filterInternal?)

利用可能なコマンド一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getCommands(filterInternal?: boolean): Thenable<string[]>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| filterInternal | boolean | - | false | true の場合、内部コマンド（_で始まる）を除外 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Thenable\<string[]\> | コマンドIDの配列 |

---

### window (ウィンドウ)

エディタウィンドウのUI要素を操作する名前空間です。

#### 3-1 window.activeTextEditor

現在アクティブなテキストエディタを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `TextEditor \| undefined` |
| 認証 | 不要 |
| 権限 | なし |

**戻り値**

| 型 | 説明 |
| --- | --- |
| TextEditor \| undefined | アクティブなエディタ。フォーカスがない場合は最後に変更されたエディタ |

---

#### 3-2 window.visibleTextEditors

現在表示されているテキストエディタの配列を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly TextEditor[]` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-3 window.showTextDocument(document, options?)

テキストドキュメントをエディタで表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showTextDocument(document: TextDocument, options?: TextDocumentShowOptions): Thenable<TextEditor>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| document | TextDocument | ○ | 表示するドキュメント |
| options | TextDocumentShowOptions | - | 表示オプション |

**TextDocumentShowOptions**

```typescript
interface TextDocumentShowOptions {
  viewColumn?: ViewColumn;
  preserveFocus?: boolean;
  preview?: boolean;
  selection?: Range;
}
```

---

#### 3-4 window.showInformationMessage(message, ...items)

情報メッセージを表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showInformationMessage(message: string, ...items: string[]): Thenable<string \| undefined>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| message | string | ○ | 表示するメッセージ |
| items | string[] | - | 選択肢の配列 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| Thenable\<string \| undefined\> | 選択されたアイテム。キャンセル時は undefined |

---

#### 3-5 window.showWarningMessage(message, ...items)

警告メッセージを表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showWarningMessage(message: string, ...items: string[]): Thenable<string \| undefined>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-6 window.showErrorMessage(message, ...items)

エラーメッセージを表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showErrorMessage(message: string, ...items: string[]): Thenable<string \| undefined>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-7 window.showQuickPick(items, options?, token?)

クイック選択UIを表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showQuickPick(items: string[] \| Thenable<string[]>, options?: QuickPickOptions, token?: CancellationToken): Thenable<string \| undefined>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| items | string[] \| Thenable\<string[]\> | ○ | 選択肢の配列 |
| options | QuickPickOptions | - | 表示オプション |
| token | CancellationToken | - | キャンセルトークン |

---

#### 3-8 window.showInputBox(options?, token?)

入力ボックスUIを表示します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `showInputBox(options?: InputBoxOptions, token?: CancellationToken): Thenable<string \| undefined>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-9 window.createTerminal(options?)

ターミナルを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createTerminal(options?: TerminalOptions): Terminal` |
| 認証 | 不要 |
| 権限 | なし |

**TerminalOptions**

```typescript
interface TerminalOptions {
  name?: string;
  shellPath?: string;
  shellArgs?: string[] | string;
  cwd?: string | Uri;
  env?: { [key: string]: string | null };
}
```

---

#### 3-10 window.createOutputChannel(name)

出力チャンネルを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createOutputChannel(name: string): OutputChannel` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-11 window.createTreeView(viewId, options)

ツリービューを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createTreeView<T>(viewId: string, options: TreeViewOptions<T>): TreeView<T>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 3-12 window.createWebviewPanel(viewType, title, showOptions, options?)

Webviewパネルを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createWebviewPanel(viewType: string, title: string, showOptions: ViewColumn, options?: WebviewPanelOptions & WebviewOptions): WebviewPanel` |
| 認証 | 不要 |
| 権限 | なし |

---

### workspace (ワークスペース)

ワークスペースとファイル操作を行う名前空間です。

#### 4-1 workspace.fs

ファイルシステムへのアクセスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `FileSystem` |
| 認証 | 不要 |
| 権限 | なし |

**FileSystemインターフェース**

```typescript
interface FileSystem {
  stat(uri: Uri): Thenable<FileStat>;
  readDirectory(uri: Uri): Thenable<[string, FileType][]>;
  createDirectory(uri: Uri): Thenable<void>;
  readFile(uri: Uri): Thenable<Uint8Array>;
  writeFile(uri: Uri, content: Uint8Array): Thenable<void>;
  delete(uri: Uri, options?: { recursive?: boolean }): Thenable<void>;
  rename(source: Uri, target: Uri, options?: { overwrite?: boolean }): Thenable<void>;
  copy(source: Uri, target: Uri, options?: { overwrite?: boolean }): Thenable<void>;
}
```

---

#### 4-2 workspace.workspaceFolders

開いているワークスペースフォルダの配列を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly WorkspaceFolder[] \| undefined` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-3 workspace.getWorkspaceFolder(uri)

URIを含むワークスペースフォルダを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getWorkspaceFolder(uri: Uri): WorkspaceFolder \| undefined` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-4 workspace.openTextDocument(uri)

テキストドキュメントを開きます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `openTextDocument(uri: Uri): Thenable<TextDocument>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-5 workspace.findFiles(include, exclude?, maxResults?, token?)

ワークスペース内のファイルを検索します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `findFiles(include: GlobPattern, exclude?: GlobPattern \| null, maxResults?: number, token?: CancellationToken): Thenable<Uri[]>` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| include | GlobPattern | ○ | 検索するファイルのglobパターン |
| exclude | GlobPattern \| null | - | 除外するファイルのglobパターン |
| maxResults | number | - | 最大結果数 |
| token | CancellationToken | - | キャンセルトークン |

---

#### 4-6 workspace.createFileSystemWatcher(globPattern, ignoreCreateEvents?, ignoreChangeEvents?, ignoreDeleteEvents?)

ファイルシステム監視を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createFileSystemWatcher(globPattern: GlobPattern, ...): FileSystemWatcher` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-7 workspace.getConfiguration(section?, scope?)

設定を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getConfiguration(section?: string, scope?: ConfigurationScope): WorkspaceConfiguration` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-8 workspace.registerTextDocumentContentProvider(scheme, provider)

仮想ドキュメントプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerTextDocumentContentProvider(scheme: string, provider: TextDocumentContentProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 4-9 workspace.registerFileSystemProvider(scheme, provider, options?)

ファイルシステムプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerFileSystemProvider(scheme: string, provider: FileSystemProvider, options?: { isCaseSensitive?: boolean; isReadonly?: boolean }): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

### languages (言語)

言語機能を提供する名前空間です。

#### 5-1 languages.getLanguages()

既知の言語識別子一覧を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getLanguages(): Thenable<string[]>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-2 languages.setTextDocumentLanguage(document, languageId)

ドキュメントの言語を変更します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `setTextDocumentLanguage(document: TextDocument, languageId: string): Thenable<TextDocument>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-3 languages.createDiagnosticCollection(name?)

診断コレクションを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createDiagnosticCollection(name?: string): DiagnosticCollection` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-4 languages.registerCompletionItemProvider(selector, provider, ...triggerCharacters)

コード補完プロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerCompletionItemProvider(selector: DocumentSelector, provider: CompletionItemProvider, ...triggerCharacters: string[]): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| selector | DocumentSelector | ○ | 適用対象のドキュメントを定義するセレクタ |
| provider | CompletionItemProvider | ○ | 補完プロバイダ |
| triggerCharacters | string[] | - | 補完をトリガーする文字 |

---

#### 5-5 languages.registerDefinitionProvider(selector, provider)

定義プロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerDefinitionProvider(selector: DocumentSelector, provider: DefinitionProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-6 languages.registerHoverProvider(selector, provider)

ホバープロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerHoverProvider(selector: DocumentSelector, provider: HoverProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-7 languages.registerCodeActionsProvider(selector, provider, metadata?)

コードアクションプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerCodeActionsProvider(selector: DocumentSelector, provider: CodeActionProvider, metadata?: CodeActionProviderMetadata): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-8 languages.registerCodeLensProvider(selector, provider)

コードレンズプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerCodeLensProvider(selector: DocumentSelector, provider: CodeLensProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-9 languages.registerDocumentFormattingEditProvider(selector, provider)

ドキュメント整形プロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerDocumentFormattingEditProvider(selector: DocumentSelector, provider: DocumentFormattingEditProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 5-10 languages.registerDocumentSymbolProvider(selector, provider, metaData?)

ドキュメントシンボルプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerDocumentSymbolProvider(selector: DocumentSelector, provider: DocumentSymbolProvider, metaData?: DocumentSymbolProviderMetadata): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

### authentication (認証)

認証プロバイダを管理する名前空間です。

#### 6-1 authentication.getSession(providerId, scopes, options?)

認証セッションを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getSession(providerId: string, scopes: readonly string[], options?: AuthenticationGetSessionOptions): Thenable<AuthenticationSession \| undefined>` |
| 認証 | 必要 |
| 権限 | スコープに依存 |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| providerId | string | ○ | 認証プロバイダID |
| scopes | readonly string[] | ○ | 要求するスコープ |
| options | AuthenticationGetSessionOptions | - | セッション取得オプション |

---

#### 6-2 authentication.registerAuthenticationProvider(id, label, provider, options?)

認証プロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerAuthenticationProvider(id: string, label: string, provider: AuthenticationProvider, options?: AuthenticationProviderOptions): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

### debug (デバッグ)

デバッガ連携を行う名前空間です。

#### 7-1 debug.activeDebugSession

現在アクティブなデバッグセッションを取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `DebugSession \| undefined` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 7-2 debug.startDebugging(folder, nameOrConfiguration, parentSessionOrOptions?)

デバッグセッションを開始します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `startDebugging(folder: WorkspaceFolder \| undefined, nameOrConfiguration: string \| DebugConfiguration, parentSessionOrOptions?: DebugSession \| DebugSessionOptions): Thenable<boolean>` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 7-3 debug.registerDebugConfigurationProvider(debugType, provider, triggerKind?)

デバッグ構成プロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerDebugConfigurationProvider(debugType: string, provider: DebugConfigurationProvider, triggerKind?: DebugConfigurationProviderTriggerKind): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

### extensions (拡張機能)

インストールされた拡張機能を管理する名前空間です。

#### 8-1 extensions.all

すべてのインストール済み拡張機能を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 型 | `readonly Extension<any>[]` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 8-2 extensions.getExtension(extensionId)

指定IDの拡張機能を取得します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `getExtension<T>(extensionId: string): Extension<T> \| undefined` |
| 認証 | 不要 |
| 権限 | なし |

---

### tasks (タスク)

タスク実行を管理する名前空間です。

#### 9-1 tasks.registerTaskProvider(type, provider)

タスクプロバイダを登録します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `registerTaskProvider(type: string, provider: TaskProvider): Disposable` |
| 認証 | 不要 |
| 権限 | なし |

---

#### 9-2 tasks.executeTask(task)

タスクを実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `executeTask(task: Task): Thenable<TaskExecution>` |
| 認証 | 不要 |
| 権限 | なし |

---

### tests (テスト)

テストフレームワーク連携を行う名前空間です。

#### 10-1 tests.createTestController(id, label)

テストコントローラを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| シグネチャ | `createTestController(id: string, label: string): TestController` |
| 認証 | 不要 |
| 権限 | なし |

---

## IPC/RPC プロトコル

VSCodeは Extension Host と Main Thread 間の通信に IPC/RPC プロトコルを使用します。

### MainThread インターフェース

| インターフェース | 説明 |
| --- | --- |
| MainThreadCommandsShape | コマンドの登録・実行 |
| MainThreadAuthenticationShape | 認証セッション管理 |
| MainThreadConfigurationShape | 設定の更新・削除 |
| MainThreadDiagnosticsShape | 診断情報の管理 |
| MainThreadDocumentsShape | ドキュメントの作成・保存 |
| MainThreadEditorsShape | エディタの操作 |
| MainThreadFileSystemShape | ファイルシステム操作 |
| MainThreadTerminalServiceShape | ターミナル管理 |
| MainThreadWebviewsShape | Webview管理 |

### ExtHost インターフェース

| インターフェース | 説明 |
| --- | --- |
| ExtHostCommandsShape | コマンドハンドラ |
| ExtHostDocumentsShape | ドキュメント管理 |
| ExtHostEditorsShape | エディタ管理 |
| ExtHostFileSystemShape | ファイルシステムプロバイダ |
| ExtHostLanguageFeaturesShape | 言語機能プロバイダ |
| ExtHostTerminalServiceShape | ターミナルイベント |

---

## イベント一覧

### グローバルイベント

| イベント | 説明 |
| --- | --- |
| env.onDidChangeTelemetryEnabled | テレメトリ有効/無効の変更 |
| env.onDidChangeShell | デフォルトシェルの変更 |
| env.onDidChangeLogLevel | ログレベルの変更 |

### ウィンドウイベント

| イベント | 説明 |
| --- | --- |
| window.onDidChangeActiveTextEditor | アクティブエディタの変更 |
| window.onDidChangeVisibleTextEditors | 表示エディタの変更 |
| window.onDidChangeTextEditorSelection | 選択範囲の変更 |
| window.onDidChangeTextEditorVisibleRanges | 表示範囲の変更 |
| window.onDidChangeTextEditorOptions | エディタオプションの変更 |
| window.onDidChangeTextEditorViewColumn | ビューカラムの変更 |
| window.onDidOpenTerminal | ターミナルのオープン |
| window.onDidCloseTerminal | ターミナルのクローズ |
| window.onDidChangeActiveTerminal | アクティブターミナルの変更 |

### ワークスペースイベント

| イベント | 説明 |
| --- | --- |
| workspace.onDidChangeWorkspaceFolders | ワークスペースフォルダの追加・削除 |
| workspace.onDidOpenTextDocument | ドキュメントのオープン |
| workspace.onDidCloseTextDocument | ドキュメントのクローズ |
| workspace.onDidChangeTextDocument | ドキュメントの変更 |
| workspace.onWillSaveTextDocument | ドキュメント保存前 |
| workspace.onDidSaveTextDocument | ドキュメント保存後 |
| workspace.onDidChangeConfiguration | 設定の変更 |
| workspace.onDidCreateFiles | ファイルの作成 |
| workspace.onDidDeleteFiles | ファイルの削除 |
| workspace.onDidRenameFiles | ファイルのリネーム |

### 言語イベント

| イベント | 説明 |
| --- | --- |
| languages.onDidChangeDiagnostics | 診断情報の変更 |

### デバッグイベント

| イベント | 説明 |
| --- | --- |
| debug.onDidStartDebugSession | デバッグセッション開始 |
| debug.onDidTerminateDebugSession | デバッグセッション終了 |
| debug.onDidChangeActiveDebugSession | アクティブセッションの変更 |
| debug.onDidChangeBreakpoints | ブレークポイントの変更 |

---

## 備考

### Proposed API

一部のAPIは "Proposed API" として提供されており、使用には `enableProposedApi` フラグが必要です。これらのAPIは将来変更される可能性があります。

### 非推奨API

以下のAPIは非推奨となっています：

- `workspace.rootPath` - `workspace.workspaceFolders` を使用してください
- `TextEditor.show()` - `window.showTextDocument` を使用してください
- `TextEditor.hide()` - `workbench.action.closeActiveEditor` コマンドを使用してください

### 参考資料

- [VSCode Extension API Reference](https://code.visualstudio.com/api/references/vscode-api)
- [Extension Guides](https://code.visualstudio.com/api/extension-guides/overview)
