# アーキテクチャ設計書

## 概要

本ドキュメントは、Visual Studio Code（VS Code）のオープンソース版であるCode-OSSのアーキテクチャ設計書です。VS Codeは、Microsoft社が開発したクロスプラットフォームのソースコードエディタであり、Electronベースのデスクトップアプリケーションとして実装されています。本設計書では、アプリケーションのレイヤー構成、モジュール構成、ミドルウェア構成、データフロー、および横断的関心事について詳細に記載します。

## システム全体構成

### アーキテクチャ概要図

システム全体のアーキテクチャ概要図は [アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照してください。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| VS Code Marketplace | REST API / HTTPS | 拡張機能のダウンロード・検索・更新 |
| GitHub | OAuth 2.0 / REST API | 認証、GitHub Copilot連携、Issue/PR管理 |
| Remote Server | SSH / WebSocket | リモート開発環境との接続 |
| Language Servers | Language Server Protocol (LSP) | 言語機能（補完、定義ジャンプ等）の提供 |
| Debug Adapters | Debug Adapter Protocol (DAP) | デバッグ機能の提供 |
| AppCenter | HTTPS | クラッシュレポート送信 |
| Tree-sitter | Native Module | 構文解析 |

## レイヤー構成

### アーキテクチャスタイル

VS Codeは**レイヤードアーキテクチャ**と**依存性注入（DI）パターン**を組み合わせた設計を採用しています。主要な特徴は以下の通りです：

1. **マルチプロセスアーキテクチャ**: Main Process、Renderer Process、Extension Host Processの分離
2. **プラットフォーム抽象化**: common/browser/node/electron-* による環境別の実装分離
3. **サービス指向設計**: 依存性注入コンテナによるサービス管理
4. **コントリビューションポイント**: 拡張性を実現するプラグインアーキテクチャ

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| base | 基盤ユーティリティ層 | common（共通ユーティリティ）、browser（DOM操作）、node（Node.jsユーティリティ）、parts（IPC通信） |
| platform | プラットフォームサービス層 | instantiation（DI）、configuration、files、storage、log、telemetry、environment等 |
| editor | コアエディタ層 | Monaco Editor、TextModel、ViewModelなど |
| workbench | ワークベンチUI層 | Parts（エディタ、サイドバー、パネル）、Services、Contributions |
| code | アプリケーションエントリ層 | electron-main、electron-browser、browser、node |

### レイヤー間の依存関係ルール

```
code -> workbench -> editor -> platform -> base
         |
         v
      extensions (Extension Host)
```

**依存の方向性:**
- 上位レイヤーは下位レイヤーに依存可能
- 下位レイヤーから上位レイヤーへの依存は禁止
- 同一レイヤー内でも依存方向は制御

**プラットフォーム依存の制約:**
- `common/`: プラットフォーム非依存（ブラウザ・Node.js両方で動作）
- `browser/`: ブラウザ専用API（DOM、Web API）
- `node/`: Node.js専用API
- `electron-main/`: Electronメインプロセス専用
- `electron-browser/`: Electronレンダラープロセス専用

**禁止される参照:**
- `common`から`browser`、`node`、`electron-*`への参照
- `browser`から`node`、`electron-main`への参照
- Native型（NativeParsedArgs、INativeEnvironmentService等）の`common`/`browser`での使用

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Editor Core | テキスト編集の中核機能 | vs/editor/common, vs/editor/browser, vs/editor/contrib |
| Workbench | IDE全体のUI・UX管理 | vs/workbench/browser, vs/workbench/contrib, vs/workbench/services |
| Platform Services | アプリケーション基盤サービス | vs/platform/* |
| Extension Host | 拡張機能の実行環境 | vs/workbench/api, vs/workbench/services/extensions |
| Remote | リモート開発機能 | vs/platform/remote, vs/workbench/contrib/remote |
| Terminal | 統合ターミナル | vs/workbench/contrib/terminal |
| Debug | デバッグ機能 | vs/workbench/contrib/debug |
| SCM | ソース管理（Git等） | vs/workbench/contrib/scm |
| Search | 検索機能 | vs/workbench/contrib/search |
| Chat/AI | AI支援機能（Copilot等） | vs/workbench/contrib/chat, vs/workbench/contrib/inlineChat |

### パッケージ構造

```
src/
├── bootstrap-*.ts          # アプリケーション起動ブートストラップ
├── main.ts                 # Electronメインプロセスエントリ
├── cli.ts                  # CLI エントリ
├── server-main.ts          # サーバーモードエントリ
└── vs/
    ├── base/               # 基盤ユーティリティ
    │   ├── common/         # プラットフォーム非依存ユーティリティ
    │   ├── browser/        # ブラウザ用ユーティリティ（DOM操作等）
    │   ├── node/           # Node.js用ユーティリティ
    │   └── parts/          # IPC、サンドボックス等
    ├── platform/           # プラットフォームサービス（90+サービス）
    │   ├── instantiation/  # 依存性注入（DIコンテナ）
    │   ├── configuration/  # 設定管理
    │   ├── files/          # ファイルシステム抽象化
    │   ├── storage/        # ストレージサービス
    │   ├── log/            # ロギング
    │   ├── telemetry/      # テレメトリ
    │   └── ...
    ├── editor/             # Monaco Editor コア
    │   ├── common/         # エディタモデル、サービス
    │   ├── browser/        # エディタUI
    │   ├── contrib/        # エディタ貢献機能（補完、スニペット等）
    │   └── standalone/     # スタンドアロンエディタ
    ├── workbench/          # ワークベンチ（IDE全体）
    │   ├── browser/        # ワークベンチUI
    │   │   ├── parts/      # エディタ、サイドバー、パネル等
    │   │   └── actions/    # ワークベンチアクション
    │   ├── contrib/        # 機能コントリビューション（60+）
    │   │   ├── chat/       # AI Chat機能
    │   │   ├── debug/      # デバッグ機能
    │   │   ├── terminal/   # ターミナル機能
    │   │   ├── scm/        # ソース管理
    │   │   └── ...
    │   ├── services/       # ワークベンチサービス（90+）
    │   └── api/            # 拡張機能API
    ├── code/               # アプリケーションエントリ
    │   ├── electron-main/  # メインプロセス
    │   ├── electron-browser/ # レンダラープロセス
    │   ├── browser/        # Webブラウザ版
    │   └── node/           # Node.jsサーバー
    └── server/             # リモートサーバー
extensions/                 # 組み込み拡張機能（100+）
├── typescript-language-features/
├── git/
├── css-language-features/
└── ...
```

### コンポーネント依存関係

```
[Extension Host Process]
        |
        | IPC (MessagePort)
        v
[Renderer Process (Workbench)]
        |
        | IPC (Electron IPC)
        v
[Main Process]
        |
        | Node.js API / Native Modules
        v
[OS / File System / Network]
```

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| RDB | SQLite | @vscode/sqlite3 5.1.11 | 設定、状態、拡張機能データの永続化 |
| KVS | IndexedDB | Browser API | Webブラウザ版でのデータ永続化 |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| V8 Code Cache | 組み込み | JavaScriptコードのコンパイル結果キャッシュ | コミットハッシュ依存 |
| In-Memory Cache | カスタム実装 | ファイル内容、構文解析結果等のキャッシュ | セッション中 |
| Extension Cache | カスタム実装 | 拡張機能マニフェスト、リソースのキャッシュ | 拡張機能バージョン依存 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Electron IPC | Electron 39.2.7 | メインプロセス-レンダラープロセス間通信 |
| MessagePort | Web API | Extension Host-Workbench間通信 |
| Worker PostMessage | Web API | Web Worker間通信 |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| ripgrep | @vscode/ripgrep 1.15.13 | 高速テキスト検索 |
| Tree-sitter | @vscode/tree-sitter-wasm 0.3.0 | 構文解析ベースの検索・ナビゲーション |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Electron | 39.2.7 | デスクトップアプリケーションフレームワーク |
| xterm.js | @xterm/xterm 6.1.0-beta | 統合ターミナル |
| node-pty | 1.2.0-beta | 疑似端末 |
| vscode-textmate | 9.3.1 | TextMate文法によるシンタックスハイライト |
| vscode-oniguruma | 1.7.0 | 正規表現エンジン（Oniguruma） |
| spdlog | @vscode/spdlog 0.15.2 | 高速ロギング |
| Parcel Watcher | @parcel/watcher 2.5.4 | ファイル監視 |
| KaTeX | 0.16.22 | 数式レンダリング |

## データフロー

### リクエスト処理の流れ

**ユーザー操作からUIレンダリングまで:**

1. ユーザーがキーボード/マウスで操作を実行
2. Renderer ProcessのDOMイベントハンドラがイベントを捕捉
3. CommandServiceがコマンドを解決・実行
4. 対応するServiceが処理を実行
5. 必要に応じてMain ProcessにIPC通信
6. 状態変更をEventEmitterで通知
7. UIコンポーネントが状態変更を受けて再レンダリング

**ファイル操作の流れ:**

1. EditorServiceがファイルオープンをリクエスト
2. FileServiceがファイルシステムプロバイダを選択
3. DiskFileSystemProviderがNode.js fsでファイル読み込み
4. TextModelが作成され、EditorWidgetに紐付け
5. TokenizationServiceがシンタックスハイライトを適用
6. ViewModelがレンダリング用データを生成

### 非同期処理の流れ

**拡張機能API呼び出しの流れ:**

1. 拡張機能コードがvscode.* APIを呼び出し
2. ExtHostApiCommandが呼び出しをシリアライズ
3. MessagePort経由でWorkbenchにIPC送信
4. MainThreadApiCommandsがリクエストを受信
5. 対応するWorkbenchサービスで処理実行
6. 結果をシリアライズしてExtension Hostに返送
7. 拡張機能にPromise解決として結果を返却

**Language Server Protocol処理:**

1. ユーザーがファイルを編集
2. TextDocumentChangeEventがLanguage Clientに通知
3. Language ClientがdidChangeをサーバーに送信
4. Language Serverが診断情報を計算
5. publishDiagnosticsでクライアントに通知
6. DiagnosticsServiceがマーカーを更新
7. EditorがデコレーションとしてUI表示

### データ永続化の流れ

**設定の保存:**

1. ユーザーが設定を変更
2. ConfigurationServiceが変更を検出
3. UserConfigurationWriterが設定ファイル（settings.json）を更新
4. FileServiceがファイル書き込みを実行
5. ConfigurationChangeEventが全プロセスにブロードキャスト

**状態の永続化:**

1. アプリケーション状態（ウィンドウサイズ、開いているファイル等）が変更
2. StateServiceが変更をバッファリング
3. 遅延書き込み（DELAYED SaveStrategy）でSQLiteに保存
4. アプリケーション終了時に最終状態を確実に保存

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| OAuth 2.0 | authenticationService | GitHub、Microsoft Account等の外部認証 |
| Token Storage | secretsService | 認証トークンの安全な保存 |
| Extension Permissions | extensionManagement | 拡張機能の権限管理 |
| Trusted Workspace | workspaceTrust | ワークスペースの信頼性管理 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | spdlog (BufferLogger -> SpdlogService) | {userData}/logs/{date}/{name}.log |
| テレメトリ | 1ds-core-js / 1ds-post-js | AppCenter / Microsoft Telemetry |
| 拡張機能ログ | OutputChannel | Output Panel / {userData}/logs/ |
| クラッシュレポート | Electron crashReporter | AppCenter |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| 予期されるエラー | ExpectedError | 静かに処理、ユーザー通知なし |
| ユーザーエラー | NotificationService | 通知バー表示、ユーザーにフィードバック |
| システムエラー | setUnexpectedErrorHandler | エラーダイアログ表示、ログ記録 |
| 拡張機能エラー | Extension Host隔離 | 拡張機能のみ終了、IDE継続動作 |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| ファイル操作 | FileService（アトミック書き込み） | 単一ファイル単位 |
| 設定変更 | ConfigurationEditingService | 設定ファイル単位 |
| Undo/Redo | UndoRedoService | エディタ/ワークスペース単位 |
| ワークスペース状態 | StateService（遅延バッチ書き込み） | アプリケーション単位 |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 依存性逆転の原則 (DIP) | 全サービス | createDecorator<T>()によるインターフェース定義、InstantiationServiceによるDI |
| 単一責任の原則 (SRP) | サービス分割 | FileService、ConfigurationService等の細分化 |
| オープン/クローズドの原則 (OCP) | Contribution Point | Registry、ExtensionPoint APIによる拡張性 |
| インターフェース分離の原則 (ISP) | プラットフォーム分離 | common/browser/node/electron-*による環境別インターフェース |
| リスコフの置換原則 (LSP) | FileSystemProvider | DiskFileSystemProvider、InMemoryFileSystemProvider等 |

### コーディング規約

**命名規則:**
- インターフェース: `I`プレフィックス（例：`IFileService`）
- サービス識別子: `createDecorator<T>(serviceId)` で定義
- イベント: `on`プレフィックス（例：`onDidChangeConfiguration`）
- プライベートメンバー: `_`プレフィックス

**ファイル構成:**
- 1ファイル1クラス/インターフェースが基本
- `.contribution.ts`：機能コントリビューション登録
- `.service.ts`：サービス実装

**非同期処理:**
- Promise/async-awaitを使用
- `Disposable`パターンによるリソース管理
- `Event`/`Emitter`パターンによるイベント通知

**レイヤー制約:**
- `layersChecker.ts`によるコンパイル時レイヤー違反チェック
- Native型のbrowser/commonでの使用禁止

## 備考

- 本ドキュメントはVS Code v1.109.0（Code-OSS）をベースに作成
- 拡張機能は`extensions/`ディレクトリに100以上の組み込み拡張機能が存在
- Web版（vscode.dev）とデスクトップ版で共通のコードベースを使用
- 依存性注入コンテナ（InstantiationService）が全サービスのライフサイクルを管理
