# Visual Studio Code (Code - OSS) コードリーディングガイドライン

## はじめに

このガイドラインは、Visual Studio Code (Code - OSS) のコードベースを効率的に理解するための手引きです。
TypeScriptに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、TypeScriptの基本構文と概念を解説します。

### 1.1 プログラム構造

VS Codeは主にTypeScriptで記述されています。ファイルはESモジュール形式で構成されており、各ファイルは `.ts` 拡張子を持ちます。

```typescript
// ファイル: src/main.ts:5-14
import * as path from 'node:path';
import * as fs from 'original-fs';
import * as os from 'node:os';
import { performance } from 'node:perf_hooks';
import { configurePortable } from './bootstrap-node.js';
import { bootstrapESM } from './bootstrap-esm.js';
import { app, protocol, crashReporter, Menu, contentTracing } from 'electron';
import minimist from 'minimist';
import { product } from './bootstrap-meta.js';
import { parse } from './vs/base/common/jsonc.js';
```

### 1.2 データ型と変数

TypeScriptは静的型付け言語で、変数宣言時に型を指定できます。VS Codeでは型定義を積極的に活用しています。

```typescript
// ファイル: src/vs/code/electron-main/main.ts:366-381
interface IArgvConfig {
    [key: string]: string | string[] | boolean | undefined;
    readonly locale?: string;
    readonly 'disable-lcd-text'?: boolean;
    readonly 'proxy-bypass-list'?: string;
    readonly 'disable-hardware-acceleration'?: boolean;
    readonly 'force-color-profile'?: string;
    readonly 'enable-crash-reporter'?: boolean;
    readonly 'crash-reporter-id'?: string;
    readonly 'enable-proposed-api'?: string[];
    readonly 'log-level'?: string | string[];
    readonly 'disable-chromium-sandbox'?: boolean;
    readonly 'use-inmemory-secretstorage'?: boolean;
    readonly 'enable-rdp-display-tracking'?: boolean;
    readonly 'remote-debugging-port'?: string;
}
```

### 1.3 制御構造

VS Codeでは標準的なTypeScript/JavaScriptの制御構造を使用しています。

```typescript
// ファイル: src/main.ts:43-54 (条件分岐の例)
if (args['sandbox'] &&
    !args['disable-chromium-sandbox'] &&
    !argvConfig['disable-chromium-sandbox']) {
    app.enableSandbox();
} else if (app.commandLine.hasSwitch('no-sandbox') &&
    !app.commandLine.hasSwitch('disable-gpu-sandbox')) {
    app.commandLine.appendSwitch('disable-gpu-sandbox');
} else {
    app.commandLine.appendSwitch('no-sandbox');
    app.commandLine.appendSwitch('disable-gpu-sandbox');
}
```

### 1.4 関数/メソッド定義

クラスメソッドと関数の両方が使用されています。アロー関数も広く使用されています。

```typescript
// ファイル: src/vs/code/electron-main/main.ts:185-198
async function onReady() {
    perf.mark('code/mainAppReady');

    try {
        const [, nlsConfig] = await Promise.all([
            mkdirpIgnoreError(codeCachePath),
            resolveNlsConfiguration()
        ]);

        await startup(codeCachePath, nlsConfig);
    } catch (error) {
        console.error(error);
    }
}
```

### 1.5 モジュール/インポート

ESモジュール構文を使用しており、`.js` 拡張子でインポートします（TypeScriptのビルド設定による）。

```typescript
// ファイル: src/vs/code/electron-main/main.ts:6-27
import { app, dialog } from 'electron';
import { unlinkSync, promises } from 'fs';
import { URI } from '../../base/common/uri.js';
import { coalesce, distinct } from '../../base/common/arrays.js';
import { Promises } from '../../base/common/async.js';
import { toErrorMessage } from '../../base/common/errorMessage.js';
```

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### 2.1.1 Electronアーキテクチャ

VS CodeはElectronをベースにしたデスクトップアプリケーションです。主に3つのプロセスタイプがあります。

1. **Main Process**: アプリケーションのライフサイクル管理
2. **Renderer Process**: 各ウィンドウのUI描画
3. **Extension Host**: 拡張機能の実行環境

```typescript
// ファイル: src/vs/code/electron-main/app.ts:132-156
export class CodeApplication extends Disposable {
    private windowsMainService: IWindowsMainService | undefined;
    private auxiliaryWindowsMainService: IAuxiliaryWindowsMainService | undefined;
    private nativeHostMainService: INativeHostMainService | undefined;

    constructor(
        private readonly mainProcessNodeIpcServer: NodeIPCServer,
        private readonly userEnv: IProcessEnvironment,
        @IInstantiationService private readonly mainInstantiationService: IInstantiationService,
        @ILogService private readonly logService: ILogService,
        // ... 他の依存関係
    ) {
        super();
        this.configureSession();
        this.registerListeners();
    }
}
```

#### 2.1.2 依存性注入（Dependency Injection）

VS Codeは独自のDIコンテナを使用しています。サービスはデコレータ構文で注入されます。

```typescript
// ファイル: src/vs/platform/instantiation/common/instantiation.ts:109-126
export function createDecorator<T>(serviceId: string): ServiceIdentifier<T> {
    if (_util.serviceIds.has(serviceId)) {
        return _util.serviceIds.get(serviceId)!;
    }

    const id = function (target: Function, key: string, index: number) {
        if (arguments.length !== 3) {
            throw new Error('@IServiceName-decorator can only be used to decorate a parameter');
        }
        storeServiceDependency(id, target, index);
    } as ServiceIdentifier<T>;

    id.toString = () => serviceId;
    _util.serviceIds.set(serviceId, id);
    return id;
}
```

**使用例:**

```typescript
// サービス識別子の定義
export const ILogService = createDecorator<ILogService>('logService');

// サービスの注入（コンストラクタパラメータのデコレータとして）
constructor(
    @ILogService private readonly logService: ILogService,
    @IConfigurationService private readonly configurationService: IConfigurationService
) { }
```

### 2.2 プロジェクト独自のパターン

#### 2.2.1 Disposableパターン

リソース管理のため、`IDisposable`インターフェースを広く使用しています。

```typescript
// ファイル: src/vs/base/common/lifecycle.ts:303-314
export interface IDisposable {
    dispose(): void;
}

export function isDisposable<E>(thing: E): thing is E & IDisposable {
    return typeof thing === 'object' && thing !== null &&
           typeof (<IDisposable><any>thing).dispose === 'function' &&
           (<IDisposable><any>thing).dispose.length === 0;
}
```

**DisposableStoreの使用例:**

```typescript
// ファイル: src/vs/base/common/lifecycle.ts:416-439
export class DisposableStore implements IDisposable {
    private readonly _toDispose = new Set<IDisposable>();
    private _isDisposed = false;

    public dispose(): void {
        if (this._isDisposed) {
            return;
        }
        markAsDisposed(this);
        this._isDisposed = true;
        this.clear();
    }

    public add<T extends IDisposable>(o: T): T {
        // ...
        this._toDispose.add(o);
        return o;
    }
}
```

#### 2.2.2 Eventパターン

イベント駆動型のパターンを広く採用しています。

```typescript
// ファイル: src/vs/base/common/event.ts:37-39
export interface Event<T> {
    (listener: (e: T) => unknown, thisArgs?: any, disposables?: IDisposable[] | DisposableStore): IDisposable;
}
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `*.contribution.ts` | モジュール/機能の登録・寄与ファイル | `snippets.contribution.ts` |
| `*Service.ts` | サービス実装ファイル | `configurationService.ts` |
| `I*Service` | サービスインターフェース | `IConfigurationService` |
| `common/` | プラットフォーム非依存コード | `vs/platform/*/common/` |
| `browser/` | ブラウザ/Renderer用コード | `vs/workbench/browser/` |
| `node/` | Node.js用コード | `vs/platform/*/node/` |
| `electron-main/` | Electron Mainプロセス用 | `vs/platform/*/electron-main/` |
| `electron-browser/` | Electron Renderer用 | `vs/platform/*/electron-browser/` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `I` (prefix) | インターフェース | `IDisposable`, `ILogService` |
| `_` (prefix) | プライベートメンバー | `_store`, `_isDisposed` |
| `Service` (suffix) | サービスクラス | `ConfigurationService` |
| `Contribution` (suffix) | 寄与/登録クラス | `WorkbenchContribution` |
| `Main` (suffix) | Mainプロセス用 | `EnvironmentMainService` |
| `Handler` (suffix) | イベントハンドラー | `URLHandler` |

### 3.3 プログラム分類一覧

| カテゴリ | ディレクトリ | 説明 |
|---------|-------------|------|
| 基盤ユーティリティ | `src/vs/base/` | 共通ユーティリティ、データ構造 |
| プラットフォーム | `src/vs/platform/` | 環境非依存のプラットフォームサービス |
| エディタコア | `src/vs/editor/` | Monaco Editor本体 |
| ワークベンチ | `src/vs/workbench/` | VS Code UI全体 |
| コード（デスクトップ） | `src/vs/code/` | Electronデスクトップアプリ固有 |
| サーバー | `src/vs/server/` | リモート開発サーバー |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
vscode-main/
├── src/                          # ソースコード
│   ├── main.ts                   # Electronメインエントリーポイント
│   ├── cli.ts                    # CLIエントリーポイント
│   ├── server-main.ts            # サーバーエントリーポイント
│   └── vs/                       # VS Codeコアコード
│       ├── base/                 # 基盤ユーティリティ
│       │   ├── common/           # プラットフォーム非依存
│       │   ├── browser/          # ブラウザ用
│       │   ├── node/             # Node.js用
│       │   └── parts/            # IPC等のパーツ
│       ├── platform/             # プラットフォームサービス
│       │   ├── configuration/    # 設定管理
│       │   ├── files/            # ファイルシステム
│       │   ├── instantiation/    # 依存性注入
│       │   ├── log/              # ログ
│       │   └── ...               # 他多数のサービス
│       ├── editor/               # Monaco Editor
│       │   ├── common/           # エディタコア
│       │   ├── browser/          # エディタUI
│       │   └── contrib/          # エディタ寄与機能
│       ├── workbench/            # ワークベンチ（VS Code UI）
│       │   ├── common/           # 共通定義
│       │   ├── browser/          # ブラウザUI
│       │   ├── contrib/          # 機能モジュール
│       │   ├── services/         # ワークベンチサービス
│       │   └── api/              # 拡張機能API
│       └── code/                 # デスクトップアプリ固有
│           ├── electron-main/    # Mainプロセス
│           └── electron-browser/ # Rendererプロセス
├── extensions/                   # 組み込み拡張機能
├── build/                        # ビルドスクリプト
├── test/                         # テスト
└── scripts/                      # 開発用スクリプト
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/vs/base/` | 共通ユーティリティ、データ構造 | `lifecycle.ts`, `event.ts`, `uri.ts` |
| `src/vs/platform/` | プラットフォームサービス | `*Service.ts`, `instantiation.ts` |
| `src/vs/editor/` | Monaco Editorコア | `editor.all.ts`, `editorBrowser.ts` |
| `src/vs/workbench/` | ワークベンチUI全体 | `workbench.common.main.ts` |
| `src/vs/workbench/contrib/` | 機能モジュール | 各機能の`*.contribution.ts` |
| `src/vs/workbench/services/` | ワークベンチサービス | 各種`*Service.ts` |
| `src/vs/code/electron-main/` | Mainプロセス | `main.ts`, `app.ts` |
| `extensions/` | 組み込み拡張機能 | 言語サポート、テーマ等 |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

VS Codeは**レイヤードアーキテクチャ**と**依存性注入**を組み合わせた構造を採用しています。

```mermaid
graph TB
    subgraph "Extensions Layer"
        EXT[拡張機能]
    end

    subgraph "Workbench Layer"
        WB[Workbench UI]
        CONTRIB[Contributions]
        WBSVC[Workbench Services]
    end

    subgraph "Platform Layer"
        PLAT[Platform Services]
        DI[Instantiation Service]
    end

    subgraph "Base Layer"
        BASE[Base Utilities]
        LIFECYCLE[Lifecycle]
        EVENT[Event System]
    end

    subgraph "Runtime"
        ELECTRON[Electron]
        NODE[Node.js]
        BROWSER[Browser APIs]
    end

    EXT --> WB
    WB --> CONTRIB
    CONTRIB --> WBSVC
    WBSVC --> PLAT
    PLAT --> DI
    DI --> BASE
    BASE --> LIFECYCLE
    BASE --> EVENT
    PLAT --> ELECTRON
    PLAT --> NODE
    PLAT --> BROWSER
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Base | 共通ユーティリティ、データ構造 | `lifecycle.ts`, `event.ts`, `uri.ts` |
| Platform | 環境抽象化、DIコンテナ | `instantiation.ts`, `*Service.ts` |
| Editor | Monaco Editorコア | `editor/common/*`, `editor/browser/*` |
| Workbench | VS Code UI、機能モジュール | `workbench/browser/*`, `workbench/contrib/*` |
| Extensions | 拡張機能API、拡張ホスト | `workbench/api/*` |

### 5.3 データフロー

1. **起動フロー**:
   - `src/main.ts` (Electronエントリー)
   - `src/vs/code/electron-main/main.ts` (CodeMain)
   - `src/vs/code/electron-main/app.ts` (CodeApplication)
   - サービスの初期化 → ウィンドウ作成

2. **サービス登録フロー**:
   - `registerSingleton()` でサービスを登録
   - `createDecorator()` でサービス識別子を作成
   - `InstantiationService` が依存関係を解決

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

**Electronアプリのエントリーポイント:**

```typescript
// ファイル: src/main.ts:147-183
app.once('ready', function () {
    if (args['trace']) {
        // トレース設定
        contentTracing.startRecording(traceOptions).finally(() => onReady());
    } else {
        onReady();
    }
});
```

**メインアプリケーションクラス:**

```typescript
// ファイル: src/vs/code/electron-main/main.ts:86-95
class CodeMain {
    main(): void {
        try {
            this.startup();
        } catch (error) {
            console.error(error.message);
            app.exit(1);
        }
    }
}
```

### 6.2 ビジネスロジック

ビジネスロジックは主に `workbench/contrib/` 配下の機能モジュールと `workbench/services/` のサービスに分散しています。

**サービス登録の例:**

```typescript
// ファイル: src/vs/platform/instantiation/common/extensions.ts:25-33
export function registerSingleton<T, Services extends BrandedService[]>(
    id: ServiceIdentifier<T>,
    ctorOrDescriptor: { new(...services: Services): T } | SyncDescriptor<any>,
    supportsDelayedInstantiation?: boolean | InstantiationType
): void {
    if (!(ctorOrDescriptor instanceof SyncDescriptor)) {
        ctorOrDescriptor = new SyncDescriptor<T>(ctorOrDescriptor as new (...args: any[]) => T, [], Boolean(supportsDelayedInstantiation));
    }
    _registry.push([id, ctorOrDescriptor]);
}
```

### 6.3 データアクセス

ファイルシステムへのアクセスは `IFileService` を通じて抽象化されています。

```typescript
// ファイル: src/vs/code/electron-main/main.ts:187-192
const fileService = new FileService(logService);
services.set(IFileService, fileService);
const diskFileSystemProvider = new DiskFileSystemProvider(logService);
fileService.registerProvider(Schemas.file, diskFileSystemProvider);
```

### 6.4 ユーティリティ/共通機能

`src/vs/base/` 配下に多くのユーティリティが用意されています。

| ファイル | 機能 |
|---------|------|
| `lifecycle.ts` | Disposableパターン、リソース管理 |
| `event.ts` | イベントシステム |
| `uri.ts` | URI操作 |
| `async.ts` | 非同期ユーティリティ |
| `strings.ts` | 文字列操作 |
| `arrays.ts` | 配列操作 |

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Disposable | リソース管理 | 高 | `lifecycle.ts` |
| Service Locator (DI) | 依存性注入 | 高 | `instantiation.ts` |
| Event Emitter | イベント駆動 | 高 | `event.ts` |
| Contribution | 機能登録 | 中 | `*.contribution.ts` |
| Registry | 登録パターン | 中 | `extensions.ts` |

### 各パターンの詳細

#### パターン1: Disposable

**目的:** リソースのライフサイクル管理、メモリリーク防止

**実装例:**
```typescript
// ファイル: src/vs/base/common/lifecycle.ts:526-557
export abstract class Disposable implements IDisposable {
    static readonly None = Object.freeze<IDisposable>({ dispose() { } });

    protected readonly _store = new DisposableStore();

    constructor() {
        trackDisposable(this);
        setParentOfDisposable(this._store, this);
    }

    public dispose(): void {
        markAsDisposed(this);
        this._store.dispose();
    }

    protected _register<T extends IDisposable>(o: T): T {
        if ((o as unknown as Disposable) === this) {
            throw new Error('Cannot register a disposable on itself!');
        }
        return this._store.add(o);
    }
}
```

**解説:** `Disposable`を継承したクラスは、内部の`_store`に登録されたリソースを`dispose()`時に一括で解放します。`_register()`メソッドでリソースを追加します。

#### パターン2: Service Decorator (依存性注入)

**目的:** サービスの依存関係を宣言的に定義し、テスト容易性を向上

**実装例:**
```typescript
// ファイル: src/vs/platform/instantiation/common/instantiation.ts:109-126
export function createDecorator<T>(serviceId: string): ServiceIdentifier<T> {
    if (_util.serviceIds.has(serviceId)) {
        return _util.serviceIds.get(serviceId)!;
    }

    const id = function (target: Function, key: string, index: number) {
        if (arguments.length !== 3) {
            throw new Error('@IServiceName-decorator can only be used to decorate a parameter');
        }
        storeServiceDependency(id, target, index);
    } as ServiceIdentifier<T>;

    id.toString = () => serviceId;
    _util.serviceIds.set(serviceId, id);
    return id;
}
```

**解説:** `createDecorator`で作成したデコレータをコンストラクタパラメータに付与することで、`InstantiationService`が自動的に依存関係を解決・注入します。

#### パターン3: Event Emitter

**目的:** コンポーネント間の疎結合な通信

**実装例:**
```typescript
// ファイル: src/vs/base/common/event.ts:37-42
export interface Event<T> {
    (listener: (e: T) => unknown, thisArgs?: any, disposables?: IDisposable[] | DisposableStore): IDisposable;
}

export namespace Event {
    export const None: Event<any> = () => Disposable.None;
}
```

**解説:** イベントはリスナーを受け取り、`IDisposable`を返す関数として定義されています。これによりリスナーの解除が容易になります。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: アプリケーション起動フロー

**概要:** VS Codeアプリケーションの起動から初期画面表示までの流れ

**処理フロー:**
```
src/main.ts → CodeMain.startup() → createServices() → CodeApplication.startup()
```

**詳細な追跡:**

1. **Electronエントリーポイント** (`src/main.ts:147`)
   ```typescript
   app.once('ready', function () {
       // ...
       onReady();
   });
   ```

2. **メインスタートアップ** (`src/vs/code/electron-main/main.ts:97-160`)
   ```typescript
   private async startup(): Promise<void> {
       setUnexpectedErrorHandler(err => console.error(err));
       const [instantiationService, instanceEnvironment, ...] = this.createServices();
       // ...
       await instantiationService.invokeFunction(async accessor => {
           return instantiationService.createInstance(CodeApplication, ...).startup();
       });
   }
   ```

3. **サービス作成** (`src/vs/code/electron-main/main.ts:162-248`)
   ```typescript
   private createServices(): [...] {
       const services = new ServiceCollection();
       // Product, Environment, Logger, Log, Files, State...
       services.set(IProductService, productService);
       services.set(IEnvironmentMainService, environmentMainService);
       // ...
       return [new InstantiationService(services, true), ...];
   }
   ```

4. **アプリケーション起動** (`src/vs/code/electron-main/app.ts`)
   ```typescript
   export class CodeApplication extends Disposable {
       // ウィンドウ作成、拡張機能読み込み等
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. `CONTRIBUTING.md` - 開発ガイド
3. このコードリーディングガイドライン
4. `src/vs/workbench/workbench.common.main.ts` - 機能一覧

#### 特定機能を理解したい場合
1. `src/vs/workbench/contrib/{機能名}/` - 機能モジュール
2. `src/vs/workbench/services/{関連サービス}/` - サービス実装
3. 関連するテストファイル

#### 改修作業を行う場合
1. 対象機能の`*.contribution.ts`
2. 関連するサービスインターフェース
3. 影響範囲の確認（依存関係グラフ）
4. テストファイル

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要 | 最初 |
| `CONTRIBUTING.md` | 開発貢献ガイド | 開発開始時 |
| Wiki (GitHub) | 詳細ドキュメント | 必要に応じて |
| `*.contribution.ts` | 機能登録ファイル | 機能理解時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: サービスの実装を見つけるには？
A: サービスインターフェース名（例: `ILogService`）で検索し、`registerSingleton`を呼び出している箇所を探してください。実装クラスがそこで登録されています。

#### Q: 機能の初期化処理を見つけるには？
A: `*.contribution.ts`ファイルを探してください。多くの機能はこのファイルでワークベンチに登録されます。

#### Q: Electron MainとRendererの通信を理解するには？
A: `src/vs/base/parts/ipc/` 配下のIPCチャネル実装を参照してください。`ProxyChannel`等を使用してサービスをプロセス間で公開しています。

#### Q: 拡張機能APIの実装を見つけるには？
A: `src/vs/workbench/api/` 配下を参照してください。`extHost*.ts`がExtension Host側、`mainThread*.ts`がMain側の実装です。

### デバッグのヒント

1. **ログ出力**: `ILogService`を使用してデバッグログを出力
2. **Developer Tools**: `Help > Toggle Developer Tools`でChromium DevToolsを開く
3. **拡張機能ホストログ**: `Output > Extension Host`で拡張機能のログを確認
4. **パフォーマンス**: `Developer: Startup Performance`コマンドで起動パフォーマンスを分析

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Disposable | `dispose()`メソッドを持つリソース解放可能なオブジェクト |
| Service | DIコンテナで管理されるシングルトンサービス |
| Contribution | ワークベンチやエディタに機能を追加する登録 |
| Extension Host | 拡張機能を実行する別プロセス |
| Workbench | VS Code UIの全体（サイドバー、エディタ、パネル等） |
| Monaco Editor | VS Codeのエディタコンポーネント |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/main.ts` | Electronエントリー | アプリ起動、初期設定 |
| `src/vs/base/common/lifecycle.ts` | Disposableパターン | リソース管理 |
| `src/vs/base/common/event.ts` | イベントシステム | Event, Emitter |
| `src/vs/platform/instantiation/` | DIコンテナ | InstantiationService |
| `src/vs/code/electron-main/` | Mainプロセス | app.ts, main.ts |
| `src/vs/workbench/` | ワークベンチ全体 | UI、サービス、機能 |
| `extensions/` | 組み込み拡張機能 | 言語サポート等 |

### C. 参考資料

- [VS Code公式ドキュメント](https://code.visualstudio.com/docs)
- [VS Code API Reference](https://code.visualstudio.com/api/references/vscode-api)
- [TypeScript公式ドキュメント](https://www.typescriptlang.org/docs/)
- [Electron公式ドキュメント](https://www.electronjs.org/docs)
- [GitHub Wiki: How to Contribute](https://github.com/microsoft/vscode/wiki/How-to-Contribute)
- [GitHub Wiki: Coding Guidelines](https://github.com/microsoft/vscode/wiki/Coding-Guidelines)
