# セキュリティ設計書

## 概要

本ドキュメントは、Visual Studio Code（VSCode）のセキュリティ設計を記述したものです。VSCodeはElectronベースのデスクトップアプリケーションであり、Webブラウザベースのリモート開発環境もサポートしています。本設計書では、認証、認可、暗号化、入出力対策、監査ログなどのセキュリティ機能について説明します。

## 認証設計

### 認証方式

VSCodeは複数の認証方式をサポートしています：

| 認証方式 | 説明 | 対応規格 |
| --- | --- | --- |
| OAuth 2.0 | 外部サービス連携用の標準認証プロトコル | RFC 6749 |
| OpenID Connect | OAuth 2.0上に構築された認証レイヤー | OpenID Connect 1.0 |
| Device Code Flow | デバイス認可グラント（CLI等の制限環境向け） | RFC 8628 |
| Dynamic Client Registration | 動的クライアント登録 | RFC 7591 |

**実装ファイル：**
- `src/vs/base/common/oauth.ts`: OAuth 2.0/OIDC認証の共通実装
- `src/vs/workbench/api/browser/mainThreadAuthentication.ts`: 認証プロバイダー管理

**認証フロー：**
1. Authorization Code Flow（PKCE対応）
2. Device Authorization Grant（device_code, user_code）
3. Refresh Token Flow

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | トークンの`expires_in`に依存 | OAuth 2.0トークンレスポンスに基づく |
| セッション固定化対策 | stateパラメータによるCSRF防止 | 認可リクエストごとに一意のstateを生成 |
| リフレッシュトークン | サポート | `refresh_token`によるトークン更新 |
| Cookie設定（Web版） | SameSite=Lax, maxAge=7日 | connectionTokenCookieに適用 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 拡張機能 | スコープベースのアクセス | OAuth scopesにより制限 |
| Webview | 制限付きパーミッション | allowedPermissionsInWebviewで定義 |
| メインウィンドウ | 拡張パーミッション | allowedPermissionsInCoreで定義 |
| 外部プロセス | サンドボックス化 | Electronサンドボックスによる分離 |

### アクセス制御

**パーミッション制御（Electron）：**

メインプロセスでは、セッションベースのパーミッション制御を実装：

```typescript
// 常時許可されるパーミッション
const alwaysAllowedPermissions = ['pointerLock', 'notifications'];

// Webview内で許可されるパーミッション
const allowedPermissionsInWebview = [
  ...alwaysAllowedPermissions,
  'clipboard-read',
  'clipboard-sanitized-write'
];

// コアウィンドウで許可されるパーミッション
const allowedPermissionsInCore = [
  ...alwaysAllowedPermissions,
  'media',
  'local-fonts'
];
```

**拡張機能のアクセス制御：**
- `IAuthenticationAccessService`: 拡張機能の認証アクセス許可を管理
- 拡張機能ごとに認証プロバイダーへのアクセス許可を個別管理
- ユーザーの明示的な同意（loginPrompt）が必要

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | OAuth認可サーバーとの通信はHTTPS必須（http://またはhttps://スキームを検証） |
| CORS | PRSS CDN (vscode.download.prss.microsoft.com) に対してAccess-Control-Allow-Origin: * を許可 |
| SVGリクエスト | 信頼されたスキーム（file, vscode-file-resource等）のみ許可 |
| リダイレクトURI | 127.0.0.1およびvscode.dev/redirectのみ許可 |

**リクエストフィルタリング：**
- `onBeforeRequest`: 不正なvscode-webview、vscode-file-resourceリクエストをブロック
- `onHeadersReceived`: SVGのContent-Typeを適切に設定、不正なSVGをブロック

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| ファイルチェックサム | SHA-256（`src/vs/base/node/crypto.ts`） |
| 文字列ハッシュ | SHA-1（`src/vs/base/common/hash.ts` - StringSHA1クラス） |
| JWTトークン | JWSサポート（署名検証） |
| クライアントシークレット | 安全なストレージに保存 |

**実装詳細：**
- `checksum()`: ファイルの整合性検証（SHA-256）
- `hashAsync()`: 非同期ハッシュ計算（Web Crypto API使用）
- `StringSHA1`: 同期的なSHA-1ハッシュ（メモリ効率重視）

### 機密情報管理

**JWTクレーム処理：**
- `getClaimsFromJWT()`: Base64デコードによるペイロード抽出
- トークンの有効期限（exp）、発行時刻（iat）、発行者（iss）を検証

**クライアント資格情報：**
- 動的クライアント登録により取得したclient_id, client_secretは専用ストレージに保存
- `IDynamicAuthenticationProviderStorageService`による安全な管理

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | DOMPurify 3.2.7によるHTMLサニタイズ |
| SQLインジェクション | N/A（データベース未使用） |
| CSRF | OAuth stateパラメータ、SameSite Cookie |
| コードインジェクション | 許可タグ・属性のホワイトリスト |
| プロトコルハイジャック | URLスキームのバリデーション |

### HTMLサニタイズ

**DOMPurify設定（`src/vs/base/browser/domSanitize.ts`）：**

許可されたHTMLタグ：
```typescript
const basicMarkupHtmlTags = [
  'a', 'abbr', 'b', 'blockquote', 'br', 'code', 'div',
  'em', 'h1'-'h6', 'hr', 'i', 'img', 'li', 'ol', 'p',
  'pre', 'span', 'strong', 'table', 'td', 'th', 'tr', 'ul', ...
];
```

許可された属性：
```typescript
const defaultAllowedAttrs = [
  'href', 'target', 'src', 'alt', 'title', 'name', 'role',
  'tabindex', 'required', 'checked', 'placeholder', 'type', ...
];
```

**リンクバリデーション：**
- デフォルトで許可されるプロトコル：`http`, `https`
- 相対パスは設定により許可可能
- `validateLink()`: URLスキームの検証

### 入力検証

**バリデーションフレームワーク（`src/vs/base/common/validation.ts`）：**
- 型ベースのバリデーター（string, number, boolean, object）
- オブジェクト構造の検証（vObj）
- 配列・タプルの検証
- Union型・Enum型のサポート

### Content Security Policy (CSP)

**Webview CSP：**
```typescript
// 拡張機能のロケーションに基づくCSPソース
public get cspSource(): string {
  if (extensionLocation.scheme === 'https' || extensionLocation.scheme === 'http') {
    return extensionCspRule + ' ' + webviewGenericCspSource;
  }
  return webviewGenericCspSource;
}
```

**CSP欠落警告：**
- `$onMissingCsp()`: CSPなしでWebviewを作成した拡張機能に警告を出力

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証イベント | プロバイダー使用状況、セッション変更 | アプリケーション実行中 |
| セキュリティ警告 | ブロックされたリクエスト、CSP警告 | ログローテーションに依存 |
| テレメトリ | 拡張機能の認証プロバイダー使用 | プライバシー設定に依存 |

**テレメトリイベント：**
- `authentication.providerUsage`: 拡張機能が使用した認証プロバイダーを記録
- `authentication.clientIdUsage`: カスタムクライアントID使用を記録
- `authentication.providerNotDeclared`: 未宣言の認証プロバイダー使用を記録

**ログ出力：**
- `ILogService`: 構造化ログサービス
- セキュリティ関連エラーは`logService.error()`で記録
- 警告は`logService.warn()`で記録

## 追加のセキュリティ対策

### Electronセキュリティ

**ウィンドウ制御：**
- `setWindowOpenHandler`: about:blank以外のウィンドウオープンを制限
- `will-navigate`: ページ内ナビゲーションを防止
- 補助ウィンドウの登録管理

**UNCパスアクセス制御（Windows）：**
- `security.restrictUNCAccess`: UNCアクセス制限の有効/無効
- `security.allowedUNCHosts`: 許可されたUNCホストのホワイトリスト

### プロトコルハンドリング

**ファイルプロトコル確認：**
- `security.promptForLocalFileProtocolHandling`: ローカルファイルプロトコルの確認ダイアログ
- `security.promptForRemoteFileProtocolHandling`: リモートファイルプロトコルの確認ダイアログ

### サンドボックス化

- Electronサンドボックスによるレンダラープロセスの分離
- Webviewは独自のオリジン（vscode-webview://）で動作
- 拡張機能ホストプロセスの分離

## 備考

- 本設計書はVSCodeのコードベースから抽出したセキュリティ機能を文書化したものです
- Electronのセキュリティガイドライン（https://electronjs.org/docs/tutorial/security）に準拠
- OAuth 2.0関連の実装はRFC 6749, RFC 7591, RFC 8628に準拠
- DOMPurifyはv3.2.7を使用（Apache License 2.0, Mozilla Public License 2.0）
