# データベース設計書

## 概要

本ドキュメントは、Visual Studio Code (VS Code) のデータベース/ストレージ設計を記述したものです。VS Code は、デスクトップアプリケーション（Electron）とWebアプリケーション（ブラウザ）の両方で動作するため、プラットフォームに応じて異なるストレージ技術を使用しています。

**デスクトップ版（Electron/Node.js環境）:**
- SQLite (`@vscode/sqlite3`) を使用した永続化ストレージ

**Web版（ブラウザ環境）:**
- IndexedDB を使用した永続化ストレージ

両プラットフォームともKey-Value形式の単一テーブル構造を採用し、3つのスコープ（Application、Profile、Workspace）でデータを管理しています。

## テーブル一覧

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| ItemTable | SQLiteStorageDatabase / IndexedDBStorageDatabase | Key-Value形式でアプリケーション状態を保存する主要テーブル |
| IndexedDB ObjectStore (ファイルシステム用) | IndexedDBFileSystemProvider | ブラウザ環境でファイルシステムをエミュレートするためのストア |

## 各テーブル定義

### 1. ItemTable（SQLiteストレージ）

デスクトップ版VS Codeで使用されるSQLiteデータベースの主要テーブルです。アプリケーションの状態、設定、ワークスペース情報などをKey-Value形式で保存します。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| key | TEXT | NOT NULL | ストレージエントリの一意キー（UNIQUE制約、競合時は置換） |
| value | BLOB | YES | ストレージエントリの値（文字列またはバイナリデータ） |

**制約・インデックス:**
- PRIMARY KEY: なし（keyカラムにUNIQUE ON CONFLICT REPLACE制約）
- UNIQUE: key (ON CONFLICT REPLACE)

**テーブル作成SQL:**
```sql
CREATE TABLE IF NOT EXISTS ItemTable (key TEXT UNIQUE ON CONFLICT REPLACE, value BLOB)
```

**ストレージファイルの配置:**
- アプリケーション/プロファイルストレージ: `{globalStorageHome}/state.vscdb`
- ワークスペースストレージ: `{workspaceStorageHome}/{workspaceId}/state.vscdb`

### 2. ItemTable（IndexedDBストレージ）

ブラウザ版VS Codeで使用されるIndexedDBのオブジェクトストアです。SQLite版と同様のKey-Value構造を持ちます。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| key | String | NOT NULL | ストレージエントリの一意キー（Object Storeのキー） |
| value | String | NO | ストレージエントリの値（文字列形式） |

**データベース命名規則:**
- プレフィックス: `vscode-web-state-db-`
- アプリケーションストレージ: `vscode-web-state-db-global`
- プロファイルストレージ: `vscode-web-state-db-global-{profileId}`
- ワークスペースストレージ: `vscode-web-state-db-{workspaceId}`

### 3. IndexedDBファイルシステム用ObjectStore

ブラウザ環境でファイルシステムをエミュレートするためのIndexedDB ObjectStoreです。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| key | String | NOT NULL | ファイルパス（Object Storeのキー） |
| value | Uint8Array / String | NO | ファイルコンテンツ（バイナリまたは文字列） |

**使用用途:**
- ブラウザ環境での仮想ファイルシステム
- 拡張機能のファイル保存
- ワークスペース設定の保存

## ストレージスコープ

VS Codeのストレージは3つのスコープで管理されています。

| スコープ | 説明 | 値 |
| --- | --- | --- |
| APPLICATION | 全プロファイル・全ワークスペース共通のデータ | -1 |
| PROFILE | 同一プロファイル内の全ワークスペース共通のデータ | 0 |
| WORKSPACE | 現在のワークスペース固有のデータ | 1 |

## ストレージターゲット

| ターゲット | 説明 | 値 |
| --- | --- | --- |
| USER | ユーザー固有のデータ（マシン間で同期可能） | 0 |
| MACHINE | マシン固有のデータ（同期対象外） | 1 |

## システム予約キー

| キー | 説明 |
| --- | --- |
| `__$__isNewStorageMarker` | ストレージが新規作成されたかどうかを示すマーカー |
| `__$__targetStorageMarker` | 各キーのストレージターゲット（USER/MACHINE）を管理するJSON |
| `telemetry.firstSessionDate` | 初回セッション日時 |
| `telemetry.lastSessionDate` | 前回セッション日時 |
| `telemetry.currentSessionDate` | 現在セッション日時 |

## データベース操作

### 主要操作メソッド

| 操作 | SQLite | IndexedDB |
| --- | --- | --- |
| 全データ取得 | `SELECT * FROM ItemTable` | `IDBObjectStore.openCursor()` |
| データ挿入/更新 | `INSERT INTO ItemTable VALUES (?,?) ON CONFLICT (key) DO UPDATE SET value = excluded.value` | `IDBObjectStore.put(value, key)` |
| データ削除 | `DELETE FROM ItemTable WHERE key IN (?)` | `IDBObjectStore.delete(key)` |
| 最適化 | `VACUUM` | 非対応 |
| 整合性チェック | `PRAGMA integrity_check` / `PRAGMA quick_check` | 非対応 |

### トランザクション管理

**SQLite:**
- `BEGIN TRANSACTION` / `END TRANSACTION` で明示的なトランザクション管理
- バッチ処理でパラメータを256個単位で分割

**IndexedDB:**
- `IDBDatabase.transaction()` によるトランザクション管理
- 読み取り専用（readonly）と読み書き（readwrite）モードをサポート

## バックアップ・リカバリ

### SQLiteストレージ

- 正常クローズ時に `.backup` ファイルを作成
- 起動時にDBが破損している場合、バックアップからの復元を試行
- 復元も失敗した場合、インメモリDBにフォールバック

### IndexedDBストレージ

- 必要なObject Storeが存在しない場合、DBを再作成
- エラー発生時、インメモリストレージにフォールバック

## 備考

### パフォーマンス考慮事項

1. **フラッシュ間隔**
   - デフォルト: 60秒（デスクトップ）、5秒（ブラウザ）
   - 遅延書き込みにより、頻繁な更新を効率的にバッチ処理

2. **SQLITE_BUSY対応**
   - タイムアウト: 2000ms
   - 別プロセスがDBをロックしている場合のリトライ機構

3. **ブラウザ環境での制限**
   - Safari: ページハング防止のため、クローズ時にトランザクションをキャンセル
   - ストレージクォータ超過時のエラーハンドリング

### クロスウィンドウ同期

- BroadcastChannel APIを使用して、複数ウィンドウ/タブ間でストレージ変更を同期
- Application/Profileスコープのストレージで有効

### インメモリストレージ

テスト環境やストレージが利用できない場合のフォールバックとして、インメモリストレージ（`InMemoryStorageDatabase`）を提供しています。
