# 機能設計書：カラーピッカー（Color Picker）

## 1. 機能概要

### 1.1 機能の目的
カラーピッカー機能は、コード内の色指定（CSS色、RGB値等）を視覚的に表示し、インタラクティブに色を編集できる機能である。開発者が色を直感的に選択・変更できるようにすることを目的とする。

### 1.2 機能の範囲
- 色コードの検出と視覚的表示（カラーデコレータ）
- ホバー時のカラーピッカーウィジェット表示
- 色の編集と即座のコード反映
- 複数の色形式（HEX、RGB、HSL等）のサポート

### 1.3 関連画面
- エディタメインビュー（カラーデコレータ）
- カラーピッカーホバーウィジェット

## 2. 機能要件

### 2.1 ユーザーストーリー
- 開発者として、色コードの横に実際の色を表示したい
- 開発者として、色をクリックしてカラーピッカーで変更したい
- 開発者として、色形式（HEX/RGB/HSL）を切り替えたい

### 2.2 入力仕様
| 項目 | 説明 |
|------|------|
| ドキュメント | 現在開いているテキストモデル（ITextModel） |
| 言語ID | ドキュメントの言語識別子 |
| 設定 | colorDecorators、colorDecoratorsLimit |

### 2.3 出力仕様
| 項目 | 説明 |
|------|------|
| IColorData[] | 検出された色情報のリスト |
| デコレーション | 色サンプルの表示 |
| カラーピッカーUI | 色選択ウィジェット |

### 2.4 処理フロー

```mermaid
sequenceDiagram
    participant Editor
    participant ColorDetector
    participant ColorProvider
    participant ColorPickerWidget

    Editor->>ColorDetector: ドキュメント変更
    ColorDetector->>ColorProvider: provideDocumentColors()
    ColorProvider-->>ColorDetector: IColorInformation[]
    ColorDetector->>Editor: カラーデコレータ表示

    Note over Editor,ColorPickerWidget: ホバー時
    Editor->>ColorPickerWidget: ホバーイベント
    ColorPickerWidget->>ColorProvider: provideColorPresentations()
    ColorProvider-->>ColorPickerWidget: 色形式オプション
    ColorPickerWidget->>Editor: カラーピッカー表示

    Note over Editor,ColorPickerWidget: 色変更時
    ColorPickerWidget->>Editor: 新しい色を適用
```

### 2.5 ビジネスルール
1. `editor.colorDecorators`がfalseの場合はカラーデコレータを表示しない
2. デコレータの最大数は`editor.colorDecoratorsLimit`で制限
3. 言語拡張機能がColorProviderを提供していない場合は機能が無効
4. 色形式の変換はColorProviderが提供

## 3. 画面設計

### 3.1 画面レイアウト
カラーデコレータは色コードの前に小さな色サンプルとして表示される。

### 3.2 画面要素
| 要素 | 説明 | 操作 |
|------|------|------|
| カラーデコレータ | 色コードの前の色サンプル | ホバーでピッカー表示 |
| カラーピッカー | 色選択UI | 色の選択、形式変更 |
| 色形式セレクタ | HEX/RGB/HSL切り替え | クリックで形式変更 |
| 透明度スライダー | アルファ値調整 | ドラッグで透明度変更 |

## 4. データ設計

### 4.1 入力データ構造
```typescript
interface DocumentColorParams {
    model: ITextModel;
    token: CancellationToken;
}
```

### 4.2 出力データ構造
```typescript
interface IColorInformation {
    range: IRange;           // 色コードの範囲
    color: IColor;           // 色情報
}

interface IColor {
    red: number;             // 0-1
    green: number;           // 0-1
    blue: number;            // 0-1
    alpha: number;           // 0-1
}

interface IColorData {
    colorInfo: IColorInformation;
    provider: DocumentColorProvider;
}

interface ColorPresentation {
    label: string;           // 表示形式（例: "#FF0000"）
    textEdit?: TextEdit;     // 適用する編集
}
```

## 5. コードリーディングガイド

### 5.1 推奨読解順序

1. **データ構造の理解**
   - `src/vs/editor/common/languages.ts` - IColor、IColorInformation等の型定義

2. **検出層**
   - `src/vs/editor/contrib/colorPicker/browser/colorDetector.ts` - 色検出機能（**29-293行目**）
     - `ColorDetector`クラスが色の検出と表示を制御
     - **52-89行目**: コンストラクタでイベントハンドラ登録
     - **124-146行目**: `updateColors()`で検出処理開始
     - **148-167行目**: `beginCompute()`で非同期に色を取得
     - **181-198行目**: `updateDecorations()`でデコレーション更新
     - **202-242行目**: `updateColorDecorators()`でカラーサンプル表示

3. **カラーピッカーウィジェット**
   - `src/vs/editor/contrib/colorPicker/browser/colorPickerWidget.ts` - ピッカーUI
     - カラーパレット、スライダー、形式切り替えの実装

### 5.2 プログラム呼び出し階層図

```
colorDetector.ts
  └── ColorDetector
        ├── updateColors()
        │     └── beginCompute()
        │           └── getColors()
        │                 └── ColorProvider.provideDocumentColors()
        ├── updateDecorations()
        │     └── エディタデコレーション更新
        └── updateColorDecorators()
              └── カラーサンプル表示

colorPickerWidget.ts (ホバー時)
  └── ColorPickerWidget
        ├── ColorProvider.provideColorPresentations()
        └── 色形式オプション表示
```

### 5.3 データフロー図

```
[ドキュメント変更]
        ↓
[ColorDetector.updateColors()]
        ↓
[ColorProvider.provideDocumentColors()]
        ↓
[IColorData[] - 色情報リスト]
        ↓
[updateColorDecorators() - カラーサンプル表示]
        ↓
[Editor Display]

[ホバー]
        ↓
[ColorPickerWidget表示]
        ↓
[ColorProvider.provideColorPresentations()]
        ↓
[色形式オプション表示]
```

### 5.4 読解のコツ
- `ColorDetector`は`IEditorContribution`を実装
- `RECOMPUTE_TIME`（1000ms）でデバウンス
- `colorDecoratorsLimit`で表示上限を制御
- RGBA8クラスで色をRGBAバイト値に変換
- `DynamicCssRules`で動的に背景色CSSを生成

### 5.5 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/editor/contrib/colorPicker/browser/colorDetector.ts` | コントローラ | 色検出機能 |
| `src/vs/editor/contrib/colorPicker/browser/colorPickerWidget.ts` | ウィジェット | カラーピッカーUI |
| `src/vs/editor/contrib/colorPicker/browser/color.ts` | ユーティリティ | 色取得ロジック |
| `src/vs/editor/contrib/colorPicker/browser/colorHoverParticipant.ts` | ホバー | ホバー時の処理 |
| `src/vs/editor/common/languages.ts` | 型定義 | IColor等の型 |
| `src/vs/base/common/color.ts` | ユーティリティ | 色変換処理 |

## 6. 非機能要件

### 6.1 パフォーマンス
- デバウンス（1000ms）で更新頻度を制御
- デコレータ数の上限設定（デフォルト500）
- `DecoratorLimitReporter`で制限到達を通知

### 6.2 エラーハンドリング
- ColorProviderが存在しない場合は機能を無効化
- 色解析エラーは個別にキャッチ

## 7. 設計上の考慮事項

### 7.1 拡張性
- Language Server Protocol (LSP)のtextDocument/documentColorに対応
- カスタムColorProviderの追加が可能

### 7.2 依存関係
- `ILanguageFeaturesService`を通じてColorProviderにアクセス
- `IConfigurationService`で設定管理
- `DynamicCssRules`で動的スタイル生成
