# 機能設計書：行番号表示（Line Numbers）

## 1. 機能概要

### 1.1 機能の目的
行番号表示機能は、エディタの左側に各行の番号を表示する機能である。開発者がコード内の位置を素早く特定し、エラーメッセージやログの行番号参照を容易にすることを目的とする。

### 1.2 機能の範囲
- 絶対行番号の表示
- 相対行番号の表示（現在行からの距離）
- 間隔行番号の表示（10行ごと等）
- 行番号のカスタムレンダリング
- 現在行のハイライト

### 1.3 関連画面
- エディタメインビュー（左マージン）

## 2. 機能要件

### 2.1 ユーザーストーリー
- 開発者として、各行の行番号を確認したい
- 開発者として、Vimスタイルの相対行番号を使用したい
- 開発者として、現在行を視覚的に区別したい

### 2.2 入力仕様
| 項目 | 説明 |
|------|------|
| ビューモデル | 現在のビューモデル（ViewContext） |
| 設定 | lineNumbers（on/off/relative/interval） |
| カーソル位置 | 現在のカーソル行番号 |

### 2.3 出力仕様
| 項目 | 説明 |
|------|------|
| 行番号HTML | 各行の行番号を含むHTML文字列 |
| スタイル | 幅、位置、アクティブ行のスタイル |

### 2.4 処理フロー

```mermaid
sequenceDiagram
    participant ViewEvents
    participant LineNumbersOverlay
    participant ViewContext
    participant Editor

    ViewEvents->>LineNumbersOverlay: onConfigurationChanged
    ViewEvents->>LineNumbersOverlay: onCursorStateChanged
    ViewEvents->>LineNumbersOverlay: onScrollChanged
    LineNumbersOverlay->>LineNumbersOverlay: prepareRender()
    LineNumbersOverlay->>ViewContext: getLineRenderLineNumber()
    LineNumbersOverlay->>Editor: render()
```

### 2.5 ビジネスルール
1. `editor.lineNumbers`がoffの場合は表示しない
2. 相対行番号モードでは現在行は絶対行番号を表示
3. 間隔モードでは10行ごとと現在行のみ表示
4. 最終行が空の場合は`renderFinalNewline`設定に従う

## 3. 画面設計

### 3.1 画面レイアウト
行番号はエディタの左側のマージン領域に表示される。

### 3.2 画面要素
| 要素 | 説明 | 操作 |
|------|------|------|
| 行番号 | 各行の番号表示 | クリックで行選択 |
| アクティブ行番号 | 現在行のハイライト | 自動更新 |
| 薄い行番号 | 最終行（dimmed） | 設定依存 |

## 4. データ設計

### 4.1 入力データ構造
```typescript
interface LineNumbersConfig {
    renderType: RenderLineNumbersType;
    renderFn: ((lineNumber: number) => string) | null;
}

enum RenderLineNumbersType {
    Off = 0,
    On = 1,
    Relative = 2,
    Interval = 3
}
```

### 4.2 出力データ構造
```typescript
interface LineNumbersRenderResult {
    lineNumbersLeft: number;    // 左位置（px）
    lineNumbersWidth: number;   // 幅（px）
    renderResult: string[];     // 行ごとのHTML
}
```

## 5. コードリーディングガイド

### 5.1 推奨読解順序

1. **設定の理解**
   - `src/vs/editor/common/config/editorOptions.ts` - RenderLineNumbersType等の設定定義

2. **ビュー層**
   - `src/vs/editor/browser/viewParts/lineNumbers/lineNumbers.ts` - 行番号表示の実装（**21-228行目**）
     - `LineNumbersOverlay`クラスが行番号表示を制御
     - **49-59行目**: `_readConfig()`で設定読み込み
     - **73-86行目**: `onCursorStateChanged()`でカーソル変更処理
     - **111-145行目**: `_getLineRenderLineNumber()`で行番号計算
     - **147-206行目**: `prepareRender()`でレンダリング準備
     - **208-217行目**: `render()`で実際の描画

### 5.2 プログラム呼び出し階層図

```
lineNumbers.ts
  └── LineNumbersOverlay
        ├── _readConfig()
        │     └── options.get(EditorOption.lineNumbers)
        ├── イベントハンドラ
        │     ├── onConfigurationChanged()
        │     ├── onCursorStateChanged()
        │     ├── onScrollChanged()
        │     └── onDecorationsChanged()
        ├── _getLineRenderLineNumber()
        │     ├── 絶対行番号
        │     ├── 相対行番号（Relative）
        │     └── 間隔行番号（Interval）
        └── prepareRender() → render()
```

### 5.3 データフロー図

```
[EditorOption.lineNumbers設定]
        ↓
[_readConfig() - 設定読み込み]
        ↓
[イベント（スクロール、カーソル移動）]
        ↓
[prepareRender() - 行番号計算]
        ↓
[_getLineRenderLineNumber() - 各行の番号取得]
        ↓
[render() - HTML出力]
```

### 5.4 読解のコツ
- `LineNumbersOverlay`は`DynamicViewOverlay`を継承
- `CLASS_NAME = 'line-numbers'`でCSSクラスを定義
- 相対行番号では`_lastCursorModelPosition`との差分を計算
- `renderCustomLineNumbers`でカスタム関数を呼び出し可能
- Linuxでは`lh-even`/`lh-odd`クラスで行高さ調整

### 5.5 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/editor/browser/viewParts/lineNumbers/lineNumbers.ts` | ビュー | 行番号表示 |
| `src/vs/editor/browser/viewParts/lineNumbers/lineNumbers.css` | スタイル | 行番号CSS |
| `src/vs/editor/common/config/editorOptions.ts` | 設定 | RenderLineNumbersType等 |
| `src/vs/editor/common/core/editorColorRegistry.ts` | 色定義 | 行番号の色 |
| `src/vs/editor/browser/view/dynamicViewOverlay.ts` | 基底クラス | オーバーレイ基底 |

## 6. 非機能要件

### 6.1 パフォーマンス
- 可視範囲のみをレンダリング
- 文字列結合でHTMLを生成（DOM操作を最小化）
- 変更があった行のみ再レンダリング

### 6.2 エラーハンドリング
- 無効な行番号は空文字列を返す
- 範囲外の行は空のレンダリング結果

## 7. 設計上の考慮事項

### 7.1 拡張性
- `renderCustomLineNumbers`でカスタムレンダリング関数を指定可能
- デコレーションで行番号にカスタムスタイルを適用可能

### 7.2 依存関係
- `ViewContext`からビューモデルにアクセス
- `EditorOption.lineNumbers`で設定を取得
- `registerThemingParticipant`でテーマ対応
