# 機能設計書: ソース管理

## 1. 機能概要

### 1.1 機能名
ソース管理（Source Control / SCM）

### 1.2 機能ID
No.23

### 1.3 概要説明
ソース管理機能は、VSCodeのサイドバーに表示されるバージョン管理システム統合機能である。Git、SVN、Mercurialなどの各種バージョン管理システムをプラグインベースで統合し、変更のステージング、コミット、プッシュ/プル、ブランチ操作などの基本的なソース管理操作を提供する。拡張機能によって複数のSCMプロバイダーを同時にサポートすることが可能である。

### 1.4 関連画面
- ソース管理ビュー（サイドバー）
- 変更ファイル一覧
- コミットメッセージ入力エリア
- リポジトリセクション

## 2. 機能要件

### 2.1 ビジネス要件
- ユーザーがバージョン管理操作を直感的に行えること
- 複数のSCMプロバイダーを統合的に管理できること
- 変更の差分表示とインライン編集が可能であること
- コミット履歴の参照と操作が可能であること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| SCM-001 | 変更検出 | ファイルの変更・追加・削除の検出 |
| SCM-002 | ステージング | 変更のステージング/アンステージング |
| SCM-003 | コミット | コミットメッセージ入力とコミット実行 |
| SCM-004 | プッシュ/プル | リモートとの同期操作 |
| SCM-005 | ブランチ操作 | ブランチの作成・切り替え・削除 |
| SCM-006 | 差分表示 | 変更前後の差分表示 |
| SCM-007 | マルチリポジトリ | 複数リポジトリの同時管理 |
| SCM-008 | プロバイダー拡張 | SCMプロバイダーの拡張機能対応 |
| SCM-009 | インライン変更 | エディタ内での変更操作 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                    SCM View Pane                        │
│  ┌──────────────────────────────────────────────────┐  │
│  │              SCMViewPane                          │  │
│  │  ┌────────────────────────────────────────────┐  │  │
│  │  │  Repository Section                        │  │  │
│  │  │  ┌──────────────────────────────────────┐  │  │  │
│  │  │  │  SCMInputWidget (Commit Message)     │  │  │  │
│  │  │  └──────────────────────────────────────┘  │  │  │
│  │  │  ┌──────────────────────────────────────┐  │  │  │
│  │  │  │  ResourceGroup List                  │  │  │  │
│  │  │  │  - Staged Changes                    │  │  │  │
│  │  │  │  - Changes                           │  │  │  │
│  │  │  │  - Merge Changes                     │  │  │  │
│  │  │  └──────────────────────────────────────┘  │  │  │
│  │  └────────────────────────────────────────────┘  │  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                  SCMService                             │
│  ┌────────────────────────────────────────────────────┐│
│  │  - repositories: ISCMRepository[]                  ││
│  │  - registerSCMProvider()                           ││
│  │  - onDidAddRepository: Event                       ││
│  │  - onDidRemoveRepository: Event                    ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│               SCM Provider (Extension)                  │
│  ┌────────────────────────────────────────────────────┐│
│  │  Git Extension / SVN Extension / etc.              ││
│  │  - ISCMProvider implementation                     ││
│  │  - ISCMResourceGroup management                    ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 クラス図

```
┌─────────────────────────┐
│     ISCMService         │
│     <<interface>>       │
├─────────────────────────┤
│ + repositories          │
│ + registerSCMProvider() │
│ + onDidAddRepository    │
│ + onDidRemoveRepository │
└───────────┬─────────────┘
            │
            ▼
┌─────────────────────────┐
│      SCMService         │
├─────────────────────────┤
│ - _repositories: Map    │
│ - _providers: Map       │
├─────────────────────────┤
│ + registerSCMProvider() │
│ + getRepositories()     │
└─────────────────────────┘

┌─────────────────────────┐      ┌─────────────────────────┐
│    ISCMRepository       │      │   ISCMResourceGroup     │
│    <<interface>>        │      │   <<interface>>         │
├─────────────────────────┤      ├─────────────────────────┤
│ + provider              │      │ + id                    │
│ + input                 │──────│ + label                 │
│ + resourceGroups        │      │ + resources             │
│ + setSelected()         │      │ + hideWhenEmpty         │
└─────────────────────────┘      └─────────────────────────┘

┌─────────────────────────┐
│    ISCMResource         │
├─────────────────────────┤
│ + resourceGroup         │
│ + sourceUri             │
│ + decorations           │
│ + contextValue          │
└─────────────────────────┘
```

## 4. 詳細設計

### 4.1 データ構造

#### ISCMRepository
```typescript
interface ISCMRepository {
    readonly provider: ISCMProvider;
    readonly input: ISCMInput;
    readonly resources: ISCMResourceGroup[];

    // 選択状態
    setSelected(selected: boolean): void;

    // 変更通知
    readonly onDidChangeResources: Event<void>;
    readonly onDidChangeSelection: Event<boolean>;
}
```

#### ISCMResourceGroup
```typescript
interface ISCMResourceGroup {
    readonly id: string;
    readonly label: string;
    readonly resources: ISCMResource[];

    // オプション
    readonly hideWhenEmpty?: boolean;

    // アクション
    readonly acceptInputCommand?: Command;

    // 変更通知
    readonly onDidChangeResources: Event<void>;
}
```

#### ISCMInput
```typescript
interface ISCMInput {
    readonly repository: ISCMRepository;

    // コミットメッセージ
    value: string;
    placeholder: string;

    // バリデーション
    validateInput: (value: string) => Promise<IInputValidation | undefined>;

    // 入力制限
    enabled: boolean;
    visible: boolean;

    // 変更通知
    readonly onDidChange: Event<string>;
    readonly onDidChangeValidation: Event<IInputValidation | undefined>;
}
```

### 4.2 処理フロー

#### SCMプロバイダー登録フロー
```
1. 拡張機能の有効化
   │
   ├─2. SCMService.registerSCMProvider()
   │    │
   │    ├─ プロバイダーの登録
   │    │
   │    └─ リポジトリの初期化
   │
   └─3. SCMViewPane.onDidAddRepository
        │
        └─ ビューの更新
```

#### コミット実行フロー
```
1. ユーザーがコミットメッセージ入力
   │
   ├─2. SCMInput.validateInput()
   │    │
   │    └─ バリデーション実行
   │
   ├─3. AcceptInputCommand実行
   │    │
   │    ├─ SCMProvider経由でGit/SVN等にコミット
   │    │
   │    └─ リソースグループの更新
   │
   └─4. SCMViewPane.refresh()
        │
        └─ ビューの更新
```

### 4.3 主要メソッド

#### SCMService.registerSCMProvider()
```typescript
registerSCMProvider(provider: ISCMProvider): ISCMRepository {
    // 1. プロバイダーの登録
    const repository = new SCMRepository(provider);
    this._repositories.set(provider, repository);

    // 2. イベント発火
    this._onDidAddRepository.fire(repository);

    // 3. 破棄処理の登録
    provider.onDidDispose(() => {
        this._repositories.delete(provider);
        this._onDidRemoveRepository.fire(repository);
    });

    return repository;
}
```

#### SCMInputWidget（コミットメッセージ入力）
```typescript
class SCMInputWidget {
    private readonly inputEditor: CodeEditorWidget;

    constructor(
        container: HTMLElement,
        @IInstantiationService private instantiationService: IInstantiationService,
        @IModelService private modelService: IModelService
    ) {
        // エディタの作成
        this.inputEditor = this.instantiationService.createInstance(
            CodeEditorWidget,
            container,
            this.getEditorOptions(),
            this.getEditorOverrides()
        );
    }

    setInput(input: ISCMInput): void {
        // モデルの設定
        const model = this.modelService.createModel(
            input.value,
            null,
            URI.parse('scm:input')
        );
        this.inputEditor.setModel(model);
    }
}
```

## 5. インターフェース設計

### 5.1 サービスインターフェース

```typescript
interface ISCMService {
    readonly _serviceBrand: undefined;

    // プロバイダー管理
    readonly repositories: Iterable<ISCMRepository>;
    registerSCMProvider(provider: ISCMProvider): ISCMRepository;

    // イベント
    readonly onDidAddRepository: Event<ISCMRepository>;
    readonly onDidRemoveRepository: Event<ISCMRepository>;
}
```

### 5.2 ビューサービスインターフェース

```typescript
interface ISCMViewService {
    readonly _serviceBrand: undefined;

    // 表示リポジトリ
    readonly visibleRepositories: Iterable<ISCMRepository>;
    focusedRepository: ISCMRepository | undefined;

    // 表示管理
    isVisible(repository: ISCMRepository): boolean;

    // イベント
    readonly onDidChangeVisibleRepositories: Event<ISCMViewVisibleRepositoryChangeEvent>;
    readonly onDidFocusRepository: Event<ISCMRepository | undefined>;
}
```

## 6. ツリー要素タイプ

### 6.1 TreeElement定義

```typescript
type TreeElement =
    | ISCMRepository      // リポジトリ
    | ISCMInput          // コミット入力
    | ISCMActionButton   // アクションボタン
    | ISCMResourceGroup  // リソースグループ
    | ISCMResource       // 個別リソース
    | IResourceNode<ISCMResource, ISCMResourceGroup>;  // ツリーノード
```

### 6.2 レンダラーテンプレート

| レンダラー | テンプレートID | 対象要素 |
|-----------|---------------|---------|
| RepositoryRenderer | repository | ISCMRepository |
| InputRenderer | input | ISCMInput |
| ActionButtonRenderer | actionButton | ISCMActionButton |
| ResourceGroupRenderer | resourceGroup | ISCMResourceGroup |
| ResourceRenderer | resource | ISCMResource |

## 7. エラーハンドリング

### 7.1 エラー種別

| エラーコード | エラー名 | 説明 | 対処方法 |
|-------------|---------|------|---------|
| SCM-E001 | NoRepository | リポジトリが見つからない | ガイダンスメッセージ表示 |
| SCM-E002 | CommitFailed | コミット失敗 | エラーメッセージと原因表示 |
| SCM-E003 | MergeConflict | マージコンフリクト | コンフリクト解消UI表示 |
| SCM-E004 | AuthenticationFailed | 認証エラー | 認証ダイアログ表示 |
| SCM-E005 | NetworkError | ネットワークエラー | リトライオプション提供 |

### 7.2 入力バリデーション

```typescript
interface IInputValidation {
    message: string;
    type: InputValidationType;  // Warning | Information | Error
}
```

## 8. パフォーマンス考慮事項

### 8.1 最適化戦略

- **差分計算の非同期化**: Git diffの非同期実行
- **仮想スクロール**: 大量の変更ファイルでも高速表示
- **デバウンス**: ファイル変更検出の集約
- **遅延読み込み**: 展開時のリソース読み込み

### 8.2 パフォーマンス目標

| 指標 | 目標値 |
|------|--------|
| 変更検出レスポンス | < 200ms |
| ステージング操作 | < 100ms |
| コミット実行 | < 1s |
| 差分表示 | < 500ms |

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **データモデル**: `src/vs/workbench/contrib/scm/common/scm.ts`
   - ISCMRepository、ISCMResourceGroup、ISCMResourceインターフェース

2. **サービス層**: `src/vs/workbench/contrib/scm/common/scmService.ts`
   - SCMServiceクラスの実装

3. **ビュー層**: `src/vs/workbench/contrib/scm/browser/scmViewPane.ts`
   - **1-120行目**: インポートとタイプ定義
   - **117-165行目**: processResourceFilterData関数
   - **167-233行目**: ActionButtonRenderer
   - **236-287行目**: SCMTreeDragAndDrop

4. **入力ウィジェット**: `src/vs/workbench/contrib/scm/browser/scmViewPane.ts`
   - **289-300行目**: InputTemplate、InputRenderer

### 9.2 プログラム呼び出し階層図

```
SCMViewlet
└── SCMViewPane
    ├── SCMInputWidget (CodeEditorWidget)
    │   ├── InputModel
    │   ├── SuggestController
    │   └── ValidationDecorations
    ├── SCMTree (WorkbenchCompressibleAsyncDataTree)
    │   ├── ActionButtonRenderer
    │   ├── RepositoryRenderer
    │   ├── InputRenderer
    │   ├── ResourceGroupRenderer
    │   └── ResourceRenderer
    └── SCMViewService
        └── SCMService
            └── ISCMProvider (Extensions)
                ├── Git Extension
                ├── SVN Extension
                └── Other SCM Extensions
```

### 9.3 データフロー図

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│  SCM Provider   │────▶│   SCMService    │────▶│  SCMViewService │
│  (Git/SVN/etc)  │     │                 │     │                 │
└─────────────────┘     └─────────────────┘     └────────┬────────┘
                                                         │
                                                         ▼
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│   User Input    │────▶│  SCMInputWidget │────▶│   SCMViewPane   │
│  (Message/Ops)  │     │                 │     │                 │
└─────────────────┘     └─────────────────┘     └────────┬────────┘
                                                         │
                                                         ▼
                        ┌─────────────────┐     ┌─────────────────┐
                        │    SCMTree      │◀────│ ISCMRepository  │
                        │   (Display)     │     │ ISCMResource[]  │
                        └─────────────────┘     └─────────────────┘
```

### 9.4 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/scm/browser/scmViewPane.ts` | ビュー | SCMビューペイン実装 |
| `src/vs/workbench/contrib/scm/common/scm.ts` | 共通 | インターフェース定義 |
| `src/vs/workbench/contrib/scm/common/scmService.ts` | サービス | SCMサービス実装 |
| `src/vs/workbench/contrib/scm/browser/scmViewService.ts` | サービス | SCMビューサービス |
| `src/vs/workbench/contrib/scm/browser/scmRepositoryRenderer.ts` | レンダラー | リポジトリレンダラー |
| `src/vs/workbench/contrib/scm/browser/util.ts` | ユーティリティ | ヘルパー関数 |

## 10. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
