# 機能設計書: デバッグ

## 1. 機能概要

### 1.1 機能名
デバッグ（Debug）

### 1.2 機能ID
No.24

### 1.3 概要説明
デバッグ機能は、VSCodeの統合デバッグ環境を提供する機能である。Debug Adapter Protocol（DAP）を通じて、様々なプログラミング言語とランタイムのデバッグをサポートする。ブレークポイントの設定、変数の監視、コールスタックの表示、ステップ実行などの標準的なデバッグ操作を提供し、拡張機能によって新しいデバッガーを追加することが可能である。

### 1.4 関連画面
- デバッグビュー（サイドバー）
- 変数パネル
- ウォッチパネル
- コールスタックパネル
- ブレークポイントパネル
- デバッグコンソール

## 2. 機能要件

### 2.1 ビジネス要件
- 複数の言語・ランタイムのデバッグをサポートすること
- 直感的なブレークポイント管理が可能であること
- 変数の値確認と変更が可能であること
- ステップ実行によるプログラム制御が可能であること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| DEBUG-001 | ブレークポイント | 行ブレークポイントの設定・解除 |
| DEBUG-002 | 条件付きブレークポイント | 条件式付きブレークポイント |
| DEBUG-003 | ログポイント | ログ出力ブレークポイント |
| DEBUG-004 | ステップ実行 | Step Over/Into/Out |
| DEBUG-005 | 変数監視 | 変数の値確認と変更 |
| DEBUG-006 | ウォッチ式 | 式の評価と監視 |
| DEBUG-007 | コールスタック | 呼び出し履歴の表示 |
| DEBUG-008 | マルチセッション | 複数デバッグセッションの同時管理 |
| DEBUG-009 | launch.json | デバッグ構成ファイルの管理 |
| DEBUG-010 | デバッグコンソール | REPL機能 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                    Debug Viewlet                        │
│  ┌──────────────────────────────────────────────────┐  │
│  │  Variables View                                   │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Watch View                                       │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Call Stack View                                  │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Breakpoints View                                 │  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                   DebugService                          │
│  ┌────────────────────────────────────────────────────┐│
│  │  - model: DebugModel                               ││
│  │  - viewModel: ViewModel                            ││
│  │  - configurationManager: ConfigurationManager      ││
│  │  - adapterManager: AdapterManager                  ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                   DebugSession                          │
│  ┌────────────────────────────────────────────────────┐│
│  │  - Debug Adapter Protocol (DAP)                    ││
│  │  - Threads, StackFrames, Variables                 ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│               Debug Adapter (Extension)                 │
│  ┌────────────────────────────────────────────────────┐│
│  │  Node.js Debugger / Python Debugger / etc.         ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 クラス図

```
┌─────────────────────────┐
│    IDebugService        │
│    <<interface>>        │
├─────────────────────────┤
│ + state: State          │
│ + getModel()            │
│ + getViewModel()        │
│ + startDebugging()      │
│ + stopSession()         │
└───────────┬─────────────┘
            │
            ▼
┌─────────────────────────┐
│     DebugService        │
├─────────────────────────┤
│ - model: DebugModel     │
│ - viewModel: ViewModel  │
│ - taskRunner: TaskRunner│
├─────────────────────────┤
│ + startDebugging()      │
│ + restartSession()      │
│ + stopSession()         │
│ - launchOrAttachSession()│
└─────────────────────────┘

┌─────────────────────────┐      ┌─────────────────────────┐
│      DebugModel         │      │      DebugSession       │
├─────────────────────────┤      ├─────────────────────────┤
│ - sessions: Map         │      │ - id: string            │
│ - breakpoints: Map      │──────│ - state: State          │
│ - functionBreakpoints   │      │ - threads: IThread[]    │
│ - dataBreakpoints       │      │ + initialize()          │
├─────────────────────────┤      │ + launchOrAttach()      │
│ + addBreakpoints()      │      │ + disconnect()          │
│ + removeBreakpoints()   │      └─────────────────────────┘
│ + getSession()          │
└─────────────────────────┘
```

## 4. 詳細設計

### 4.1 データ構造

#### IDebugSession
```typescript
interface IDebugSession {
    readonly id: string;
    readonly configuration: IConfig;
    readonly parentSession: IDebugSession | undefined;
    readonly state: State;

    // スレッド/スタック
    readonly threads: readonly IThread[];
    getThread(threadId: number): IThread | undefined;

    // 操作
    initialize(dbgr: IDebugger): Promise<void>;
    launchOrAttach(config: IConfig): Promise<void>;
    restart(): Promise<void>;
    disconnect(restart?: boolean): Promise<void>;

    // ブレークポイント
    sendBreakpoints(uri: URI, bpts: IBreakpoint[]): Promise<void>;

    // 評価
    evaluate(expression: string, frameId?: number): Promise<string>;
}
```

#### State（デバッグ状態）
```typescript
enum State {
    Inactive = 0,
    Initializing = 1,
    Stopped = 2,
    Running = 3
}
```

#### IBreakpoint
```typescript
interface IBreakpoint {
    readonly id: string;
    readonly uri: URI;
    readonly lineNumber: number;
    readonly column?: number;
    readonly enabled: boolean;
    readonly condition?: string;
    readonly hitCondition?: string;
    readonly logMessage?: string;
    readonly verified: boolean;
}
```

### 4.2 処理フロー

#### デバッグセッション開始フロー
```
1. ユーザーがデバッグ開始（F5等）
   │
   ├─2. DebugService.startDebugging()
   │    │
   │    ├─ ConfigurationManager.resolveConfigurationByType()
   │    │
   │    └─ preLaunchTask実行（必要に応じて）
   │
   ├─3. DebugSession.initialize()
   │    │
   │    └─ Debug Adapterとの接続確立
   │
   └─4. DebugSession.launchOrAttach()
        │
        ├─ ブレークポイント送信
        │
        └─ プログラム実行開始
```

#### ブレークポイント設定フロー
```
1. エディタ上でブレークポイント設定
   │
   ├─2. DebugModel.addBreakpoints()
   │    │
   │    └─ ブレークポイントの保存
   │
   └─3. DebugSession.sendBreakpoints()（セッション中の場合）
        │
        └─ Debug Adapterへ送信
             │
             └─ verified状態の更新
```

### 4.3 主要メソッド

#### DebugService.startDebugging()
```typescript
async startDebugging(
    launch: ILaunch | undefined,
    configOrName?: IConfig | string,
    options?: IDebugSessionOptions
): Promise<boolean> {
    // 1. 初期化状態の開始
    this.startInitializingState(options);

    // 2. 設定の解決
    const config = await this.configurationManager.resolveConfigurationByType(
        launch, configOrName
    );

    // 3. preLaunchTaskの実行
    const taskResult = await this.taskRunner.runTaskAndCheckErrors(
        launch?.workspace,
        config.preLaunchTask
    );

    if (taskResult !== TaskRunResult.Success) {
        return false;
    }

    // 4. セッションの作成と開始
    const session = await this.createSession(config, options);
    await this.launchOrAttachToSession(session);

    return true;
}
```

#### DebugSession.initialize()
```typescript
async initialize(dbgr: IDebugger): Promise<void> {
    // 1. Debug Adapterの起動
    const adapter = await dbgr.createDebugAdapter(this);

    // 2. 初期化リクエスト
    const response = await adapter.initialize({
        clientID: 'vscode',
        clientName: 'Visual Studio Code',
        adapterID: dbgr.type,
        pathFormat: 'path',
        linesStartAt1: true,
        columnsStartAt1: true,
        supportsVariableType: true,
        // ... その他のcapabilities
    });

    // 3. capabilitiesの保存
    this.capabilities = response.body || {};
}
```

## 5. インターフェース設計

### 5.1 サービスインターフェース

```typescript
interface IDebugService {
    readonly _serviceBrand: undefined;

    // 状態
    readonly state: State;
    readonly initializingOptions: IDebugSessionOptions | undefined;

    // モデル/ビューモデル
    getModel(): IDebugModel;
    getViewModel(): IViewModel;
    getConfigurationManager(): IConfigurationManager;
    getAdapterManager(): IAdapterManager;

    // セッション制御
    startDebugging(
        launch: ILaunch | undefined,
        configOrName?: IConfig | string,
        options?: IDebugSessionOptions
    ): Promise<boolean>;

    restartSession(session: IDebugSession): Promise<void>;
    stopSession(session: IDebugSession): Promise<void>;

    // イベント
    readonly onDidChangeState: Event<State>;
    readonly onDidNewSession: Event<IDebugSession>;
    readonly onDidEndSession: Event<{ session: IDebugSession; restart: boolean }>;
}
```

### 5.2 コンテキストキー

```typescript
// デバッグ関連コンテキストキー
const CONTEXT_DEBUG_TYPE: RawContextKey<string>;
const CONTEXT_DEBUG_STATE: RawContextKey<string>;
const CONTEXT_IN_DEBUG_MODE: RawContextKey<boolean>;
const CONTEXT_DEBUG_UX: RawContextKey<string>;
const CONTEXT_HAS_DEBUGGED: RawContextKey<boolean>;
const CONTEXT_BREAKPOINTS_EXIST: RawContextKey<boolean>;
const CONTEXT_DISASSEMBLY_VIEW_FOCUS: RawContextKey<boolean>;
```

## 6. Debug Adapter Protocol (DAP)

### 6.1 主要リクエスト

| リクエスト | 説明 |
|-----------|------|
| initialize | 初期化 |
| launch | プログラム起動 |
| attach | プロセスアタッチ |
| setBreakpoints | ブレークポイント設定 |
| configurationDone | 設定完了 |
| continue | 実行継続 |
| next | ステップオーバー |
| stepIn | ステップイン |
| stepOut | ステップアウト |
| pause | 一時停止 |
| disconnect | 切断 |

### 6.2 主要イベント

| イベント | 説明 |
|---------|------|
| initialized | 初期化完了 |
| stopped | 停止（ブレークポイント等） |
| continued | 実行継続 |
| exited | プログラム終了 |
| terminated | デバッグ終了 |
| thread | スレッド状態変更 |
| output | 出力 |

## 7. エラーハンドリング

### 7.1 エラー種別

| エラーコード | エラー名 | 説明 | 対処方法 |
|-------------|---------|------|---------|
| DEBUG-E001 | AdapterNotFound | デバッガーが見つからない | 拡張機能インストール案内 |
| DEBUG-E002 | LaunchFailed | 起動失敗 | エラーメッセージ表示 |
| DEBUG-E003 | AttachFailed | アタッチ失敗 | 原因と対処法を表示 |
| DEBUG-E004 | TaskFailed | preLaunchTask失敗 | タスク出力を表示 |
| DEBUG-E005 | ConfigurationError | 設定エラー | launch.jsonを開く |

### 7.2 エラーハンドリングフロー

```
1. デバッグ操作実行
   │
   ├─ 成功 → 状態更新とUI反映
   │
   └─ 失敗
        │
        ├─ AdapterNotFound → 拡張機能推奨
        │
        ├─ LaunchFailed → 詳細ダイアログ
        │
        ├─ TaskFailed → 問題パネルを開く
        │
        └─ その他 → エラー通知
```

## 8. パフォーマンス考慮事項

### 8.1 最適化戦略

- **非同期DAP通信**: すべてのDAP通信を非同期で実行
- **変数の遅延読み込み**: 展開時に子要素を取得
- **デバウンス**: 連続したブレークポイント変更の集約
- **キャッシュ**: スタックフレームと変数のキャッシュ

### 8.2 パフォーマンス目標

| 指標 | 目標値 |
|------|--------|
| デバッグ開始 | < 2s |
| ステップ実行 | < 100ms |
| 変数表示 | < 200ms |
| ブレークポイント検証 | < 100ms |

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **サービス層**: `src/vs/workbench/contrib/debug/browser/debugService.ts`
   - **63-92行目**: DebugServiceクラスのプロパティ定義
   - **95-216行目**: コンストラクタと初期化処理
   - **218-236行目**: initContextKeys（コンテキストキーの初期化）
   - **262-300行目**: 状態管理メソッド

2. **デバッグモデル**: `src/vs/workbench/contrib/debug/common/debugModel.ts`
   - Breakpoint、FunctionBreakpoint、DataBreakpointクラス

3. **デバッグセッション**: `src/vs/workbench/contrib/debug/browser/debugSession.ts`
   - DebugSessionクラスの実装

4. **設定管理**: `src/vs/workbench/contrib/debug/browser/debugConfigurationManager.ts`
   - ConfigurationManagerの実装

### 9.2 プログラム呼び出し階層図

```
DebugViewlet
├── VariablesView
│   └── VariablesRenderer
├── WatchView
│   └── WatchExpressionsRenderer
├── CallStackView
│   └── CallStackRenderer
└── BreakpointsView
    └── BreakpointsRenderer
         │
         ▼
    DebugService
    ├── DebugModel
    │   ├── Breakpoints
    │   ├── FunctionBreakpoints
    │   ├── DataBreakpoints
    │   └── Sessions
    ├── ViewModel
    │   ├── FocusedSession
    │   ├── FocusedThread
    │   └── FocusedStackFrame
    ├── ConfigurationManager
    │   └── Launch configurations
    └── AdapterManager
        └── Debug Adapters
```

### 9.3 データフロー図

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│  User Action    │────▶│  DebugService   │────▶│  DebugSession   │
│  (F5/Breakpoint)│     │                 │     │                 │
└─────────────────┘     └─────────────────┘     └────────┬────────┘
                                                         │
                                                         ▼
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│   Debug Views   │◀────│   DebugModel    │◀────│  Debug Adapter  │
│   (Variables,   │     │  (Breakpoints,  │     │  (DAP Protocol) │
│    CallStack)   │     │   Sessions)     │     │                 │
└─────────────────┘     └─────────────────┘     └─────────────────┘
```

### 9.4 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/debug/browser/debugService.ts` | サービス | デバッグサービス実装 |
| `src/vs/workbench/contrib/debug/common/debugModel.ts` | モデル | デバッグモデル |
| `src/vs/workbench/contrib/debug/browser/debugSession.ts` | セッション | デバッグセッション |
| `src/vs/workbench/contrib/debug/common/debug.ts` | 共通 | インターフェース定義 |
| `src/vs/workbench/contrib/debug/browser/debugConfigurationManager.ts` | 設定 | 設定管理 |
| `src/vs/workbench/contrib/debug/browser/debugAdapterManager.ts` | アダプタ | アダプタ管理 |
| `src/vs/workbench/contrib/debug/common/debugViewModel.ts` | ビューモデル | ビューモデル |

## 10. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
