# 機能設計書: 拡張機能

## 1. 機能概要

### 1.1 機能名
拡張機能（Extensions）

### 1.2 機能ID
No.25

### 1.3 概要説明
拡張機能機能は、VSCodeの機能拡張システムを管理するビューである。マーケットプレイスからの拡張機能の検索、インストール、アンインストール、有効化/無効化などの操作を提供する。インストール済み拡張機能の管理、更新確認、推奨拡張機能の表示なども行う。

### 1.4 関連画面
- 拡張機能ビュー（サイドバー）
- 拡張機能詳細パネル
- 拡張機能検索

## 2. 機能要件

### 2.1 ビジネス要件
- マーケットプレイスから拡張機能を検索・インストールできること
- インストール済み拡張機能を一元管理できること
- 拡張機能の有効/無効を切り替えられること
- 拡張機能の更新を管理できること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| EXT-001 | マーケットプレイス検索 | キーワードによる拡張機能検索 |
| EXT-002 | インストール | 拡張機能のインストール |
| EXT-003 | アンインストール | 拡張機能の削除 |
| EXT-004 | 有効化/無効化 | 拡張機能の有効/無効切り替え |
| EXT-005 | 更新管理 | 拡張機能の更新確認と適用 |
| EXT-006 | 推奨表示 | ワークスペースに応じた推奨表示 |
| EXT-007 | カテゴリ分類 | カテゴリによる絞り込み |
| EXT-008 | ビルトイン表示 | 組み込み拡張機能の表示 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                 Extensions Viewlet                      │
│  ┌──────────────────────────────────────────────────┐  │
│  │              Search Input                         │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Installed Extensions View                        │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Popular Extensions View                          │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Recommended Extensions View                      │  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│            IExtensionsWorkbenchService                  │
│  ┌────────────────────────────────────────────────────┐│
│  │  - queryGallery()                                  ││
│  │  - install()                                       ││
│  │  - uninstall()                                     ││
│  │  - setEnablement()                                 ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│            IExtensionManagementService                  │
│  ┌────────────────────────────────────────────────────┐│
│  │  Platform Extension Management                     ││
│  │  (Local / Remote / Web)                            ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 ビュー構成

```typescript
// ビュー定義
const viewDescriptors: IViewDescriptor[] = [
    // インストール済み拡張機能ビュー
    ServerInstalledExtensionsView,
    // 人気拡張機能ビュー
    DefaultPopularExtensionsView,
    // 推奨拡張機能ビュー
    DefaultRecommendedExtensionsView,
    // 有効な拡張機能ビュー
    EnabledExtensionsView,
    // 無効な拡張機能ビュー
    DisabledExtensionsView,
    // 更新可能な拡張機能ビュー
    OutdatedExtensionsView,
    // 組み込み拡張機能ビュー
    BuiltInExtensionsView,
];
```

## 4. 詳細設計

### 4.1 データ構造

#### IExtension
```typescript
interface IExtension {
    readonly type: ExtensionType;  // User | System
    readonly identifier: IExtensionIdentifier;
    readonly manifest: IExtensionManifest;
    readonly location: URI;
    readonly readmeUrl?: URI;
    readonly changelogUrl?: URI;

    // 状態
    readonly state: ExtensionState;
    readonly local?: ILocalExtension;
    readonly gallery?: IGalleryExtension;

    // 評価
    readonly rating?: number;
    readonly ratingCount?: number;
    readonly downloadCount?: number;
}
```

#### ExtensionState
```typescript
enum ExtensionState {
    Installing,
    Installed,
    Uninstalling,
    Uninstalled
}
```

### 4.2 主要メソッド

#### ExtensionsViewletViewsContribution.registerViews()
```typescript
private registerViews(): void {
    const viewDescriptors: IViewDescriptor[] = [];

    // デフォルトビュー
    viewDescriptors.push(...this.createDefaultExtensionsViewDescriptors());

    // 検索ビュー
    viewDescriptors.push(...this.createSearchExtensionsViewDescriptors());

    // 推奨ビュー
    viewDescriptors.push(...this.createRecommendedExtensionsViewDescriptors());

    // ビルトインビュー
    viewDescriptors.push(...this.createBuiltinExtensionsViewDescriptors());

    // ビューの登録
    const viewRegistry = Registry.as<IViewsRegistry>(Extensions.ViewsRegistry);
    viewRegistry.registerViews(viewDescriptors, this.container);
}
```

## 5. インターフェース設計

### 5.1 サービスインターフェース

```typescript
interface IExtensionsWorkbenchService {
    readonly _serviceBrand: undefined;

    // クエリ
    queryLocal(server?: IExtensionManagementServer): Promise<IExtension[]>;
    queryGallery(token: CancellationToken): Promise<IPagedModel<IExtension>>;

    // インストール/アンインストール
    install(extension: IExtension): Promise<IExtension>;
    uninstall(extension: IExtension): Promise<void>;

    // 有効化/無効化
    setEnablement(extension: IExtension, enablement: EnablementState): Promise<void>;

    // 更新
    checkForUpdates(): Promise<void>;

    // イベント
    readonly onChange: Event<IExtension | undefined>;
}
```

## 6. コンテキストキー

```typescript
// 拡張機能関連コンテキストキー
const CONTEXT_HAS_GALLERY: RawContextKey<boolean>;
const SearchMarketplaceExtensionsContext: RawContextKey<boolean>;
const SearchHasTextContext: RawContextKey<boolean>;
const InstalledExtensionsContext: RawContextKey<boolean>;
const HasInstalledExtensionsContext: RawContextKey<boolean>;
const BuiltInExtensionsContext: RawContextKey<boolean>;
const RecommendedExtensionsContext: RawContextKey<boolean>;
```

## 7. エラーハンドリング

### 7.1 エラー種別

| エラーコード | エラー名 | 説明 | 対処方法 |
|-------------|---------|------|---------|
| EXT-E001 | InstallFailed | インストール失敗 | エラー詳細を表示 |
| EXT-E002 | NetworkError | ネットワークエラー | リトライオプション |
| EXT-E003 | IncompatibleVersion | 非互換バージョン | 対応バージョン案内 |
| EXT-E004 | Deprecated | 非推奨拡張機能 | 代替拡張機能を案内 |

## 8. コードリーディングガイド

### 8.1 推奨読解順序

1. **ビューレット**: `src/vs/workbench/contrib/extensions/browser/extensionsViewlet.ts`
   - **74-92行目**: コンテキストキー定義
   - **99-166行目**: ExtensionsViewletViewsContributionクラス
   - **168-300行目**: ビュー登録処理

2. **拡張機能ビュー**: `src/vs/workbench/contrib/extensions/browser/extensionsViews.ts`
   - 各種ExtensionsListViewの実装

3. **拡張機能アクション**: `src/vs/workbench/contrib/extensions/browser/extensionsActions.ts`
   - インストール/アンインストールアクション

### 8.2 プログラム呼び出し階層図

```
ExtensionsViewlet
├── SuggestEnabledInput (検索入力)
├── ExtensionsListView
│   ├── ServerInstalledExtensionsView
│   ├── DefaultPopularExtensionsView
│   ├── DefaultRecommendedExtensionsView
│   ├── EnabledExtensionsView
│   ├── DisabledExtensionsView
│   └── OutdatedExtensionsView
└── IExtensionsWorkbenchService
    └── IExtensionManagementService
        ├── LocalExtensionManagementServer
        ├── RemoteExtensionManagementServer
        └── WebExtensionManagementServer
```

### 8.3 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/extensions/browser/extensionsViewlet.ts` | ビュー | ビューレット実装 |
| `src/vs/workbench/contrib/extensions/browser/extensionsViews.ts` | ビュー | 各種リストビュー |
| `src/vs/workbench/contrib/extensions/browser/extensionsActions.ts` | アクション | 拡張機能操作 |
| `src/vs/workbench/contrib/extensions/common/extensions.ts` | 共通 | インターフェース定義 |

## 9. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
