# 機能設計書: ターミナル

## 1. 機能概要

### 1.1 機能名
ターミナル（Terminal）

### 1.2 機能ID
No.26

### 1.3 概要説明
ターミナル機能は、VSCode内に統合されたコマンドラインインターフェースを提供する機能である。複数のターミナルインスタンスの作成、分割表示、カスタムシェルの設定、ターミナルプロファイルの管理などを行う。xterm.jsを基盤としたターミナルエミュレータを提供し、ローカルおよびリモート環境でのシェル操作をサポートする。

### 1.4 関連画面
- ターミナルパネル
- ターミナルタブ
- ターミナル分割ビュー

## 2. 機能要件

### 2.1 ビジネス要件
- VSCode内でシェルコマンドを実行できること
- 複数のターミナルを同時に管理できること
- カスタムシェルプロファイルを設定できること
- リモート環境でのターミナル接続をサポートすること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| TERM-001 | ターミナル作成 | 新規ターミナルインスタンスの作成 |
| TERM-002 | ターミナル分割 | ターミナルの水平/垂直分割 |
| TERM-003 | プロファイル管理 | シェルプロファイルの管理 |
| TERM-004 | シェル統合 | シェル統合機能 |
| TERM-005 | リンク検出 | URL/ファイルパスのリンク化 |
| TERM-006 | コマンド検出 | コマンド履歴の検出 |
| TERM-007 | リモートターミナル | リモート環境でのターミナル |
| TERM-008 | 永続化 | セッション永続化 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                   Terminal Panel                        │
│  ┌──────────────────────────────────────────────────┐  │
│  │  Terminal Tabs (TerminalTabList)                  │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Terminal Instance (xterm.js)                     │  │
│  │  ┌────────────────┬─────────────────────────────┐│  │
│  │  │ Terminal 1     │ Terminal 2                  ││  │
│  │  │ (Split View)   │ (Split View)                ││  │
│  │  └────────────────┴─────────────────────────────┘│  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                  TerminalService                        │
│  ┌────────────────────────────────────────────────────┐│
│  │  - instances: ITerminalInstance[]                  ││
│  │  - createTerminal()                                ││
│  │  - onDidCreateInstance                             ││
│  │  - onDidDisposeInstance                            ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                 TerminalInstance                        │
│  ┌────────────────────────────────────────────────────┐│
│  │  - xterm: Terminal (xterm.js)                      ││
│  │  - ptyProcess: ITerminalProcess                    ││
│  │  - capabilities: TerminalCapabilityStore           ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 クラス図

```
┌─────────────────────────┐
│   ITerminalService      │
│   <<interface>>         │
├─────────────────────────┤
│ + instances             │
│ + activeInstance        │
│ + createTerminal()      │
│ + setActiveInstance()   │
└───────────┬─────────────┘
            │
            ▼
┌─────────────────────────┐
│    TerminalService      │
├─────────────────────────┤
│ - _hostActiveTerminals  │
│ - _detachedXterms       │
│ - _backgroundedInstances│
├─────────────────────────┤
│ + createTerminal()      │
│ + showProfileQuickPick()│
│ + getReconnectedTerms() │
└─────────────────────────┘

┌─────────────────────────┐      ┌─────────────────────────┐
│   ITerminalInstance     │      │  ITerminalGroupService  │
│   <<interface>>         │      │   <<interface>>         │
├─────────────────────────┤      ├─────────────────────────┤
│ + instanceId            │      │ + groups                │
│ + cols / rows           │──────│ + activeGroup           │
│ + processId             │      │ + createGroup()         │
│ + shellLaunchConfig     │      │ + showPanel()           │
└─────────────────────────┘      └─────────────────────────┘
```

## 4. 詳細設計

### 4.1 データ構造

#### ITerminalInstance
```typescript
interface ITerminalInstance {
    readonly instanceId: number;
    readonly cols: number;
    readonly rows: number;
    readonly processId: number | undefined;
    readonly shellLaunchConfig: IShellLaunchConfig;
    readonly shellType: TerminalShellType | undefined;
    readonly capabilities: ITerminalCapabilityStore;

    // 操作
    sendText(text: string, addNewLine?: boolean): Promise<void>;
    dispose(): void;

    // イベント
    readonly onData: Event<string>;
    readonly onProcessIdReady: Event<ITerminalInstance>;
    readonly onTitleChanged: Event<ITerminalInstance>;
}
```

#### IShellLaunchConfig
```typescript
interface IShellLaunchConfig {
    executable?: string;
    args?: string[] | string;
    cwd?: string | URI;
    env?: ITerminalEnvironment;
    name?: string;
    icon?: ThemeIcon | URI | { light: URI; dark: URI };
    color?: string;
    hideFromUser?: boolean;
    isFeatureTerminal?: boolean;
}
```

### 4.2 処理フロー

#### ターミナル作成フロー
```
1. createTerminal()呼び出し
   │
   ├─2. TerminalInstance作成
   │    │
   │    ├─ xterm.jsインスタンス作成
   │    │
   │    └─ PTYプロセス起動
   │
   ├─3. TerminalGroup/Editorへの追加
   │    │
   │    └─ タブの作成
   │
   └─4. onDidCreateInstance発火
        │
        └─ プロファイル更新
```

### 4.3 主要メソッド

#### TerminalService.createTerminal()
```typescript
async createTerminal(options?: ICreateTerminalOptions): Promise<ITerminalInstance> {
    // 1. 設定の解決
    const config = this.resolveShellLaunchConfig(options);

    // 2. インスタンスの作成
    const instance = this._terminalInstanceService.createInstance(config);

    // 3. 配置先の決定
    if (options?.location === TerminalLocation.Editor) {
        await this._terminalEditorService.openEditor(instance);
    } else {
        await this._terminalGroupService.createGroup(instance);
    }

    // 4. イベント発火
    this._onDidCreateInstance.fire(instance);

    return instance;
}
```

## 5. インターフェース設計

### 5.1 サービスインターフェース

```typescript
interface ITerminalService {
    readonly _serviceBrand: undefined;

    // インスタンス管理
    readonly instances: readonly ITerminalInstance[];
    readonly activeInstance: ITerminalInstance | undefined;
    createTerminal(options?: ICreateTerminalOptions): Promise<ITerminalInstance>;
    setActiveInstance(instance: ITerminalInstance): void;

    // 接続状態
    readonly connectionState: TerminalConnectionState;
    readonly whenConnected: Promise<void>;

    // イベント
    readonly onDidCreateInstance: Event<ITerminalInstance>;
    readonly onDidDisposeInstance: Event<ITerminalInstance>;
    readonly onDidChangeActiveInstance: Event<ITerminalInstance | undefined>;
}
```

## 6. コンテキストキー

```typescript
// ターミナル関連コンテキストキー
TerminalContextKeys.processSupported;
TerminalContextKeys.terminalHasBeenCreated;
TerminalContextKeys.count;
TerminalContextKeys.terminalEditorActive;
TerminalContextKeys.shellType;
```

## 7. コードリーディングガイド

### 7.1 推奨読解順序

1. **サービス層**: `src/vs/workbench/contrib/terminal/browser/terminalService.ts`
   - **66-106行目**: TerminalServiceクラスのプロパティ
   - **169-238行目**: コンストラクタと初期化
   - **240-278行目**: showProfileQuickPick()

2. **インスタンス**: `src/vs/workbench/contrib/terminal/browser/terminalInstance.ts`
   - TerminalInstanceクラスの実装

3. **xterm統合**: `src/vs/workbench/contrib/terminal/browser/xterm/xtermTerminal.ts`
   - XtermTerminalクラス

### 7.2 プログラム呼び出し階層図

```
TerminalService
├── TerminalGroupService
│   └── TerminalGroup[]
│       └── TerminalInstance[]
├── TerminalEditorService
│   └── TerminalEditorInput[]
├── TerminalInstanceService
│   └── TerminalInstance
│       └── XtermTerminal (xterm.js)
└── TerminalProfileService
    └── Terminal Profiles
```

### 7.3 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/terminal/browser/terminalService.ts` | サービス | ターミナルサービス |
| `src/vs/workbench/contrib/terminal/browser/terminalInstance.ts` | インスタンス | ターミナルインスタンス |
| `src/vs/workbench/contrib/terminal/browser/xterm/xtermTerminal.ts` | xterm | xterm.js統合 |
| `src/vs/workbench/contrib/terminal/common/terminal.ts` | 共通 | インターフェース定義 |

## 8. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
