# 機能設計書: アウトライン

## 1. 機能概要

### 1.1 機能名
アウトライン（Outline）

### 1.2 機能ID
No.27

### 1.3 概要説明
アウトライン機能は、現在開いているドキュメントの構造をツリー形式で表示するビューである。シンボル（関数、クラス、変数など）の階層構造を視覚化し、コード内の特定の位置への素早いナビゲーションを可能にする。言語サービスが提供するシンボル情報を基に、ドキュメントの概要を提供する。

### 1.4 関連画面
- アウトラインビュー（サイドバー/エクスプローラー内）
- シンボルツリー

## 2. 機能要件

### 2.1 ビジネス要件
- アクティブなエディタのドキュメント構造を表示すること
- シンボルクリックで該当位置にジャンプできること
- ソート順（位置順/名前順/種類順）を切り替えられること
- フィルタリング機能を提供すること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| OUT-001 | シンボル表示 | ドキュメントシンボルのツリー表示 |
| OUT-002 | ナビゲーション | シンボルクリックで位置にジャンプ |
| OUT-003 | ソート機能 | 位置順/名前順/種類順のソート |
| OUT-004 | フィルタ機能 | シンボル名でのフィルタリング |
| OUT-005 | カーソル追従 | エディタカーソル位置に追従 |
| OUT-006 | 折りたたみ | ツリーの展開/折りたたみ |
| OUT-007 | 複数プロバイダー | 複数のアウトラインプロバイダー対応 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                    Outline Pane                         │
│  ┌──────────────────────────────────────────────────┐  │
│  │  Progress Bar                                     │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Message Area (No symbols / Loading)              │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Outline Tree (WorkbenchDataTree)                 │  │
│  │  ┌────────────────────────────────────────────┐  │  │
│  │  │ Class Symbol                               │  │  │
│  │  │   ├── Method Symbol                        │  │  │
│  │  │   └── Property Symbol                      │  │  │
│  │  │ Function Symbol                            │  │  │
│  │  └────────────────────────────────────────────┘  │  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                  IOutlineService                        │
│  ┌────────────────────────────────────────────────────┐│
│  │  - canCreateOutline()                              ││
│  │  - createOutline()                                 ││
│  │  - onDidChange                                     ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│           Language Service / Outline Provider           │
│  ┌────────────────────────────────────────────────────┐│
│  │  Document Symbol Provider                          ││
│  │  (Extension API)                                   ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 クラス図

```
┌─────────────────────────┐
│      OutlinePane        │
├─────────────────────────┤
│ - _domNode              │
│ - _message              │
│ - _progressBar          │
│ - _tree                 │
│ - _outlineViewState     │
├─────────────────────────┤
│ + collapseAll()         │
│ + expandAll()           │
│ + focus()               │
└───────────┬─────────────┘
            │
            ▼
┌─────────────────────────┐      ┌─────────────────────────┐
│   OutlineViewState      │      │    IOutlineService      │
├─────────────────────────┤      │    <<interface>>        │
│ - followCursor          │      ├─────────────────────────┤
│ - filterOnType          │      │ + canCreateOutline()    │
│ - sortBy                │      │ + createOutline()       │
├─────────────────────────┤      │ + onDidChange           │
│ + restore()             │      └─────────────────────────┘
│ + save()                │
└─────────────────────────┘

┌─────────────────────────┐
│  OutlineSortOrder       │
│      <<enum>>           │
├─────────────────────────┤
│ ByPosition              │
│ ByName                  │
│ ByKind                  │
└─────────────────────────┘
```

## 4. 詳細設計

### 4.1 データ構造

#### IOutline
```typescript
interface IOutline<E> {
    readonly uri: URI | undefined;
    readonly isEmpty: boolean;
    readonly outlineKind: string;

    // 設定
    readonly config: IOutlineConfiguration<E>;

    // イベント
    readonly onDidChange: Event<void>;
}
```

#### OutlineViewState
```typescript
class OutlineViewState {
    followCursor: boolean;
    filterOnType: boolean;
    sortBy: OutlineSortOrder;

    restore(storageService: IStorageService): void;
    save(storageService: IStorageService): void;
}
```

#### OutlineSortOrder
```typescript
enum OutlineSortOrder {
    ByPosition = 0,
    ByName = 1,
    ByKind = 2
}
```

### 4.2 処理フロー

#### アウトライン更新フロー
```
1. エディタ変更検出
   │
   ├─2. OutlinePane._handleEditorChanged()
   │    │
   │    └─ エディタコントロール変更の処理
   │
   ├─3. IOutlineService.createOutline()
   │    │
   │    └─ シンボルプロバイダーからシンボル取得
   │
   └─4. ツリー更新
        │
        ├─ OutlineTreeSorter.compare()
        │
        └─ WorkbenchDataTree.setInput()
```

### 4.3 主要メソッド

#### OutlinePane._handleEditorControlChanged()
```typescript
private async _handleEditorControlChanged(pane: IEditorPane | undefined): Promise<void> {
    // 1. 状態の保持
    const resource = EditorResourceAccessor.getOriginalUri(pane?.input);
    const didCapture = this._captureViewState();

    // 2. リソースチェック
    if (!pane || !this._outlineService.canCreateOutline(pane) || !resource) {
        return this._showMessage(localize('no-editor', "..."));
    }

    // 3. アウトラインの作成
    const cts = new CancellationTokenSource();
    const newOutline = await this._outlineService.createOutline(
        pane, OutlineTarget.OutlinePane, cts.token
    );

    // 4. ツリーの構築
    const tree = this._instantiationService.createInstance(
        WorkbenchDataTree,
        'OutlinePane',
        this._treeContainer,
        newOutline.config.delegate,
        newOutline.config.renderers,
        newOutline.config.treeDataSource,
        { sorter, ... }
    );

    // 5. 入力設定
    tree.setInput(newOutline);
}
```

#### OutlineTreeSorter.compare()
```typescript
compare(a: E, b: E): number {
    if (this.order === OutlineSortOrder.ByKind) {
        return this._comparator.compareByType(a, b);
    } else if (this.order === OutlineSortOrder.ByName) {
        return this._comparator.compareByName(a, b);
    } else {
        return this._comparator.compareByPosition(a, b);
    }
}
```

## 5. インターフェース設計

### 5.1 サービスインターフェース

```typescript
interface IOutlineService {
    readonly _serviceBrand: undefined;

    // アウトライン作成
    canCreateOutline(pane: IEditorPane): boolean;
    createOutline(
        pane: IEditorPane,
        target: OutlineTarget,
        token: CancellationToken
    ): Promise<IOutline<unknown> | undefined>;

    // イベント
    readonly onDidChange: Event<void>;
}
```

### 5.2 ペインインターフェース

```typescript
interface IOutlinePane {
    collapseAll(): void;
    expandAll(): void;
    get outlineViewState(): OutlineViewState;
}
```

## 6. コンテキストキー

```typescript
// アウトライン関連コンテキストキー
const ctxFollowsCursor: RawContextKey<boolean>;
const ctxFilterOnType: RawContextKey<boolean>;
const ctxSortMode: RawContextKey<OutlineSortOrder>;
const ctxAllCollapsed: RawContextKey<boolean>;
const ctxFocused: RawContextKey<boolean>;
```

## 7. コードリーディングガイド

### 7.1 推奨読解順序

1. **ペイン層**: `src/vs/workbench/contrib/outline/browser/outlinePane.ts`
   - **40-56行目**: OutlineTreeSorterクラス
   - **58-124行目**: OutlinePaneクラスのプロパティとコンストラクタ
   - **133-159行目**: renderBody()メソッド
   - **200-300行目**: _handleEditorControlChanged()

2. **ビュー状態**: `src/vs/workbench/contrib/outline/browser/outlineViewState.ts`
   - OutlineViewStateクラス

3. **アウトラインサービス**: `src/vs/workbench/services/outline/browser/outline.ts`
   - IOutlineService実装

### 7.2 プログラム呼び出し階層図

```
OutlinePane
├── OutlineViewState
│   ├── followCursor
│   ├── filterOnType
│   └── sortBy
├── WorkbenchDataTree
│   ├── OutlineTreeSorter
│   ├── OutlineConfig.delegate
│   ├── OutlineConfig.renderers
│   └── OutlineConfig.treeDataSource
└── IOutlineService
    └── IOutline
        └── Document Symbol Provider (Extensions)
```

### 7.3 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/outline/browser/outlinePane.ts` | ペイン | アウトラインペイン実装 |
| `src/vs/workbench/contrib/outline/browser/outlineViewState.ts` | 状態 | ビュー状態管理 |
| `src/vs/workbench/contrib/outline/browser/outline.ts` | 定義 | コンテキストキー定義 |
| `src/vs/workbench/services/outline/browser/outline.ts` | サービス | アウトラインサービス |

## 8. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
