# 機能設計書: 問題パネル

## 1. 機能概要

### 1.1 機能名
問題パネル（Problems Panel / Markers View）

### 1.2 機能ID
No.28

### 1.3 概要説明
問題パネル機能は、コード内のエラー、警告、情報メッセージを集約して表示するパネルである。言語サービスや拡張機能が検出した問題をリスト表示し、フィルタリング、ソート、問題箇所へのナビゲーション機能を提供する。ツリービューとテーブルビューの切り替えも可能である。

### 1.4 関連画面
- 問題パネル（下部パネル）
- フィルタ入力
- 問題リスト/ツリー

## 2. 機能要件

### 2.1 ビジネス要件
- コード内のエラー、警告、情報を一覧表示すること
- 問題箇所へのワンクリックナビゲーションが可能なこと
- 重要度（エラー/警告/情報）でフィルタリングできること
- ファイル単位での問題グループ化ができること

### 2.2 機能要件詳細

| 要件ID | 要件名 | 説明 |
|--------|--------|------|
| PROB-001 | 問題一覧表示 | エラー、警告、情報の一覧表示 |
| PROB-002 | ナビゲーション | 問題箇所へのジャンプ |
| PROB-003 | 重要度フィルタ | エラー/警告/情報のフィルタ |
| PROB-004 | テキストフィルタ | メッセージによるフィルタリング |
| PROB-005 | アクティブファイル | 現在のファイルのみ表示 |
| PROB-006 | ファイル除外 | files.excludeによる除外 |
| PROB-007 | ビュー切替 | ツリー/テーブルビュー切替 |
| PROB-008 | クイックフィックス | 問題の自動修正 |

## 3. アーキテクチャ設計

### 3.1 コンポーネント構成

```
┌─────────────────────────────────────────────────────────┐
│                    Markers View                         │
│  ┌──────────────────────────────────────────────────┐  │
│  │  Filter Input (FilterWidget)                      │  │
│  ├──────────────────────────────────────────────────┤  │
│  │  Problems Widget                                  │  │
│  │  ┌────────────────────────────────────────────┐  │  │
│  │  │ Tree View / Table View                     │  │  │
│  │  │  ├── ResourceMarkers (file.ts)             │  │  │
│  │  │  │   ├── Marker (Error: line 10)           │  │  │
│  │  │  │   │   └── RelatedInformation            │  │  │
│  │  │  │   └── Marker (Warning: line 20)         │  │  │
│  │  │  └── ResourceMarkers (file2.ts)            │  │  │
│  │  └────────────────────────────────────────────┘  │  │
│  └──────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                   MarkersModel                          │
│  ┌────────────────────────────────────────────────────┐│
│  │  - resourceMarkers: Map<URI, ResourceMarkers>      ││
│  │  - cachedSortedResources: ResourceMarkers[]        ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────┐
│                   IMarkerService                        │
│  ┌────────────────────────────────────────────────────┐│
│  │  Platform Marker Service                           ││
│  │  - read()                                          ││
│  │  - changeOne()                                     ││
│  │  - onMarkerChanged                                 ││
│  └────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────┘
```

### 3.2 クラス図

```
┌─────────────────────────┐
│      MarkersView        │
│   extends FilterViewPane│
├─────────────────────────┤
│ - markersModel          │
│ - widget                │
│ - filters               │
│ - markersViewModel      │
├─────────────────────────┤
│ + getTitle()            │
│ + focus()               │
│ + focusFilter()         │
│ + showQuickFixes()      │
└───────────┬─────────────┘
            │
            ▼
┌─────────────────────────┐      ┌─────────────────────────┐
│     MarkersModel        │      │     MarkersFilters      │
├─────────────────────────┤      ├─────────────────────────┤
│ - _markersByResource    │      │ + showErrors            │
│ - _total                │      │ + showWarnings          │
├─────────────────────────┤      │ + showInfos             │
│ + resourceMarkers()     │      │ + excludedFiles         │
│ + total                 │      │ + activeFile            │
│ + onDidChange           │      └─────────────────────────┘
└─────────────────────────┘

┌─────────────────────────┐
│    ResourceMarkers      │
├─────────────────────────┤
│ + id: string            │
│ + resource: URI         │
│ + markers: Marker[]     │
└─────────────────────────┘
```

## 4. 詳細設計

### 4.1 データ構造

#### IMarkersPanelState
```typescript
interface IMarkersPanelState {
    filter?: string;
    filterHistory?: string[];
    showErrors?: boolean;
    showWarnings?: boolean;
    showInfos?: boolean;
    useFilesExclude?: boolean;
    activeFile?: boolean;
    multiline?: boolean;
    viewMode?: MarkersViewMode;
}
```

#### ResourceMarkers
```typescript
class ResourceMarkers {
    readonly id: string;
    readonly resource: URI;
    readonly markers: Marker[];

    constructor(resource: URI, markers: Marker[]);
}
```

#### Marker
```typescript
class Marker {
    readonly id: string;
    readonly marker: IMarker;
    readonly relatedInformation: RelatedInformation[];

    get resource(): URI;
    get range(): IRange;
    get severity(): MarkerSeverity;
    get message(): string;
}
```

### 4.2 処理フロー

#### フィルタ更新フロー
```
1. フィルタ入力変更
   │
   ├─2. MarkersFilters更新
   │    │
   │    └─ フィルタ条件の適用
   │
   ├─3. FilterOptions構築
   │    │
   │    └─ テキスト/重要度/除外パターン
   │
   └─4. widget.filterMarkers()
        │
        └─ ツリー/テーブルの更新
```

### 4.3 主要メソッド

#### MarkersView.renderBody()
```typescript
protected override renderBody(parent: HTMLElement): void {
    super.renderBody(parent);

    parent.classList.add('markers-panel');

    // キーボードイベント処理
    this._register(dom.addDisposableListener(parent, 'keydown', e => {
        // フィルタへのフォーカス処理
    }));

    const panelContainer = dom.append(parent, dom.$('.markers-panel-container'));

    this.createArialLabelElement(panelContainer);
    this.createMessageBox(panelContainer);

    this.widgetContainer = dom.append(panelContainer, dom.$('.widget-container'));
    this.createWidget(this.widgetContainer);

    this.updateFilter();
    this.renderContent();
}
```

## 5. インターフェース設計

### 5.1 ウィジェットインターフェース

```typescript
interface IProblemsWidget {
    readonly contextKeyService: IContextKeyService;

    // イベント
    readonly onContextMenu: Event<ITreeContextMenuEvent<MarkerElement | null>>;
    readonly onDidChangeFocus: Event<ITreeEvent<MarkerElement | null>>;
    readonly onDidChangeSelection: Event<ITreeEvent<MarkerElement | null>>;
    readonly onDidOpen: Event<IOpenEvent<MarkerElement | undefined>>;

    // 操作
    collapseMarkers(): void;
    dispose(): void;
    domFocus(): void;
    filterMarkers(resourceMarkers: ResourceMarkers[], filterOptions: FilterOptions): void;
    getFocus(): (MarkerElement | null)[];
    getSelection(): (MarkerElement | null)[];
    reset(resourceMarkers: ResourceMarkers[]): void;
    revealMarkers(activeResource: ResourceMarkers | null, focus: boolean, top: number): void;
}
```

### 5.2 ビューインターフェース

```typescript
interface IMarkersView {
    readonly filters: MarkersFilters;
    readonly onDidChangeVisibility: Event<boolean>;

    focusFilter(): void;
    clearFilterText(): void;
    showQuickFixes(marker: Marker): void;
    openFileAtElement(element: any, preserveFocus: boolean, sideBeside: boolean, pinned: boolean): boolean;
}
```

## 6. コンテキストキー

```typescript
// マーカー関連コンテキストキー
MarkersContextKeys.MarkerViewFilterFocusContextKey;
MarkersContextKeys.MarkerFocusContextKey;
MarkersContextKeys.MarkerViewMode;
MarkersContextKeys.RelatedInformationFocusContextKey;
```

## 7. コードリーディングガイド

### 7.1 推奨読解順序

1. **ビュー層**: `src/vs/workbench/contrib/markers/browser/markersView.ts`
   - **61-68行目**: createResourceMarkersIterator関数
   - **70-80行目**: IMarkersPanelState
   - **109-200行目**: MarkersViewクラスのプロパティとコンストラクタ
   - **220-247行目**: renderBody()メソッド

2. **モデル層**: `src/vs/workbench/contrib/markers/browser/markersModel.ts`
   - MarkersModel、ResourceMarkers、Markerクラス

3. **フィルタ**: `src/vs/workbench/contrib/markers/browser/markersViewActions.ts`
   - MarkersFiltersクラス

### 7.2 プログラム呼び出し階層図

```
MarkersView
├── FilterWidget
│   └── MarkersFilters
├── ProblemsWidget
│   ├── MarkersTree (WorkbenchObjectTree)
│   │   ├── MarkerRenderer
│   │   ├── ResourceMarkersRenderer
│   │   └── RelatedInformationRenderer
│   └── MarkersTable
├── MarkersModel
│   └── ResourceMarkers[]
│       └── Marker[]
└── IMarkerService
    └── Platform Markers
```

### 7.3 関連ファイル一覧

| パス | 種別 | 役割 |
|------|------|------|
| `src/vs/workbench/contrib/markers/browser/markersView.ts` | ビュー | 問題ビュー実装 |
| `src/vs/workbench/contrib/markers/browser/markersModel.ts` | モデル | マーカーモデル |
| `src/vs/workbench/contrib/markers/browser/markersTreeViewer.ts` | レンダラー | ツリーレンダラー |
| `src/vs/workbench/contrib/markers/browser/markersViewActions.ts` | アクション | フィルタアクション |
| `src/vs/workbench/contrib/markers/common/markers.ts` | 共通 | 定数定義 |

## 8. 変更履歴

| 日付 | バージョン | 変更内容 | 作成者 |
|------|-----------|---------|--------|
| 2026-01-27 | 1.0 | 初版作成 | Claude |
