# 機能設計書 43-コメント

## 概要

本ドキュメントは、VS Codeのコードレビューコメント機能に関する機能設計書である。ソースコードへのコメント付与と管理を行う機能を定義する。

### 本機能の処理概要

コメント機能は、ソースコード上の特定の行や範囲に対してコメントを付与・管理する機能である。GitHub Pull Request等のコードレビューツールとの連携により、コードレビューをVS Code内で完結させることができる。

**業務上の目的・背景**：コードレビューはソフトウェア品質向上の重要なプロセスである。開発者がエディタ内でコメントの確認・返信ができることで、レビュープロセスの効率化とコンテキストスイッチの削減を実現する。

**機能の利用シーン**：
- GitHub Pull Requestのコードレビュー
- GitLab Merge Requestのコードレビュー
- Azure DevOps Pull Requestのコードレビュー
- コード内の疑問点に対するインラインコメント
- チームメンバー間のコード上でのディスカッション

**主要な処理内容**：
1. コメントコントローラーの登録と管理（拡張機能からのプロバイダー登録）
2. コメントスレッドの作成・更新・削除
3. ドキュメント・ノートブックのコメント情報取得
4. リアクション（絵文字）の切り替え
5. コメントの永続化（Continue On Comments機能）
6. コメント可能範囲の管理
7. コメントビューでの一覧表示

**関連システム・外部連携**：
- 拡張機能ホスト（Extension Host）経由でGitHub/GitLab等と連携
- ノートブック機能との連携（セル単位のコメント）
- ストレージサービスとの連携（下書きコメントの永続化）
- レイアウトサービスとの連携（Zenモード時の無効化）

**権限による制御**：コメント機能は設定（`comments.visible`）で有効/無効を切り替え可能。Zenモード時は自動的に無効化される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 32 | コメントビュー | 主画面 | コードコメントの一覧表示 |

## 機能種別

イベント駆動処理 / データ管理 / UI連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| uniqueOwner | string | Yes | コメントコントローラーの一意識別子 | - |
| resource | URI | Yes | 対象ドキュメントのURI | - |
| range | Range \| ICellRange | No | コメント対象範囲 | - |
| commentInfos | ICommentInfo[] | Yes | コメント情報配列 | - |
| thread | CommentThread | Yes | コメントスレッド | - |
| comment | Comment | No | 個別コメント | - |
| reaction | CommentReaction | No | リアクション | - |

### ICommentController詳細

| パラメータ名 | 型 | 必須 | 説明 |
|-------------|-----|-----|------|
| id | string | Yes | コントローラーID |
| label | string | Yes | 表示ラベル |
| features | object | Yes | 機能フラグ（reactionGroup, reactionHandler, options） |
| options | CommentOptions | No | コメントオプション |
| contextValue | string | No | コンテキスト値 |
| owner | string | Yes | オーナー識別子 |
| activeComment | object | No | アクティブなコメント情報 |

### 入力データソース

- 拡張機能からのコメントプロバイダー登録
- ユーザーのコメント入力（SimpleCommentEditor）
- ストレージからの下書きコメント復元

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| ICommentInfo | interface | ドキュメントコメント情報 |
| INotebookCommentInfo | interface | ノートブックコメント情報 |
| ICommentsModel | interface | コメントモデル |

### 出力先

- コメントビュー（一覧表示）
- エディタ内（インラインコメントウィジェット）
- ストレージ（下書きコメントの永続化）

## 処理フロー

### 処理シーケンス

```
1. コメントコントローラー登録
   └─ registerCommentController() で拡張機能がコントローラーを登録

2. コメント可能範囲の通知
   └─ updateCommentingRanges() で対象リソースのスキームを通知
   └─ _workspaceHasCommenting コンテキストキーを更新

3. ドキュメントコメントの取得
   └─ getDocumentComments() で全コントローラーからコメントを収集
   └─ Continue On Commentsをマージ
   └─ コメント可能範囲を更新

4. コメントスレッドの作成
   └─ createCommentThreadTemplate() でテンプレート作成
   └─ エディタ内にウィジェット表示

5. コメント更新・削除
   └─ updateComments() / disposeCommentThread() で状態更新
   └─ CommentsModelへの反映
   └─ イベント発火

6. 永続化処理
   └─ onWillSaveState で下書きコメントをストレージに保存
   └─ Continue On Commentsの復元
```

### フローチャート

```mermaid
flowchart TD
    A[拡張機能がコントローラー登録] --> B[registerCommentController]
    B --> C[_commentControlsに追加]
    C --> D[onDidSetDataProvider発火]
    D --> E{ドキュメントオープン?}
    E -->|Yes| F[getDocumentComments]
    F --> G[全コントローラーに問い合わせ]
    G --> H[コメント情報収集]
    H --> I[コメント可能範囲更新]
    I --> J[CommentsModel更新]
    J --> K[UIへ反映]
    K --> L{ユーザー操作}
    L -->|新規コメント| M[createCommentThreadTemplate]
    L -->|リアクション| N[toggleReaction]
    L -->|コメント削除| O[disposeCommentThread]
    M --> J
    N --> J
    O --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-43-01 | Zenモード連動 | Zenモード有効時はコメント機能を自動無効化 | Zenモード切替時 |
| BR-43-02 | 設定連動 | comments.visible設定でコメント機能の有効/無効を制御 | 設定変更時 |
| BR-43-03 | スキーム除外 | vscodeSourceControlスキームのモデルはコメント取得対象外 | モデル追加時 |
| BR-43-04 | Continue On Comments | 未送信のコメントはワークスペーススコープで永続化 | ワークスペース保存時 |

### 計算ロジック

**コメント可能範囲の判定**：
```typescript
// resourceHasCommentingRanges() より
resourceHasCommentingRanges(resource: URI): boolean {
    return this._commentingRangeResourceHintSchemes.has(resource.scheme)
        || this._commentingRangeResources.has(resource.toString());
}
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 下書き保存 | IStorageService | INSERT/UPDATE | Continue On Commentsの保存 |
| 下書き読込 | IStorageService | SELECT | Continue On Commentsの復元 |

### テーブル別操作詳細

#### StorageService（ワークスペーススコープ）

| 操作 | 項目（キー） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT/UPDATE | comments.continueOnComments | PendingCommentThread[] | 下書きコメント配列 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| CONTROLLER_NOT_FOUND | 業務エラー | 指定されたコントローラーが存在しない | 操作をスキップ |
| REACTION_NOT_SUPPORTED | 業務エラー | リアクションハンドラーが未設定 | "Not supported" エラーをスロー |
| COMMENT_FETCH_FAILED | システムエラー | コメント取得中にエラー | nullを返して処理継続 |

### リトライ仕様

コメント取得の失敗時はnullを返して処理を継続。リトライは行わない。

## トランザクション仕様

コメントの操作はトランザクション制御を行わない。各操作は即座にイベントとして通知される。

## パフォーマンス要件

- ストレージイベントの500msデバウンス
- モデル追加時のコメント取得（バックグラウンド）
- コメントプロバイダーへの並列問い合わせ

## セキュリティ考慮事項

- コメントデータは拡張機能のサーバー（GitHub等）に保存される
- 下書きコメントはローカルストレージに保存（ワークスペーススコープ）
- コメント内容の検証は拡張機能側で実施

## 備考

- コメント機能は拡張機能API（vscode.comments）を通じて利用される
- GitHub Pull Requests拡張機能が主要なコメントプロバイダー

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

コメント機能で使用される主要なインターフェースと型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | commentService.ts | `src/vs/workbench/contrib/comments/browser/commentService.ts` | ICommentService（81-122行目）、ICommentController（56-75行目）、ICommentInfo（34-37行目）を確認 |
| 1-2 | commentModel.ts | `src/vs/workbench/contrib/comments/common/commentModel.ts` | ICommentThreadChangedEvent等の変更通知型 |
| 1-3 | commentsModel.ts | `src/vs/workbench/contrib/comments/browser/commentsModel.ts` | CommentsModel、ICommentsModel |

**読解のコツ**: ICommentControllerが拡張機能側のインターフェース、ICommentServiceがワークベンチ側のサービス。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | commentService.ts | `src/vs/workbench/contrib/comments/browser/commentService.ts` | CommentService クラス（126-554行目） |

**主要処理フロー**:
1. **27行目**: `ICommentService = createDecorator<ICommentService>('commentService')` でサービスID定義
2. **186-249行目**: コンストラクタで設定、Zenモード、ストレージの初期化
3. **368-371行目**: `registerCommentController()` でコントローラー登録
4. **465-490行目**: `getDocumentComments()` でコメント取得

#### Step 3: UI層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | commentsView.ts | `src/vs/workbench/contrib/comments/browser/commentsView.ts` | コメント一覧ビューの実装 |
| 3-2 | commentThreadWidget.ts | `src/vs/workbench/contrib/comments/browser/commentThreadWidget.ts` | インラインコメントウィジェット |
| 3-3 | commentThreadZoneWidget.ts | `src/vs/workbench/contrib/comments/browser/commentThreadZoneWidget.ts` | エディタゾーンウィジェット |

**主要処理フロー**:
- CommentsViewでコメント一覧をツリー表示
- CommentThreadWidgetでスレッド単位のUI
- CommentThreadZoneWidgetでエディタ内配置

#### Step 4: エディタ連携を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | commentsEditorContribution.ts | `src/vs/workbench/contrib/comments/browser/commentsEditorContribution.ts` | エディタへのコメント機能組み込み |
| 4-2 | commentsController.ts | `src/vs/workbench/contrib/comments/browser/commentsController.ts` | コメントの制御ロジック |

### プログラム呼び出し階層図

```
CommentService (commentService.ts)
    │
    ├─ _commentControls: Map<string, ICommentController>
    │      └─ 拡張機能から登録されたコントローラー
    │
    ├─ CommentsModel (commentsModel.ts)
    │      └─ コメントスレッドのモデル管理
    │
    ├─ CommentMenus (commentMenus.ts)
    │      └─ コメント関連のメニュー定義
    │
    └─ Storage (continueOnComments)
           └─ 下書きコメントの永続化
```

### データフロー図

```
[入力]                    [処理]                         [出力]

拡張機能登録       ───▶ registerCommentController() ───▶ _commentControlsに追加
(ICommentController)

getDocumentComments ───▶ 全コントローラーに問い合わせ ───▶ ICommentInfo[]
リクエスト                   │
                            ▼
                    Continue On Commentsマージ
                            │
                            ▼
                    CommentsModel更新      ───────────▶ UI更新
                            │                          (CommentsView)
                            ▼
                    イベント発火           ───────────▶ エディタ内表示
                                                       (CommentThreadWidget)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| commentService.ts | `src/vs/workbench/contrib/comments/browser/commentService.ts` | ソース | サービス実装 |
| commentsModel.ts | `src/vs/workbench/contrib/comments/browser/commentsModel.ts` | ソース | モデル管理 |
| commentsView.ts | `src/vs/workbench/contrib/comments/browser/commentsView.ts` | ソース | 一覧ビュー |
| commentThreadWidget.ts | `src/vs/workbench/contrib/comments/browser/commentThreadWidget.ts` | ソース | スレッドウィジェット |
| commentThreadZoneWidget.ts | `src/vs/workbench/contrib/comments/browser/commentThreadZoneWidget.ts` | ソース | ゾーンウィジェット |
| commentNode.ts | `src/vs/workbench/contrib/comments/browser/commentNode.ts` | ソース | コメントノード |
| commentReply.ts | `src/vs/workbench/contrib/comments/browser/commentReply.ts` | ソース | 返信UI |
| simpleCommentEditor.ts | `src/vs/workbench/contrib/comments/browser/simpleCommentEditor.ts` | ソース | コメントエディタ |
| commentMenus.ts | `src/vs/workbench/contrib/comments/browser/commentMenus.ts` | ソース | メニュー定義 |
| reactionsAction.ts | `src/vs/workbench/contrib/comments/browser/reactionsAction.ts` | ソース | リアクションUI |
| commentContextKeys.ts | `src/vs/workbench/contrib/comments/common/commentContextKeys.ts` | ソース | コンテキストキー |
| commentsConfiguration.ts | `src/vs/workbench/contrib/comments/common/commentsConfiguration.ts` | 設定 | 設定項目定義 |
| comments.contribution.ts | `src/vs/workbench/contrib/comments/browser/comments.contribution.ts` | ソース | 機能登録 |
