# 機能設計書 46-ファイルシステム

## 概要

本ドキュメントは、VS Codeのファイルシステム機能に関する機能設計書である。ファイルの読み書き、監視、プロバイダー管理等の基盤機能を定義する。

### 本機能の処理概要

ファイルシステム機能は、VS Code内でのすべてのファイル操作の基盤となるプラットフォームサービスである。ローカルファイルシステムだけでなく、リモートファイルシステムやカスタムスキーマのファイルシステムも統一的に扱える。

**業務上の目的・背景**：様々なファイルシステム（ローカル、リモート、仮想等）を統一的なAPIで操作できるようにし、拡張機能やワークベンチ機能がファイル操作を抽象化して実装できるようにする。

**機能の利用シーン**：
- ファイルの読み込み・書き込み
- ファイル・フォルダの作成・削除・移動・コピー
- ファイル変更の監視
- ファイル統計情報の取得
- カスタムファイルシステムプロバイダーの登録

**主要な処理内容**：
1. ファイルシステムプロバイダーの登録・管理
2. ファイル読み込み（バッファ/ストリーム）
3. ファイル書き込み（アトミック/追記対応）
4. ファイル・フォルダの作成・削除・移動・コピー
5. ファイル変更の監視（相関/非相関）
6. ファイル統計情報の取得

**関連システム・外部連携**：
- エクスプローラービュー（ファイルツリー表示）
- エディタ（ファイル内容の読み書き）
- ローカル履歴機能（ファイルスナップショット）
- リモート接続機能（リモートファイルシステム）
- 拡張機能（カスタムファイルシステムプロバイダー）

**権限による制御**：ファイルシステムプロバイダーのcapabilitiesによりサポートする操作を制御。読み取り専用、追記専用等の制限が可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 20 | エクスプローラービュー | API連携 | ファイル読み書き・監視のAPIを使用 |

## 機能種別

プラットフォームサービス / ファイル操作

## 入力仕様

### 入力パラメータ（主要メソッド）

#### readFile

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| resource | URI | Yes | 読み込み対象のURI | プロバイダー存在確認 |
| options | IReadFileOptions | No | 読み込みオプション | - |
| token | CancellationToken | No | キャンセルトークン | - |

#### writeFile

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| resource | URI | Yes | 書き込み先のURI | - |
| bufferOrReadableOrStream | VSBuffer等 | Yes | 書き込み内容 | - |
| options | IWriteFileOptions | No | 書き込みオプション | - |

### オプション詳細

| オプション | 型 | 説明 |
|-----------|-----|------|
| overwrite | boolean | 既存ファイルを上書きするか |
| create | boolean | ファイルが存在しない場合に作成するか |
| unlock | boolean | 書き込みロックを解除するか |
| atomic | IFileAtomicOptions | アトミック書き込み設定 |
| append | boolean | 追記モード |

### 入力データソース

- 拡張機能・ワークベンチからのAPI呼び出し
- ユーザーのファイル操作（エディタ保存等）
- ファイルシステムからの変更イベント

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| IFileStat | interface | ファイル統計情報 |
| IFileStatWithMetadata | interface | メタデータ付き統計情報 |
| IFileContent | interface | ファイル内容（バッファ） |
| IFileStreamContent | interface | ファイル内容（ストリーム） |
| FileChangesEvent | class | ファイル変更イベント |

### 出力先

- 呼び出し元（Promise戻り値）
- イベントリスナー（onDidFilesChange等）

## 処理フロー

### 処理シーケンス

```
1. プロバイダー登録
   └─ registerProvider() でスキーマ別にプロバイダー登録

2. ファイル操作リクエスト
   └─ URI.scheme からプロバイダーを解決

3. 操作実行
   └─ プロバイダーの対応メソッドを呼び出し
   └─ capabilities チェック

4. イベント発火
   └─ onDidRunOperation で操作完了通知
   └─ onDidFilesChange で変更通知

5. 結果返却
   └─ Promise で結果を返却
```

### フローチャート

```mermaid
flowchart TD
    A[ファイル操作リクエスト] --> B{プロバイダー存在?}
    B -->|No| C[プロバイダー活性化試行]
    C --> D{活性化成功?}
    D -->|No| E[エラー]
    D -->|Yes| F[プロバイダー取得]
    B -->|Yes| F
    F --> G{capability確認}
    G -->|未対応| H[エラー]
    G -->|対応| I[操作実行]
    I --> J{成功?}
    J -->|No| K[エラー処理]
    J -->|Yes| L[イベント発火]
    L --> M[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | プロバイダー解決 | URIのschemeに基づいてプロバイダーを解決 | 全操作 |
| BR-46-02 | 能力チェック | 操作前にプロバイダーのcapabilitiesを確認 | 全操作 |
| BR-46-03 | アトミック書き込み | atomic指定時は一時ファイル経由で書き込み | writeFile |
| BR-46-04 | 上書き確認 | overwrite=falseで既存ファイルがある場合はエラー | writeFile/copy/move |

### FileType列挙

```typescript
export enum FileType {
    Unknown = 0,
    File = 1,
    Directory = 2,
    SymbolicLink = 64
}
```

## データベース操作仕様

ファイルシステムサービスはデータベースを直接操作しない。ファイルシステムへの操作のみ。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| FILE_NOT_FOUND | 業務エラー | ファイルが存在しない | FileOperationError発生 |
| FILE_EXISTS | 業務エラー | 上書き禁止で既存ファイルあり | FileOperationError発生 |
| FILE_IS_DIRECTORY | 業務エラー | ディレクトリに対するファイル操作 | FileOperationError発生 |
| FILE_TOO_LARGE | 業務エラー | ファイルサイズが制限超過 | FileOperationError発生 |
| NO_PERMISSIONS | 権限エラー | 操作権限がない | FileOperationError発生 |
| UNKNOWN | システムエラー | 不明なエラー | FileOperationError発生 |

## トランザクション仕様

ファイル操作にトランザクションは提供されない。アトミック書き込みオプションにより擬似的な原子性を確保可能。

## パフォーマンス要件

- ストリーム読み込み（readFileStream）による大規模ファイル対応
- 相関ウォッチャーによるイベントフィルタリング
- プロバイダーの遅延活性化

## セキュリティ考慮事項

- プロバイダーのcapabilitiesによるアクセス制御
- ファイル操作はプロバイダー実装に依存
- リモートファイルシステムのセキュリティは接続層で管理

## 備考

- 主要なプロバイダー：file（ローカル）、vscode-remote（リモート）
- 拡張機能は`FileSystemProvider`を実装してカスタムスキーマを提供可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェースを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | files.ts | `src/vs/platform/files/common/files.ts` | IFileService（26-270行目）が中心 |

**主要処理フロー**:
1. **26行目**: `IFileService = createDecorator<IFileService>('fileService')` でサービスID定義
2. **28-270行目**: IFileServiceインターフェースの全メソッド定義
3. **51行目**: `registerProvider()` でプロバイダー登録
4. **118-119行目**: `resolve()` でファイル情報取得
5. **152行目**: `readFile()` でファイル読み込み
6. **165行目**: `writeFile()` でファイル書き込み
7. **201行目**: `cloneFile()` でファイルクローン

#### Step 2: オプション・型を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | files.ts | `src/vs/platform/files/common/files.ts` | 272-448行目のオプション定義 |

**主要な型**:
- **272-279行目**: `IFileOverwriteOptions` - 上書きオプション
- **378-391行目**: `IFileWriteOptions` - 書き込みオプション
- **421-447行目**: `IFileDeleteOptions` - 削除オプション
- **449行目〜**: `FileType` 列挙型

### プログラム呼び出し階層図

```
IFileService
    │
    ├─ registerProvider() ──▶ スキーマ別プロバイダー登録
    │
    ├─ 読み込み系
    │      ├─ resolve() ──▶ ファイル情報取得
    │      ├─ stat() ──▶ 統計情報のみ取得
    │      ├─ readFile() ──▶ バッファ読み込み
    │      └─ readFileStream() ──▶ ストリーム読み込み
    │
    ├─ 書き込み系
    │      ├─ writeFile() ──▶ ファイル書き込み
    │      ├─ createFile() ──▶ 新規作成
    │      ├─ createFolder() ──▶ フォルダ作成
    │      └─ cloneFile() ──▶ ファイルクローン
    │
    ├─ 操作系
    │      ├─ move() ──▶ 移動
    │      ├─ copy() ──▶ コピー
    │      └─ del() ──▶ 削除
    │
    └─ 監視系
           ├─ watch() ──▶ 非相関監視
           └─ createWatcher() ──▶ 相関監視
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URI + オプション   ───▶ IFileService            ───▶ 結果（IFileStat等）
                              │
                              ├──▶ getProvider(scheme) ───▶ IFileSystemProvider
                              │
                              └──▶ プロバイダーメソッド呼び出し
                                        │
ファイルシステム   ◀───────────────────────┘
変更

                    onDidFilesChange       ───▶ FileChangesEvent
                    onDidRunOperation      ───▶ FileOperationEvent
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| files.ts | `src/vs/platform/files/common/files.ts` | ソース | インターフェース定義 |
| fileService.ts | `src/vs/platform/files/common/fileService.ts` | ソース | サービス実装 |
| diskFileSystemProvider.ts | `src/vs/platform/files/node/diskFileSystemProvider.ts` | ソース | ディスクプロバイダー |
| watcher.ts | `src/vs/platform/files/common/watcher.ts` | ソース | ファイル監視 |
